<?php

namespace App\Http\Controllers\BackEnd;

use Carbon\Carbon;
use App\Traits\DeleteTrait;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\BackEnd\Expense;
use Illuminate\Support\Facades\DB;
use App\Models\BackEnd\Transaction;
use App\Http\Controllers\Controller;
use Yajra\DataTables\Facades\DataTables;

class ExpenseController extends Controller
{
    use DeleteTrait;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {

        if ($request->ajax()) {

            $data = DB::table('expenses')->whereNull('deleted_at')->orderBy('id', 'DESC');

            return Datatables::of($data)

                ->addColumn('date', function ($data) {
                    $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
                    $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];
                    $date = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->date)->format('d/m/ Y'));

                    return $date;
                })

                ->addColumn('amount', function ($data) {
                    $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
                    $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

                    $amount = str_replace($en_numbers, $bn_numbers,$data->amount);
                    return $amount;
                })

                ->addColumn('purpose', function ($data) {
                        $result = isset($data->purpose) ? strip_tags($data->purpose) : '--' ;
                        return Str::limit( $result, 60) ;
                    })

                ->addColumn('action', function ($data) {

                        $result = '';

                        $details = '<li><a class="dropdown-item" href="' . route('expense.show', $data->id) . ' " ><i class="fa fa-eye" ></i> Details</a></li>';

                        $edit = '<li><a class="dropdown-item" href="' . route('expense.edit', $data->id) . ' "><i class="fa fa-edit"></i> Edit</a></li>';

                        $delete = '<li><a class="dropdown-item btn-delete" href="#" data-remote=" ' . route('expense.destroy', $data->id) . ' "><i class="fa fa-trash"></i> Delete</a></li>';

                        $result = $details. $edit. $delete;

                        return '<div class="btn-group open">
                        <a class="badge badge-primary dropdown-toggle" href="#" role="button"  data-toggle="dropdown">Actions<i class="ik ik-chevron-down mr-0 align-middle"></i></a>
                        <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">'.$result.'
                        </ul>
                    </div>';
                })

                ->addIndexColumn()
                ->rawColumns(['action'])
                ->toJson();
            }
         return view('backend.expense.index');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        return view('backend.expense.create');
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $messages = array(
            'expenseby.required' => 'Enter expense by',
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
            'purpose.required' => 'Enter purpose',
        );

        $this->validate($request, array(
            'date' => 'required',
            'expenseby' => 'required',
            'amount' => 'required',
            'purpose' => 'required',
            'image.*' => 'max:2048|mimes:jpeg,png,jpg',
        ), $messages);

        DB::beginTransaction();

        try {

            $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
            $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

            $data = new Expense();

            if ($request->hasfile('image')) {
                $file = $request->file('image');
                $filename = time() . $file->getClientOriginalName();
                $file->move(public_path('/backend/img/expense/'), $filename);
                $data->image = $filename;
            }

            $data->date               = Carbon::parse($request->date)->format('Y-m-d');
            $data->expenseby          = $request->expenseby;
            $data->amount             = str_replace($bn_numbers, $en_numbers, $request->amount);
            $data->purpose            = $request->purpose;
            $data->note               = $request->note;
            $data->save();

            $transaction = new Transaction();
            $transaction->expense_id            = $data->id;
            $transaction->date                  = Carbon::parse($request->date)->format('Y-m-d');
            $transaction->transaction_purpose   = $request->purpose;
            $transaction->transaction_type      = 10;
            $transaction->payment_type          = 2;
            $transaction->amount                = str_replace($bn_numbers, $en_numbers, $request->amount);
            $transaction->save();

            DB::commit();

            return redirect()->route('expense.index')
                ->with('success', 'Expense created successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $data = Expense::findOrFail($id);
        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

        $date = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->date)->format('d/m/Y'));
        return view('backend.expense.show', compact('data', 'date'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $data = Expense::findOrFail($id);
        return view('backend.expense.edit', compact('data'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $messages = array(
            'expenseby.required' => 'Enter expense by',
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
            'purpose.required' => 'Enter on account of',
        );

        $this->validate($request, array(
            'date' => 'required',
            'expenseby' => 'required',
            'amount' => 'required',
            'purpose' => 'required',
            'image.*' => 'max:2048|mimes:jpeg,png,jpg',
        ), $messages);

        DB::beginTransaction();

        try {
            $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
            $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

            $data = Expense::findOrFail($id);
            $image = $request->file('image');
            if($request->hasfile('image')) {
                $filename = time() . $image->getClientOriginalName();
                $image->move(public_path('/backend/img/expense/'), $filename);
                $data->image = $filename;
            }else{
                $data->image = $request->current_image;
            }

            $data->date               = Carbon::parse($request->date)->format('Y-m-d');
            $data->expenseby          = $request->expenseby;
            $data->amount             = str_replace($bn_numbers, $en_numbers, $request->amount);
            $data->purpose            = $request->purpose;
            $data->note               = $request->note;
            $data->update();

            $transaction = Transaction::where('expense_id', $id)->first();
            $transaction->expense_id            = $data->id;
            $transaction->date                  = Carbon::parse($request->date)->format('Y-m-d');
            $transaction->transaction_purpose   = $request->purpose;
            $transaction->transaction_type      = 10;
            $transaction->payment_type          = 2;
            $transaction->amount                = str_replace($bn_numbers, $en_numbers, $request->amount);
            $transaction->update();

            DB::commit();

            return redirect()->route('expense.index')
                ->with('success', 'Expense updated successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try{
            $data = Expense::findOrFail($id);

            if($data){
                $transaction = Transaction::where('expense_id', $id)->first();
                $transaction->delete();
                $data->delete();
            }

            return response()->json([
                'success' => true,
                'message' => 'Expense deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception,
            ]);
        }
    }

}
