<?php
namespace App\Http\Controllers;

use App\Models\Donar;
use App\Models\DonationType;
use App\Models\ExpenseType;
use App\Models\Fund;
use App\Models\FundDetail;
use App\Models\InvoiceDesign;
use App\Models\Result;
use App\Models\ResultDetail;
use App\Models\Setting;
use App\Models\SmsStatus;
use App\Models\Student;
use App\Models\StudentClass;
use App\Models\StudentExam;
use App\Models\StudentSession;
use App\Models\SubjectGrade;
use App\Traits\DateFormatter;
use App\Traits\DeleteTrait;
use App\Traits\SmsTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\Facades\DataTables;

class FundController extends Controller
{
    use SmsTrait;
    use DateFormatter;
    use DeleteTrait;

    public function funds(Request $request)
    {
        try {
            if ($request->ajax()) {

                $data = Fund::select('id', 'name', 'amount', 'fund_type')->get();

                return Datatables::of($data)

                    ->addColumn('type', function ($data) {
                        $type = '';
                        if ($data->fund_type == 1) {
                            $type = 'General';
                        } elseif ($data->fund_type == 2) {
                            $type = 'Guraba';
                        } elseif ($data->fund_type == 3) {
                            $type = 'Mahfil';
                        } elseif ($data->fund_type == 4) {
                            $type = 'Others';
                        }
                        return $type;
                    })

                    ->addColumn("balance", function ($data) {
                        $income  = FundDetail::where('payment_type', 1)->where('fund_id', $data->id)->sum('amount');
                        $expense = FundDetail::where('payment_type', 2)->where('fund_id', $data->id)->sum('amount');
                        $balance = $income - $expense;
                        return $balance;
                    })

                    ->addColumn('action', function ($data) {

                        $edit = '<a id="edit" href="' . route('fund.edit', $data->id) . ' " class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editClass"><i class="fa fa-edit"></i></a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '';

                        }

                        return $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['type', 'action'])
                    ->toJson();
            }
            $funds = Fund::all();
            return view('dashboard.funds.fund_list', compact('funds'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function FundStore(Request $request)
    {
        if ($request->ajax()) {
            $data = Validator::make($request->all(), [
                'name' => 'required|string|unique:funds,name,NULL,id,deleted_at,NULL',
            ]);

            if ($data->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $data->errors()->all(),
                ]);
            }

            DB::beginTransaction();

            try {
                $data             = new Fund();
                $data->name       = $request->name;
                $data->amount     = $request->amount;
                $data->fund_type  = $request->fund_type;
                $data->note       = $request->note;
                $data->created_by = Auth::user()->id;
                $data->save();

                $details               = new FundDetail();
                $details->fund_id      = $data->id;
                $details->amount       = $data->amount;
                $details->type         = 3;
                $details->payment_type = 1;
                $details->fund_type    = 1;
                $details->purpose      = 'Opening Balance';
                $details->date         = now()->format('Y-m-d');
                $details->created_by   = Auth::user()->id;
                $details->save();

                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => 'Fund created successfully',
                ]);

            } catch (\Exception $exception) {
                DB::rollBack();
                return response()->json([
                    'error'   => false,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }

    //fund transfer
    public function FundTransfer(Request $request)
    {
        DB::beginTransaction();
        try {
            $fromFundId = $request->from_fund;
            $toFundId   = $request->to_fund;
            $amount     = $request->amount;
            $note       = $request->note;

            $fromFund = Fund::findOrFail($fromFundId);
            $toFund   = Fund::findOrFail($toFundId);

            // From Fund transaction (expense)
            $fromFundTransaction               = new FundDetail();
            $fromFundTransaction->fund_id      = $fromFundId;
            $fromFundTransaction->amount       = $amount;
            $fromFundTransaction->type         = 18; // transfer
            $fromFundTransaction->payment_type = 2;  // expense
            $fromFundTransaction->fund_type    = $fromFund->fund_type;
            $fromFundTransaction->purpose      = "Transfer to " . $toFund->name;
            $fromFundTransaction->date         = now()->format('Y-m-d');
            $fromFundTransaction->note         = $note;
            $fromFundTransaction->created_by   = Auth::id();
            $fromFundTransaction->save();

            // To Fund transaction (income)
            $toFundTransaction               = new FundDetail();
            $toFundTransaction->fund_id      = $toFundId;
            $toFundTransaction->amount       = $amount;
            $toFundTransaction->type         = 18; // transfer
            $toFundTransaction->payment_type = 1;  // income
            $toFundTransaction->fund_type    = $toFund->fund_type;
            $toFundTransaction->purpose      = "Transfer from " . $fromFund->name;
            $toFundTransaction->date         = now()->format('Y-m-d');
            $toFundTransaction->note         = $note;
            $toFundTransaction->created_by   = Auth::id();
            $toFundTransaction->save();

            DB::commit();
            return back()->with('success', 'Fund transfer successfully');

        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Something went wrong: ' . $e->getMessage());
        }
    }

    public function FundEdit($id)
    {
        $data = Fund::findOrFail($id);
        return response()->json($data);
    }

    public function FundUpdate(Request $request, $id)
    {
        $data = Validator::make($request->all(), [
            'name' => 'required|string|unique:funds,name,' . $id . ',id,deleted_at,NULL',
        ]);

        if ($data->fails()) {
            return response()->json([
                'success' => false,
                'message' => $data->errors()->all(),
            ]);
        }

        DB::beginTransaction();

        try {
            $data             = Fund::findOrFail($id);
            $data->name       = $request->name;
            $data->amount     = $request->amount;
            $data->fund_type  = $request->fund_type;
            $data->note       = $request->note;
            $data->updated_by = Auth::user()->id;
            $data->save();

            $details               = FundDetail::where('fund_id', $data->id)->first();
            $details->fund_id      = $data->id;
            $details->amount       = $data->amount;
            $details->type         = 3;
            $details->payment_type = 1;
            $details->fund_type    = 1;
            $details->purpose      = 'Opening Balance';
            $details->date         = $details->date;
            $details->updated_by   = Auth::user()->id;
            $details->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Fund updated successfully',
            ]);

        } catch (\Exception $exception) {
            DB::rollBack();
            return response()->json([
                'error'   => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }

    public function FundDestroy($id)
    {
        try {
            $data   = Fund::findOrFail($id);
            $tables = DB::select("
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE COLUMN_NAME = 'fund_id'
            AND TABLE_SCHEMA = DATABASE()
            ");

            foreach ($tables as $table) {
                if ($table->TABLE_NAME !== 'funds') {
                    DB::table($table->TABLE_NAME)->where('fund_id', $data->id)->delete();
                }
            }

            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Fund deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Fund delete failed',
            ]);
        }
    }

    public function FundCollect(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = FundDetail::with('funds')->where('type', 1)->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('fund', function ($data) {
                        $fund = isset($data->funds) ? $data->funds->name : '--';
                        return $fund;
                    })

                    ->addColumn('donar', function ($data) {
                        $name  = $data->name ?? '';
                        $phone = $data->phone ?? '';
                        return $name . '<br>' . $phone;
                    })

                    ->addColumn('type', function ($data) {
                        $type = '';
                        if ($data->type == 1) {
                            $type = '<b>Income</b>';
                        } elseif ($data->type == 2) {
                            $type = '<b>Expense</b>';
                        } elseif ($data->type == 3) {
                            $type = '<b>Opening</b>';
                        }

                        return $type;
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? 0;
                    })

                    ->addColumn('action', function ($data) {

                        $result = '';

                        $details = '<li><a class="dropdown-item" href="' . route('fund.collect.show', $data->id) . ' " ><i class="fa fa-eye text-primary" ></i> Details</a></li>';

                        $edit = '<li><a class="dropdown-item" href="' . route('fund.collect.edit', $data->id) . ' "><i class="fa fa-edit text-success"></i> Edit</a></li>';

                        $delete = '<li><a class="dropdown-item btn-delete" href="#" data-remote=" ' . route('fund.collect.destroy', $data->id) . ' "><i class="fa fa-trash text-red"></i> Delete</a></li>';

                        $result = $details . $edit . $delete;

                        return '<div class="btn-group open">
                            <a class="badge badge-primary dropdown-toggle" href="#" role="button"  data-toggle="dropdown">Actions<i class="ik ik-chevron-down mr-0 align-middle"></i></a>
                            <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">' . $result . '
                            </ul>
                        </div>';
                    })

                    ->addIndexColumn()
                    ->rawColumns(['date', 'fund', 'donar', 'type', 'amount', 'action'])
                    ->toJson();
            }
            return view('dashboard.funds.fund_collect');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function FundCollectCreate()
    {
        try {
            $funds          = Fund::where('fund_type', 1)->get();
            $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                ->orderByDesc('receipt_no')
                ->first();
            $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;

            return view('dashboard.funds.fund_collect_create', compact('funds', 'receipt_no'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function GurabaCollectCreate()
    {
        try {
            $funds          = Fund::where('fund_type', 2)->get();
            $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                ->orderByDesc('receipt_no')
                ->first();
            $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;
            return view('dashboard.funds.guraba_collect_create', compact('funds', 'receipt_no'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function FundCollectStore(Request $request)
    {
        $messages = [
            'amount.required'  => 'Enter amount',
            'date.required'    => 'Enter date',
            'fund_id.required' => 'Select fund',
            'purpose.required' => 'Write purpose',
        ];

        $request->validate([
            'amount'  => 'required',
            'date'    => 'required',
            'fund_id' => 'required',
            'purpose' => 'required',
        ], $messages);

        try {
            $data               = new FundDetail();
            $data->receipt_no   = $request->receipt_no;
            $data->date         = $this->formatAnyDateToYmd($request->date);
            $data->fund_id      = $request->fund_id;
            $data->name         = $request->name;
            $data->phone        = $request->phone;
            $data->amount       = $request->amount;
            $data->purpose      = $request->purpose;
            $data->address      = $request->address;
            $data->note         = $request->note;
            $data->type         = 1;
            $data->fund_type    = $request->fund_type;
            $data->payment_type = 1;
            $data->created_by   = Auth::user()->id;
            $data->save();

            //  $msg = "দারুল উলুম সরকারপাড়া মাদরাসা, রিসিট নং : $request->receipt_no, তারিখ : $request->date, নাম : $request->name, মোবাইল : $request->phone, আপনি $month $request->amount টাকা পরিশোধ করেছেন, জাযাকাল্লাহ । ";

            // $this->SendSms($request->phone, $msg);

            $type = Fund::findOrFail($request->fund_id);
            if ($type->fund_type == 1) {
                return redirect()->route('general.fund')->with('success', 'Fund collected successfully');
            } elseif ($type->fund_type == 2) {
                return redirect()->route('guraba.fund')->with('success', 'Fund collected successfully');
            }

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function FundCollectShow($id)
    {
        try {
            $data = FundDetail::with('users:id,name', 'funds', 'students:id,name', 'donar')->findOrFail($id);
            return view('dashboard.funds.fund_collect_show', compact('data'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function FundCollectEdit($id)
    {
        try {
            $funds = Fund::where('fund_type', 1)->get();
            $data  = FundDetail::with('users:id,name', 'funds')->findOrFail($id);
            return view('dashboard.funds.fund_collect_edit', compact('data', 'funds'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function GurabaCollectEdit($id)
    {
        try {
            $funds = Fund::where('fund_type', 2)->get();
            $data  = FundDetail::with('users:id,name', 'funds')->findOrFail($id);
            return view('dashboard.funds.guraba_collect_edit', compact('data', 'funds'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function FundCollectPrint($id)
    {
        try {
            $data          = FundDetail::with('users:id,name', 'funds', 'donar:id,name')->findOrFail($id);
            $setting       = Setting::first();
            $amountInWords = $this->convertNumberToWordsInBangla($data->amount);
            $invoice       = InvoiceDesign::first();
            return view('dashboard.funds.guraba_collect_print', compact('data', 'setting', 'amountInWords', 'invoice'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function FundCollectUpdate(Request $request, $id)
    {
        $messages = [
            'amount.required'  => 'Enter amount',
            'date.required'    => 'Enter date',
            'fund_id.required' => 'Select fund',
            'purpose.required' => 'Write purpose',
        ];

        $request->validate([
            'amount'  => 'required',
            'date'    => 'required',
            'fund_id' => 'required',
            'purpose' => 'required',
        ], $messages);

        try {
            $data               = FundDetail::findOrFail($id);
            $data->receipt_no   = $request->receipt_no;
            $data->date         = $this->formatAnyDateToYmd($request->date);
            $data->fund_id      = $request->fund_id;
            $data->name         = $request->name;
            $data->phone        = $request->phone;
            $data->amount       = $request->amount;
            $data->purpose      = $request->purpose;
            $data->address      = $request->address;
            $data->note         = $request->note;
            $data->type         = 1;
            $data->payment_type = 1;
            $data->fund_type    = $request->fund_type;
            $data->created_by   = Auth::user()->id;
            $data->save();

            //$msg = "দারুল উলুম সরকারপাড়া মাদরাসা, রিসিট নং : $request->receipt_no, তারিখ : $request->date, নাম : $request->name, মোবাইল : $request->phone, আপনি $month $request->amount টাকা পরিশোধ করেছেন, জাযাকাল্লাহ । ";

            //$this->SendSms($request->phone, $msg);

            $type = Fund::findOrFail($request->fund_id);
            if ($type->fund_type == 1) {
                return redirect()->route('general.fund')->with('success', 'Fund collected successfully');
            } elseif ($type->fund_type == 2) {
                return redirect()->route('guraba.fund')->with('success', 'Fund collected successfully');
            }

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function FundCollectDestroy($id)
    {
        try {
            $data = FundDetail::findOrFail($id);
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Fund collecte deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Fund collecte delete failed',
            ]);
        }
    }

    public function FundExpense(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = FundDetail::with(['funds', 'extypes'])->where('type', 2)->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('fund', function ($data) {
                        $fund = isset($data->funds) ? $data->funds->name : '--';
                        return $fund;
                    })

                    ->addColumn('expenseBy', function ($data) {
                        $name  = $data->name ?? '';
                        $phone = $data->phone ?? '';
                        return $name . '<br>' . $phone;
                    })

                    ->addColumn('type', function ($data) {
                        $type = isset($data->extypes) ? $data->extypes->name : '--';
                        return $type;
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? 0;
                    })

                    ->addColumn('invoice', function ($data) {
                        return $data->invoice_no ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $details = '<a href="' . route('fund.expense.show', $data->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i> Details</a> ';
                        $edit    = '<a href="' . route('fund.expense.edit', $data->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i> Edit</a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<a href="#" data-remote="' . route('fund.expense.destroy', $data->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i> Delete</a>';
                        }

                        return $details . $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['date', 'fund', 'expenseBy', 'type', 'amount', 'action', 'invoice'])
                    ->toJson();
            }
            return view('dashboard.funds.fund_expense');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function FundExpenseCreate()
    {
        try {
            $funds          = Fund::all();
            $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                ->orderByDesc('receipt_no')
                ->first();
            $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;
            $types      = ExpenseType::all();
            return view('dashboard.funds.fund_expense_create', compact('funds', 'receipt_no', 'types'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function FundExpenseStore(Request $request)
    {
        $messages = [
            'date.required'    => 'Enter date',
            'amount.required'  => 'Enter amount',
            'fund_id.required' => 'Select fund',
            'name.required'    => 'Enter donar name',
            'purpose.required' => 'Write purpose',
        ];

        $request->validate([
            'amount'  => 'required',
            'fund_id' => 'required',
            'name'    => 'required',
            'purpose' => 'required',
            'date'    => 'required',
        ], $messages);

        try {
            $data = new FundDetail();

            if ($request->hasfile('cash_memo')) {
                $file     = $request->file('cash_memo');
                $filename = time() . $file->getClientOriginalName();
                $file->move(public_path('/backend/expense/'), $filename);
                $data->cash_memo = $filename;
            }
            $data->invoice_no = $request->invoice_no;
            $data->receipt_no = $request->receipt_no;
            $data->date       = $this->formatAnyDateToYmd($request->date);

            $data->fund_id         = $request->fund_id;
            $data->name            = $request->name;
            $data->phone           = $request->phone;
            $data->amount          = $request->amount;
            $data->purpose         = $request->purpose;
            $data->address         = $request->address;
            $data->note            = $request->note;
            $data->expense_type_id = $request->expense_type_id;
            $data->type            = 2;
            $data->payment_type    = 2;
            $data->created_by      = Auth::user()->id;
            $data->save();

            //  $msg = "দারুল উলুম সরকারপাড়া মাদরাসা, রিসিট নং : $request->receipt_no, তারিখ : $request->date, নাম : $request->name, মোবাইল : $request->phone, আপনি $month $request->amount টাকা পরিশোধ করেছেন, জাযাকাল্লাহ । ";

            // $this->SendSms($request->phone, $msg);

            return redirect()->route('fund.expense')->with('success', 'Expense created successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function FundExpenseShow($id)
    {
        try {
            $data    = FundDetail::with('users:id,name', 'funds')->findOrFail($id);
            $invoice = InvoiceDesign::first();
            return view('dashboard.funds.fund_expense_show', compact('data', 'invoice'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function FundExpenseEdit($id)
    {
        try {
            $funds = Fund::all();
            $types = ExpenseType::all();
            $data  = FundDetail::with('users:id,name', 'funds', 'extypes')->findOrFail($id);
            return view('dashboard.funds.fund_expense_edit', compact('data', 'funds', 'types'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function FundExpenseUpdate(Request $request, $id)
    {
        $messages = [
            'amount.required'  => 'Enter amount',
            'fund_id.required' => 'Select fund',
            'name.required'    => 'Enter donar name',
            'purpose.required' => 'Write purpose',
        ];

        $request->validate([
            'amount'  => 'required',
            'fund_id' => 'required',
            'name'    => 'required',
            'purpose' => 'required',
        ], $messages);

        try {
            $data = FundDetail::findOrFail($id);

            $image = $request->file('cash_memo');
            if ($request->hasfile('cash_memo')) {
                $filename = time() . $image->getClientOriginalName();
                $image->move(public_path('/backend/expense/'), $filename);
                $data->cash_memo = $filename;
            } else {
                $data->cash_memo = $request->current_image;
            }

            $data->receipt_no      = $request->receipt_no;
            $data->invoice_no      = $request->invoice_no;
            $data->date            = $this->formatAnyDateToYmd($request->date);
            $data->fund_id         = $request->fund_id;
            $data->name            = $request->name;
            $data->phone           = $request->phone;
            $data->amount          = $request->amount;
            $data->purpose         = $request->purpose;
            $data->address         = $request->address;
            $data->note            = $request->note;
            $data->expense_type_id = $request->expense_type_id;
            $data->type            = 2;
            $data->payment_type    = 2;
            $data->created_by      = Auth::user()->id;
            $data->save();

            //$msg = "দারুল উলুম সরকারপাড়া মাদরাসা, রিসিট নং : $request->receipt_no, তারিখ : $request->date, নাম : $request->name, মোবাইল : $request->phone, আপনি $month $request->amount টাকা পরিশোধ করেছেন, জাযাকাল্লাহ । ";

            //$this->SendSms($request->phone, $msg);

            return redirect()->route('fund.expense')->with('success', 'Expense updated successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function FundExpenseDestroy($id)
    {
        try {
            $data = FundDetail::findOrFail($id);
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Expense deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Expense delete failed',
            ]);
        }
    }

    public function FundExpenseCashMemoDownload($id)
    {
        $data = FundDetail::findOrFail($id);
        $path = storage_path('backend/expense/' . $data->cash_memo);
        if (file_exists($path)) {
            return response()->download($path);
        } else {
            return abort(404);
        }
    }

    public function ExpenseType(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = DB::table('expense_types')
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'desc')->get();

                return Datatables::of($data)

                    ->addColumn('action', function ($data) {

                        $edit = '<a id="edit" href="' . route('expense.type.edit', $data->id) . ' " class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editClass"><i class="fa fa-edit"></i></a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('expense.type.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }

                        return $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action'])
                    ->toJson();
            }
            return view('dashboard.expense.expense_type');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function ExpenseTypeStore(Request $request)
    {
        if ($request->ajax()) {

            $data = Validator::make($request->all(), [
                'name' => 'required|string|unique:expense_types,name,NULL,id,deleted_at,NULL',
            ]);

            if ($data->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $data->errors()->all(),
                ]);
            }

            try {
                $data       = new ExpenseType();
                $data->name = $request->name;
                $data->save();

                return response()->json([
                    'success' => true,
                    'message' => 'Expense type created successfully',
                ]);

            } catch (\Exception $exception) {
                return response()->json([
                    'success' => true,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }

    public function ExpenseTypeEdit($id)
    {
        $data = ExpenseType::findOrFail($id);
        return response()->json($data);
    }

    public function ExpenseTypeUpdate(Request $request, $id)
    {
        $data = Validator::make($request->all(), [
            'name' => 'required|string|unique:expense_types,name,' . $id . ',id,deleted_at,NULL',
        ]);

        if ($data->fails()) {
            return response()->json([
                'success' => false,
                'message' => $data->errors()->all(),
            ]);
        }

        try {
            $data       = ExpenseType::findOrFail($id);
            $data->name = $request->name;
            $data->update();

            return response()->json([
                'success' => true,
                'message' => 'Expense type updated successfully',
            ]);

        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function ExpenseTypeDestroy($id)
    {
        try {
            $data   = ExpenseType::findOrFail($id);
            $tables = DB::select("
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE COLUMN_NAME = 'expense_type_id'
            AND TABLE_SCHEMA = DATABASE()
            ");

            foreach ($tables as $table) {
                if ($table->TABLE_NAME !== 'expense_types') {
                    DB::table($table->TABLE_NAME)->where('expense_type_id', $data->id)->delete();
                }
            }

            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Expense type deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Expense type delete failed',
            ]);
        }
    }

// Controller method: FundBalanceSheet
// use Carbon\Carbon; এবং প্রযোজ্য Model গুলো উপরে use করে নেবেন।

public function FundBalanceSheet(Request $request)
{
    try {
        if ($request->ajax()) {
            $fund_id           = $request->fund_id;
            $start_date        = $request->start_date ? \Carbon\Carbon::createFromFormat('d/m/Y', $request->start_date)->format('Y-m-d') : null;
            $end_date          = $request->end_date ? \Carbon\Carbon::createFromFormat('d/m/Y', $request->end_date)->format('Y-m-d') : null;
            $searchTerm        = $request->search['value'] ?? null;   // DataTables global search
            $purposeSearchTerm = $request->search_term ?? null;       // Item Search (existing)
            $typeFilter        = $request->type_filter ?? null;       // NEW: 1=Income, 2=Expense
            $nameTerm          = $request->name_term ?? null;         // NEW: donor/student/teacher/manual name

            $fundName = null;
            if ($fund_id) {
                $fund     = \App\Models\Fund::find($fund_id);
                $fundName = $fund ? $fund->name : null;
            }

            // Previous balance (opening আলাদা না ধরলেও, আগের মতোই)
            if ($start_date) {
                $income = \App\Models\FundDetail::where('fund_id', $fund_id)
                    ->where('payment_type', 1)
                    ->where('date', '<', $start_date)
                    ->sum('amount');

                $expense = \App\Models\FundDetail::where('fund_id', $fund_id)
                    ->where('payment_type', 2)
                    ->where('date', '<', $start_date)
                    ->sum('amount');

                // opening = type/payment_type 3 — prevBalance এ না ধরলেও চলবে
                $previous_balance = $income - $expense;
            } else {
                $previous_balance = 0;
            }

            $dataQuery = \App\Models\FundDetail::with([
                    'funds',
                    'students:id,name',
                    'teacher:id,name,phone',
                    'donar:id,name,phone'
                ])
                ->where('fund_id', $fund_id);

            // Date filters
            if ($start_date && $end_date) {
                $dataQuery->whereBetween('date', [$start_date, $end_date]);
            } elseif ($start_date) {
                $dataQuery->where('date', '>=', $start_date);
            } elseif ($end_date) {
                $dataQuery->where('date', '<=', $end_date);
            }

            // NEW: Type filter (Income/Expense only)
            if ($typeFilter && in_array((int)$typeFilter, [1, 2], true)) {
                $dataQuery->where('payment_type', (int)$typeFilter);
            }

            // NEW: Name filter across donor/student/teacher + manual name column
            if (!empty($nameTerm)) {
                $nameTermLike = '%' . $nameTerm . '%';
                $dataQuery->where(function ($q) use ($nameTermLike) {
                    $q->whereHas('donar',     fn($s) => $s->where('name', 'like', $nameTermLike))
                      ->orWhereHas('students',fn($s) => $s->where('name', 'like', $nameTermLike))
                      ->orWhereHas('teacher', fn($s) => $s->where('name', 'like', $nameTermLike))
                      ->orWhere('name', 'like', $nameTermLike); // manual name on FundDetail
                });
            }

            // DT global search
            if ($searchTerm) {
                $dataQuery->where(function ($q) use ($searchTerm) {
                    $q->where('date', 'like', '%' . $searchTerm . '%')
                      ->orWhere('purpose', 'like', '%' . $searchTerm . '%')
                      ->orWhere('amount', 'like', '%' . $searchTerm . '%')
                      ->orWhere('fund_id', 'like', '%' . $searchTerm . '%');
                });
            }

            // Item search (purpose / date)
            if ($purposeSearchTerm) {
                $dataQuery->where(function ($q) use ($purposeSearchTerm) {
                    $q->where('date', 'like', '%' . $purposeSearchTerm . '%')
                      ->orWhere('purpose', 'like', '%' . $purposeSearchTerm . '%');
                });
            }

            $dataQuery->orderBy('date', 'asc');
            $data = $dataQuery->get();

            return \DataTables::of($data)
                ->addColumn('date', fn(\App\Models\FundDetail $d) => \Carbon\Carbon::parse($d->date)->format('d-M-Y'))
                ->addColumn('income', function (\App\Models\FundDetail $d) {
                    // opening (payment_type=3) আগের মতোই income কলামে দেখাতে চাইলে এই লাইন রাখুন
                    return (($d->payment_type == 1 || $d->payment_type == 3) && !empty($d->amount)) ? $d->amount : '00.00';
                })
                ->addColumn('expense', fn(\App\Models\FundDetail $d) => ($d->payment_type == 2 && !is_null($d->amount)) ? $d->amount : '00.00')
                ->addColumn('purpose', function (\App\Models\FundDetail $d) {
                    $type  = $d->type;
                    $names = [];
                    switch ($type) {
                        case 1: case 4: case 6: case 14:
                            if ($d->students) { $names[] = '<strong>'.e($d->students->name).'</strong>'; }
                            break;
                        case 13:
                            if ($d->teacher) { $names[] = '<strong>'.e($d->teacher->name).'</strong>'; }
                            break;
                        case 3:
                            if ($d->funds) { $names[] = '<strong>'.e($d->funds->name).'</strong>'; }
                            break;
                        case 16:
                            if ($d->donar)    { $names[] = '<strong>'.e($d->donar->name).'</strong>'; }
                            if ($d->students) { $names[] = '<strong>'.e($d->students->name).'</strong>'; }
                            break;
                        case 5:
                            if ($d->donar) { $names[] = '<strong>'.e($d->donar->name).'</strong>'; }
                            break;
                    }
                    $related = implode(' | ', $names);
                    $purpose = e($d->purpose ?: 'Opening Amount');
                    return $related ? "$related - $purpose" : $purpose;
                })
                ->addColumn('current_balance', function ($d) use (&$previous_balance) {
                    if ($d->payment_type == 1 || $d->payment_type == 3) {
                        $previous_balance += $d->amount;
                    } elseif ($d->payment_type == 2) {
                        $previous_balance -= $d->amount;
                    }
                    return $previous_balance;
                })
                ->with('prevBalance', $previous_balance)
                ->with('fundName', $fundName)
                ->rawColumns(['income','expense','current_balance','purpose'])
                ->toJson();
        }

        $invoice = \App\Models\InvoiceDesign::first();
        return view('dashboard.funds.fund_balance_sheet', compact('invoice'));
    } catch (\Exception $exception) {
        return redirect()->back()->with('error', $exception->getMessage());
    }
}

    public function GetFundOpeningBalance(Request $request, $id)
    {
        if ($request->ajax()) {
            $amount = Fund::find($id)->amount;
            return response()->json($amount);
        }
    }

    public function FindFund(Request $request)
    {
        $funds = Fund::where('fund_type', $request->fund_type)->get();
        return response()->json($funds);
    }
    public function FundAmount(Request $request)
    {
        $income           = FundDetail::where('payment_type', 1)->where('fund_id', $request->fund_id)->sum('amount');
        $expense          = FundDetail::where('payment_type', 2)->where('fund_id', $request->fund_id)->sum('amount');
        $availableBalance = $income - $expense;
        return response()->json($availableBalance);
    }

    public function sonad(Request $request)
    {
        try {

            if ($request->ajax()) {

                $query = FundDetail::with('funds', 'students:id,name,class_id,register_no', 'users:id,name')->where('type', 4);

                if ($request->class_id) {
                    $query->whereHas('students', function ($q) use ($request) {
                        $q->where('class_id', $request->class_id);
                    });
                }

                if ($request->session_id) {
                    $query->whereHas('students', function ($q) use ($request) {
                        $q->where('session_id', $request->session_id);
                    });
                }

                $data = $query->whereNull('deleted_at')->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('fund', function ($data) {
                        $fund = isset($data->funds) ? $data->funds->name : '--';
                        return $fund;
                    })

                    ->addColumn('student', function ($data) {
                        $name  = isset($data->students) ? $data->students->name : '--';
                        $roll  = isset($data->students) ? $data->students->register_no : '--';
                        $class = isset($data->students) ? $data->students->stuclass->name : '--';
                        return $name . '<br>' . $class . '<br>' . $roll;
                    })

                    ->addColumn('givedBy', function ($data) {
                        return $data->users->name ?? '';;
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? 0;
                    })

                    ->addColumn('action', function ($data) {
                        $details = '<a href="' . route('sonad.show', $data->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i></a> ';
                        $edit    = '<a href="' . route('sonad.edit', $data->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i></a> ';
                        $print   = '<a href="' . route('sonad.print', $data->id) . '" class="btn btn-sm btn-primary" title="A4 Page Print"><i class="fa fa-file"></i></a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<a href="#" data-remote="' . route('sonad.destroy', $data->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i></a>';
                        }

                        return $details . $edit . $print . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['date', 'fund', 'givedBy', 'student', 'amount', 'action'])
                    ->toJson();
            }
            $classes  = StudentClass::all();
            $sessions = StudentSession::all();
            return view('dashboard.funds.sonad', compact('classes', 'sessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function SonadCreate()
    {
        try {
            $funds          = Fund::all();
            $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                ->orderByDesc('receipt_no')
                ->first();
            $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;
            $students   = Result::with(['students:id,name,class_id', 'stusession', 'exam', 'stusession'])
                ->groupBy('student_id')
                ->get();
            return view('dashboard.funds.sonad_create', compact('funds', 'receipt_no', 'students'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function SonadStore(Request $request)
    {
        $messages = [
            'amount.required'     => 'Enter amount',
            'date.required'       => 'Enter date',
            'fund_id.required'    => 'Select fund',
            'student_id.required' => 'Select fund',
        ];

        $request->validate([
            'amount'     => 'required',
            'date'       => 'required',
            'student_id' => 'required',
            'fund_id'    => 'required',
        ], $messages);

        try {

            $student = Student::findOrFail($request->student_id);

            $existing = FundDetail::where('student_id', $request->student_id)
                ->where('type', 4)
                ->where('class_id', $student->class_id)
                ->where('exam_id', $request->exam_id)
                ->where('session_id', $student->session_id)
                ->exists();

            if ($existing) {
                return redirect()->back()->with('error', 'Sonad has already been given.');
            }

            $data               = new FundDetail();
            $data->receipt_no   = $request->receipt_no;
            $data->date         = $this->formatAnyDateToYmd($request->date);
            $data->student_id   = $request->student_id;
            $data->fund_id      = $request->fund_id;
            $data->amount       = $request->amount;
            $data->exam_id      = $request->exam_id;
            $data->class_id     = $student->class_id;
            $data->session_id   = $student->session_id;
            $data->note         = $request->note;
            $data->type         = 4;
            $data->purpose      = 'Sonad Collect Amount';
            $data->payment_type = 1;
            $data->created_by   = Auth::user()->id;
            $data->save();

            $exam    = StudentExam::findOrFail($data->exam_id);
            $student = Student::findOrFail($data->student_id);
            $site    = DB::table('settings')->first();
            $to      = $student->gurdian_mobile;

            $text   = $student->name . ' আপনি ' . $exam->name . ' ' . $site->name . ' থেকে সনদপ্রাপ্ত হয়েছেন';
            $status = optional(SmsStatus::first())->student_sonad;
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            return redirect()->route('sonad')->with('success', 'Sonad created successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function SonadShow($id)
    {
        try {
            $data = FundDetail::with('users:id,name', 'funds', 'students', 'exam')->findOrFail($id);
            return view('dashboard.funds.sonad_show', compact('data'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function SonadPrint($id)
    {
        try {
            $data = FundDetail::with('users:id,name', 'funds', 'students', 'exam')
                ->findOrFail($id);
            $setting = Setting::first();

            $resultData = Result::where('student_id', $data->student_id)->first();

            $results = ResultDetail::with('subjects:id,name')->where('result_id', $resultData->id)->get();
            $grades  = SubjectGrade::all();

            $totalMark = 0;
            $totalSub  = $results->count();

            foreach ($results as $result) {
                $grade = $grades->filter(function ($grade) use ($result) {
                    return (int) $result->mark >= (int) $grade->mark;
                })->sortByDesc('mark')->first();

                $result->grade = $grade ? $grade->name : 'N/A';
                $totalMark += (int) ($result->mark ?? 0);
            }

            $avgGrade = $grades->filter(function ($grade) use ($totalMark, $totalSub) {
                $averageMark = $totalSub > 0 ? $totalMark / $totalSub : 0;
                return $averageMark >= (int) $grade->mark;
            })->sortByDesc('mark')->first();

            $invoice = InvoiceDesign::first();

            return view('dashboard.funds.sonad_print', compact('data', 'setting', 'results', 'avgGrade', 'totalMark', 'invoice'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function SonadEdit($id)
    {
        try {
            $funds = Fund::all();
            $data  = FundDetail::with(['users:id,name', 'funds'])->findOrFail($id);

            $students = Result::with([
                'students:id,name,class_id',
                'students.stuclass:id,name',
                'stusession:id,name',
            ])
                ->select('student_id')
                ->groupBy('student_id')
                ->get();

            $exams = Result::with([
                'exam:id,name',
            ])
                ->where('class_id', $data->class_id)
                ->where('session_id', $data->session_id)
                ->where('student_id', $data->student_id)
                ->select('exam_id')
                ->groupBy('exam_id')
                ->get();

            return view('dashboard.funds.sonad_edit', compact('funds', 'data', 'students', 'exams'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function SonadUpdate(Request $request, $id)
    {
        $messages = [
            'amount.required'     => 'Enter amount',
            'fund_id.required'    => 'Select fund',
            'student_id.required' => 'Select fund',
        ];

        $request->validate([
            'amount'     => 'required',
            'student_id' => 'required',
            'fund_id'    => 'required',
        ], $messages);

        try {

            $student = Student::findOrFail($request->student_id);

            $existing = FundDetail::where('student_id', $request->student_id)
                ->where('type', 4)
                ->where('class_id', $student->class_id)
                ->where('exam_id', $request->exam_id)
                ->where('session_id', $student->session_id)
                ->where('id', '!=', $id)
                ->exists();

            if ($existing) {
                return redirect()->back()->with('error', 'Sonad has already been given.');
            }

            $data               = FundDetail::findOrFail($id);
            $data->receipt_no   = $request->receipt_no;
            $data->date         = $this->formatAnyDateToYmd($request->date);
            $data->student_id   = $request->student_id;
            $data->fund_id      = $request->fund_id;
            $data->amount       = $request->amount;
            $data->exam_id      = $request->exam_id;
            $data->note         = $request->note;
            $data->type         = 4;
            $data->purpose      = 'Sonad Collect Amount';
            $data->payment_type = 1;
            $data->created_by   = Auth::user()->id;
            $data->save();

            $exam    = StudentExam::findOrFail($data->exam_id);
            $student = Student::findOrFail($data->student_id);
            $site    = DB::table('settings')->first();
            $to      = $student->gurdian_mobile;

            $text   = $student->name . ' আপনি ' . $exam->name . ' ' . $site->name . ' থেকে সনদপ্রাপ্ত হয়েছেন';
            $status = optional(SmsStatus::first())->student_sonad;
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            return redirect()->route('sonad')->with('success', 'Sonad updated successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function SonadDestroy($id)
    {
        try {
            $data = FundDetail::findOrFail($id);

            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Sonad deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Sonad delete failed',
            ]);
        }
    }

    public function DonationType(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = DB::table('donation_types')
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'desc')->get();

                return Datatables::of($data)

                    ->addColumn('action', function ($data) {

                        $edit = '<a id="edit" href="' . route('donation.type.edit', $data->id) . ' " class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editClass"><i class="fa fa-edit"></i></a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('donation.type.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }

                        return $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action'])
                    ->toJson();
            }
            return view('dashboard.donations.donation_type');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function DonationTypeStore(Request $request)
    {
        if ($request->ajax()) {

            $data = Validator::make($request->all(), [
                'name' => 'required|string|unique:donation_types,name,NULL,id,deleted_at,NULL',
            ]);

            if ($data->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $data->errors()->all(),
                ]);
            }

            try {
                $data       = new DonationType();
                $data->name = $request->name;
                $data->save();

                return response()->json([
                    'success' => true,
                    'message' => 'Donation type created successfully',
                ]);

            } catch (\Exception $exception) {
                return response()->json([
                    'success' => true,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }

    public function DonationTypeEdit($id)
    {
        $data = DonationType::findOrFail($id);
        return response()->json($data);
    }

    public function DonationTypeUpdate(Request $request, $id)
    {
        $data = Validator::make($request->all(), [
            'name' => 'required|string|unique:donation_types,name,' . $id . ',id,deleted_at,NULL',
        ]);

        if ($data->fails()) {
            return response()->json([
                'success' => false,
                'message' => $data->errors()->all(),
            ]);
        }

        try {
            $data       = DonationType::findOrFail($id);
            $data->name = $request->name;
            $data->update();

            return response()->json([
                'success' => true,
                'message' => 'Donation type updated successfully',
            ]);

        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function DonationTypeDestroy($id)
    {
        try {
            $data   = DonationType::findOrFail($id);
            $tables = DB::select("
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE COLUMN_NAME = 'donation_type_id'
            AND TABLE_SCHEMA = DATABASE()
            ");

            foreach ($tables as $table) {
                if ($table->TABLE_NAME !== 'donation_types') {
                    DB::table($table->TABLE_NAME)->where('donation_type_id', $data->id)->delete();
                }
            }

            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Donation type deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Donation type delete failed',
            ]);
        }
    }

    public function donation(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = FundDetail::with('funds', 'users:id,name', 'donationType')->where('type', 5)->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('fund', function ($data) {
                        $fund = isset($data->funds) ? $data->funds->name : '--';
                        return $fund;
                    })

                    ->addColumn('type', function ($data) {
                        $name = isset($data->donationType) ? $data->donationType->name : '--';
                        return $name;
                    })

                    ->addColumn('donar', function ($data) {
                        $name  = $data->name ? $data->name : '--';
                        $phone = $data->phone ? $data->phone : '--';
                        return $name . '<br>' . $phone;
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? 0;
                    })

                    ->addColumn('action', function ($data) {
                        $details = '<a href="' . route('donation.show', $data->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i> Details</a> ';
                        $edit    = '<a href="' . route('donation.edit', $data->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i> Edit</a> ';
                        $print   = '<a href="' . route('donation.print', $data->id) . '" class="btn btn-sm btn-primary" title="Print"><i class="fa fa-file text-danger"></i> Print</a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<a href="#" data-remote="' . route('donation.destroy', $data->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i> Delete</a>';
                        }

                        return $details . $edit . $print . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['date', 'fund', 'donar', 'amount', 'type', 'action'])
                    ->toJson();
            }
            return view('dashboard.donations.donation_list');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function DonationCreate()
    {
        try {
            $funds          = Fund::all();
            $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                ->orderByDesc('receipt_no')
                ->first();
            $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;
            $types      = DonationType::all();
            return view('dashboard.donations.donation_create', compact('funds', 'receipt_no', 'types'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function DonationStore(Request $request)
    {
        $messages = [
            'amount.required'           => 'Enter amount',
            'date.required'             => 'Enter date',
            'fund_id.required'          => 'Select fund',
            'donation_type_id.required' => 'Select fund',
        ];

        $request->validate([
            'amount'           => 'required',
            'date'             => 'required',
            'donation_type_id' => 'required',
            'fund_id'          => 'required',
        ], $messages);

        try {
            $data                   = new FundDetail();
            $data->receipt_no       = $request->receipt_no;
            $data->date             = Carbon::parse($request->date)->format('Y-m-d');
            $data->donation_type_id = $request->donation_type_id;
            $data->fund_id          = $request->fund_id;
            $data->name             = $request->name;
            $data->phone            = $request->phone;
            $data->purpose          = $request->purpose;
            $data->amount           = $request->amount;
            $data->address          = $request->address;
            $data->note             = $request->note;
            $data->type             = 5;
            $data->payment_type     = 1;
            $data->fund_type        = 2;
            $data->created_by       = Auth::user()->id;
            $data->save();

            $status = optional(SmsStatus::first())->donation;
            $site   = DB::table('settings')->first();
            $to     = $data->phone;
            $text   = $site->name . ', ' . $data->name . ', আপনার ' . $request->purpose . ' ' . $data->amount . ' tk , আমরা গ্রহণ করেছি';
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            return redirect()->route('donations')->with('success', 'Donation created successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function DonationShow($id)
    {
        try {
            $data = FundDetail::with('users:id,name', 'funds')->findOrFail($id);
            return view('dashboard.donations.donation_show', compact('data'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function DonationPrint($id)
    {
        try {
            $data          = FundDetail::with('users:id,name', 'funds')->findOrFail($id);
            $setting       = Setting::first();
            $amountInWords = $this->convertNumberToWordsInBangla($data->amount);
            $invoice       = InvoiceDesign::first();
            return view('dashboard.donations.donation_print', compact('data', 'setting', 'amountInWords', 'invoice'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    private function convertNumberToWordsInBangla($number)
    {
        $units     = ['', 'এক', 'দুই', 'তিন', 'চার', 'পাঁচ', 'ছয়', 'সাত', 'আট', 'নয়', 'দশ', 'এগারো', 'বারো', 'তেরো', 'চোদ্দো', 'পনেরো', 'ষোলো', 'সতেরো', 'আঠারো', 'উনিশ'];
        $tens      = ['', '', 'বিশ', 'ত্রিশ', 'চল্লিশ', 'পঞ্চাশ', 'ষাট', 'সত্তর', 'অশি', 'নব্বই'];
        $thousands = ['', 'হাজার', 'লক্ষ', 'কোটি'];

        if ($number == 0) {
            return 'শূন্য';
        }

        $result = '';
        $crore  = floor($number / 10000000);
        $number %= 10000000;

        $lakh = floor($number / 100000);
        $number %= 100000;

        $thousand = floor($number / 1000);
        $number %= 1000;

        $hundred = floor($number / 100);
        $number %= 100;

        $ten  = floor($number / 10);
        $unit = $number % 10;

        if ($crore > 0) {
            $result .= $this->convertGroupInBangla($crore, 3, $units, $tens, $thousands) . ' ';
        }

        if ($lakh > 0) {
            $result .= $this->convertGroupInBangla($lakh, 2, $units, $tens, $thousands) . ' ';
        }

        if ($thousand > 0) {
            $result .= $this->convertGroupInBangla($thousand, 1, $units, $tens, $thousands) . ' ';
        }

        if ($hundred > 0) {
            $result .= $units[$hundred] . ' শত ';
        }

        if ($number > 0) {
            if ($number < 20) {
                $result .= $units[$number] . ' ';
            } else {
                $result .= $tens[$ten] . ' ';
                if ($unit > 0) {
                    $result .= $units[$unit] . ' ';
                }
            }
        }

        return trim($result);
    }

    private function convertGroupInBangla($number, $groupIndex, $units, $tens, $thousands)
    {
        $result = '';

        if ($number == 0) {
            return '';
        }

        if ($number < 20) {
            $result .= $units[$number] . ' ';
        } elseif ($number < 100) {
            $ten  = floor($number / 10);
            $unit = $number % 10;
            $result .= $tens[$ten] . ' ';
            if ($unit > 0) {
                $result .= $units[$unit] . ' ';
            }
        }

        if ($groupIndex > 0) {
            $result .= $thousands[$groupIndex] . ' ';
        }

        return trim($result);
    }

    private function convertGroup($number, $groupIndex, $units, $tens, $thousands)
    {
        $result = '';

        if ($number == 0) {
            return '';
        }

        if ($number >= 100) {
            $result .= $units[intval($number / 100)] . ' Hundred ';
            $number %= 100;
        }

        if ($number >= 20) {
            $result .= $tens[intval($number / 10)] . ' ';
            $number %= 10;
        }

        if ($number > 0) {
            $result .= $units[$number] . ' ';
        }

        if ($groupIndex > 0) {
            $result .= $thousands[$groupIndex] . ' ';
        }

        return trim($result);
    }

    public function DonationEdit($id)
    {
        try {
            $funds = Fund::all();
            $data  = FundDetail::with('users:id,name', 'funds')->findOrFail($id);
            $types = DonationType::all();
            return view('dashboard.donations.donation_edit', compact('funds', 'data', 'types'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function DonationUpdate(Request $request, $id)
    {
        $messages = [
            'amount.required'           => 'Enter amount',
            'date.required'             => 'Enter date',
            'fund_id.required'          => 'Select fund',
            'donation_type_id.required' => 'Select fund',
        ];

        $request->validate([
            'amount'           => 'required',
            'date'             => 'required',
            'donation_type_id' => 'required',
            'fund_id'          => 'required',
        ], $messages);

        try {
            $data                   = FundDetail::findOrFail($id);
            $data->receipt_no       = $request->receipt_no;
            $data->date             = Carbon::parse($request->date)->format('Y-m-d');
            $data->donation_type_id = $request->donation_type_id;
            $data->fund_id          = $request->fund_id;
            $data->name             = $request->name;
            $data->phone            = $request->phone;
            $data->purpose          = $request->purpose;
            $data->amount           = $request->amount;
            $data->address          = $request->address;
            $data->type             = 5;
            $data->payment_type     = 1;
            $data->fund_type        = 2;
            $data->note             = $request->note;
            $data->created_by       = Auth::user()->id;
            $data->save();

            $status = optional(SmsStatus::first())->donation;
            $site   = DB::table('settings')->first();
            $to     = $data->phone;
            $text   = $site->name . ', ' . $data->name . ', আপনার ' . $request->purpose . ' ' . $data->amount . ' tk , আমরা গ্রহণ করেছি';
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            return redirect()->route('donations')->with('success', 'Donation updated successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function DonationDestroy($id)
    {
        try {
            $data = FundDetail::findOrFail($id);
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Donation deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Donation delete failed',
            ]);
        }
    }

    public function AnnualAccounting(Request $request)
    {
        try {
            if ($request->ajax()) {
                $fund_id = $request->fund_id;
                $year    = $request->year;

                if (! $year) {
                    return response()->json(['error' => 'Year is required'], 422);
                }

                $start_date = Carbon::createFromDate($year, 1, 1)->format('Y-m-d');
                $end_date   = Carbon::createFromDate($year, 12, 31)->format('Y-m-d');

                if ($fund_id) {
                    $income = FundDetail::where('fund_id', $fund_id)
                        ->where('payment_type', 1)
                        ->whereBetween('date', [$start_date, $end_date])
                        ->sum('amount');

                    $expense = FundDetail::where('fund_id', $fund_id)
                        ->where('payment_type', 2)
                        ->whereBetween('date', [$start_date, $end_date])
                        ->sum('amount');

                    $balance = $income - $expense;
                    $name    = Fund::findOrFail($fund_id);

                    return response()->json([
                        'name'    => $name->name,
                        'income'  => round($income),
                        'expense' => round($expense),
                        'balance' => round($balance),
                        'year'    => $year,
                    ]);
                } else {
                    $funds   = Fund::all();
                    $results = [];

                    foreach ($funds as $fund) {
                        $income = FundDetail::where('fund_id', $fund->id)
                            ->where('payment_type', 1)
                            ->whereBetween('date', [$start_date, $end_date])
                            ->sum('amount');

                        $expense = FundDetail::where('fund_id', $fund->id)
                            ->where('payment_type', 2)
                            ->whereBetween('date', [$start_date, $end_date])
                            ->sum('amount');

                        $balance = $income - $expense;

                        $results[] = [
                            'name'    => $fund->name,
                            'income'  => round($income),
                            'expense' => round($expense),
                            'balance' => round($balance),
                            'year'    => $year,
                        ];
                    }

                    return response()->json(['funds' => $results]);
                }
            }

            $funds = Fund::all();
            return view('dashboard.funds.annual_balance', compact('funds'));

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function ReceiptAndPayment(Request $request)
    {
        try {
            if ($request->ajax()) {
                $parseDate = function ($str) {
                    if (empty($str)) {
                        return null;
                    }

                    try {
                        return \Carbon\Carbon::createFromFormat('d/m/Y', $str)->format('Y-m-d');
                    } catch (\Throwable $e) {
                        return null;
                    }
                };

                $start_date = $parseDate($request->start_date);
                $end_date   = $parseDate($request->end_date);

                $fundTypes = [
                    1 => 'General',
                    2 => 'Guraba',
                    3 => 'Mahfil',
                    4 => 'Other',
                ];

                $rows = [];
                foreach ($fundTypes as $typeId => $typeLabel) {

                    // common base query (join + guards)
                    $base = DB::table('fund_details as fd')
                        ->join('funds as f', 'fd.fund_id', '=', 'f.id')
                        ->where('f.fund_type', $typeId)
                        ->whereNull('fd.deleted_at')
                        ->whereNotNull('fd.fund_id');

                    // ---------- Previous balance (only if start_date given) ----------
                    $prevIncome  = 0.0;
                    $prevExpense = 0.0;

                    if ($start_date) {
                        $prevIncome = (clone $base)
                            ->where('fd.payment_type', 1)
                            ->where('fd.date', '<', $start_date)
                            ->sum('fd.amount');

                        $prevExpense = (clone $base)
                            ->where('fd.payment_type', 2)
                            ->where('fd.date', '<', $start_date)
                            ->sum('fd.amount');
                    }

                    $previous_balance = (float) $prevIncome - (float) $prevExpense;

                    // ---------- Period income ----------
                    $periodIncomeQ = (clone $base)->where('fd.payment_type', 1);
                    $periodIncome  = $periodIncomeQ
                        ->when($start_date && $end_date, fn($q) => $q->whereBetween('fd.date', [$start_date, $end_date]))
                        ->when($start_date && ! $end_date, fn($q) => $q->where('fd.date', '>=', $start_date))
                        ->when(! $start_date && $end_date, fn($q) => $q->where('fd.date', '<=', $end_date))
                        ->sum('fd.amount');

                    // ---------- Period expense ----------
                    $periodExpenseQ = (clone $base)->where('fd.payment_type', 2);
                    $periodExpense  = $periodExpenseQ
                        ->when($start_date && $end_date, fn($q) => $q->whereBetween('fd.date', [$start_date, $end_date]))
                        ->when($start_date && ! $end_date, fn($q) => $q->where('fd.date', '>=', $start_date))
                        ->when(! $start_date && $end_date, fn($q) => $q->where('fd.date', '<=', $end_date))
                        ->sum('fd.amount');

                    // ---------- Current balance ----------
                    $current_balance = $previous_balance + ((float) $periodIncome - (float) $periodExpense);

                    // ---------- Action buttons ----------
                    $actionHtml = '
        <div class="btn-group btn-group-sm" role="group">

            <a class="btn btn-outline-secondary" href="' . route('receipt.payment.audit', ['fundType' => $typeId, 'start_date' => $start_date, 'end_date' => $end_date]) . '"><i class="fa fa-search"></i> Audit</a>
        </div>
    ';

                    // ---------- Push row ----------
                    $rows[] = [
                        'date'             => ($request->start_date ?? '') . ' - ' . ($request->end_date ?? ''),
                        'fund_type'        => $typeLabel,
                        'previous_balance' => number_format((float) $previous_balance, 2, '.', ''),
                        'income'           => number_format((float) $periodIncome, 2, '.', ''),
                        'expense'          => number_format((float) $periodExpense, 2, '.', ''),
                        'current_balance'  => number_format((float) $current_balance, 2, '.', ''),
                        'action'           => $actionHtml,
                    ];
                }

                return DataTables::of(collect($rows))
                    ->rawColumns(['action'])
                    ->toJson();
            }

            $invoice = \App\Models\InvoiceDesign::first();
            return view('dashboard.funds.receipt_payment', compact('invoice'));

        } catch (\Throwable $e) {
            Log::error('ReceiptAndPayment error', ['msg' => $e->getMessage(), 'line' => $e->getLine()]);
            if ($request->ajax()) {
                return response()->json(['error' => true, 'data' => []], 500);
            }
            return back()->with('error', 'Something went wrong');
        }
    }

    // print audit
    public function Audit(Request $request, $fundType)
   {
    $fundtype   = $fundType;
    $start_date = $request->start_date;
    $end_date   = $request->end_date;
    $invoice    = \App\Models\InvoiceDesign::first();
    $fundLabel  = $fundType == 1 ? 'General' : ($fundType == 2 ? 'Guraba' : ($fundType == 3 ? 'Mahfil' : 'Other'));

    if (!$start_date && !$end_date) {
        $start_date = date('Y-m-d');
        $end_date   = date('Y-m-d');
    }

    $base = DB::table('fund_details as fd')
        ->join('funds as f', 'fd.fund_id', '=', 'f.id')
        ->where('f.fund_type', $fundType)
        ->whereNull('fd.deleted_at')
        ->whereNotNull('fd.fund_id');

    // ---------- Previous balance ----------
    $prevIncome = 0.0; $prevExpense = 0.0;
    if ($start_date) {
        $prevIncome = (clone $base)->where('fd.payment_type', 1)
            ->where('fd.date', '<', $start_date)->sum('fd.amount');
        $prevExpense = (clone $base)->where('fd.payment_type', 2)
            ->where('fd.date', '<', $start_date)->sum('fd.amount');
    }
    $previous_balance = (float)$prevIncome - (float)$prevExpense;

    // ---------- Period income ----------
    $periodIncome = (clone $base)->where('fd.payment_type', 1)
        ->when($start_date && $end_date, fn($q)=>$q->whereBetween('fd.date', [$start_date,$end_date]))
        ->when($start_date && !$end_date, fn($q)=>$q->where('fd.date','>=',$start_date))
        ->when(!$start_date && $end_date, fn($q)=>$q->where('fd.date','<=',$end_date))
        ->sum('fd.amount');

    // ---------- Period expense ----------
    $periodExpense = (clone $base)->where('fd.payment_type', 2)
        ->when($start_date && $end_date, fn($q)=>$q->whereBetween('fd.date', [$start_date,$end_date]))
        ->when($start_date && !$end_date, fn($q)=>$q->where('fd.date','>=',$start_date))
        ->when(!$start_date && $end_date, fn($q)=>$q->where('fd.date','<=',$end_date))
        ->sum('fd.amount');

    // ---------- Type totals ----------
    $incomeDetails = (clone $base)->where('fd.payment_type', 1)
        ->groupBy('fd.type')
        ->selectRaw('fd.type, SUM(fd.amount) as total_amount')
        ->pluck('total_amount','fd.type');

    $expenseDetails = (clone $base)->where('fd.payment_type', 2)
        ->groupBy('fd.type')
        ->selectRaw('fd.type, SUM(fd.amount) as total_amount')
        ->pluck('total_amount','fd.type');

    // ---------- Optional subtypes (fee_id) ----------
    // fee name দরকার হলে fees টেবিল join করো (name column ধরলাম 'name')
    $incomeSubRows = (clone $base)->where('fd.payment_type', 1)
        ->whereNotNull('fd.fee_id')
        ->leftJoin('student_fees as fe','fe.id','=','fd.fee_id')  // যদি fees টেবিল থাকে
        ->groupBy('fd.type','fd.fee_id','fe.name')
        ->selectRaw('fd.type, fd.fee_id, COALESCE(fe.name, CONCAT("Fee #",fd.fee_id)) as fee_name, SUM(fd.amount) as amount')
        ->get();

    $expenseSubRows = (clone $base)->where('fd.payment_type', 2)
        ->whereNotNull('fd.fee_id')
        ->leftJoin('student_fees as fe','fe.id','=','fd.fee_id')
        ->groupBy('fd.type','fd.fee_id','fe.name')
        ->selectRaw('fd.type, fd.fee_id, COALESCE(fe.name, CONCAT("Fee #",fd.fee_id)) as fee_name, SUM(fd.amount) as amount')
        ->get();

    // nested structure: [type => ['total' => x, 'subtypes' => [ ['fee_id'=>..,'name'=>..,'amount'=>..], ... ]]]
    $incomeTree = [];
    foreach ($incomeDetails as $t => $total) {
        $incomeTree[$t] = ['total' => (float)$total, 'subtypes' => []];
    }
    foreach ($incomeSubRows as $r) {
        $incomeTree[$r->type]['subtypes'][] = [
            'fee_id' => $r->fee_id,
            'name'   => $r->fee_name,
            'amount' => (float)$r->amount,
        ];
    }

    $expenseTree = [];
    foreach ($expenseDetails as $t => $total) {
        $expenseTree[$t] = ['total' => (float)$total, 'subtypes' => []];
    }
    foreach ($expenseSubRows as $r) {
        $expenseTree[$r->type]['subtypes'][] = [
            'fee_id' => $r->fee_id,
            'name'   => $r->fee_name,
            'amount' => (float)$r->amount,
        ];
    }

    // ---------- Current balance ----------
    $current_balance = $previous_balance + ((float)$periodIncome - (float)$periodExpense);
    $incomeWithPreviousBalance = $previous_balance + $periodIncome;
    $expenseWithCurrentBalance = $current_balance + $periodExpense;

    return view('dashboard.funds.audit', compact(
        'invoice','fundType','fundLabel',
        'incomeDetails','expenseDetails',      // আগের মতো type total
        'incomeTree','expenseTree',            // নতুন: subtype সহ
        'previous_balance','periodIncome','periodExpense','current_balance',
        'start_date','end_date',
        'incomeWithPreviousBalance','expenseWithCurrentBalance',
    ));
}


     private function buildAuditData(int $fundType, ?string $start_date, ?string $end_date): array
{
    $invoice    = \App\Models\InvoiceDesign::first();
    $fundLabel  = $fundType == 1 ? 'General' : ($fundType == 2 ? 'Guraba' : ($fundType == 3 ? 'Mahfil' : 'Other'));

    if (!$start_date && !$end_date) {
        $start_date = date('Y-m-d');
        $end_date   = date('Y-m-d');
    }

    $base = DB::table('fund_details as fd')
        ->join('funds as f', 'fd.fund_id', '=', 'f.id')
        ->where('f.fund_type', $fundType)
        ->whereNull('fd.deleted_at')
        ->whereNotNull('fd.fund_id');

    // ---------- Previous balance (upto day before start_date) ----------
    $prevIncome = (clone $base)
        ->where('fd.payment_type', 1)
        ->when($start_date, fn($q) => $q->where('fd.date', '<', $start_date))
        ->sum('fd.amount');

    $prevExpense = (clone $base)
        ->where('fd.payment_type', 2)
        ->when($start_date, fn($q) => $q->where('fd.date', '<', $start_date))
        ->sum('fd.amount');

    $previous_balance = (float)$prevIncome - (float)$prevExpense;

    // ---------- Period totals ----------
    $periodIncome = (clone $base)->where('fd.payment_type', 1)
        ->when($start_date && $end_date, fn($q)=>$q->whereBetween('fd.date', [$start_date, $end_date]))
        ->when($start_date && !$end_date, fn($q)=>$q->where('fd.date', '>=', $start_date))
        ->when(!$start_date && $end_date, fn($q)=>$q->where('fd.date', '<=', $end_date))
        ->sum('fd.amount');

    $periodExpense = (clone $base)->where('fd.payment_type', 2)
        ->when($start_date && $end_date, fn($q)=>$q->whereBetween('fd.date', [$start_date, $end_date]))
        ->when($start_date && !$end_date, fn($q)=>$q->where('fd.date', '>=', $start_date))
        ->when(!$start_date && $end_date, fn($q)=>$q->where('fd.date', '<=', $end_date))
        ->sum('fd.amount');

    // ---------- Type-wise details (WITH same date filters) ----------
    $incomeDetails = (clone $base)
        ->where('fd.payment_type', 1)
        ->when($start_date && $end_date, fn($q)=>$q->whereBetween('fd.date', [$start_date, $end_date]))
        ->when($start_date && !$end_date, fn($q)=>$q->where('fd.date', '>=', $start_date))
        ->when(!$start_date && $end_date, fn($q)=>$q->where('fd.date', '<=', $end_date))
        ->groupBy('fd.type')
        ->selectRaw('fd.type, SUM(fd.amount) as total_amount')
        ->pluck('total_amount', 'fd.type');

    $expenseDetails = (clone $base)
        ->where('fd.payment_type', 2)
        ->when($start_date && $end_date, fn($q)=>$q->whereBetween('fd.date', [$start_date, $end_date]))
        ->when($start_date && !$end_date, fn($q)=>$q->where('fd.date', '>=', $start_date))
        ->when(!$start_date && $end_date, fn($q)=>$q->where('fd.date', '<=', $end_date))
        ->groupBy('fd.type')
        ->selectRaw('fd.type, SUM(fd.amount) as total_amount')
        ->pluck('total_amount', 'fd.type');

    // ---------- Optional sub-rows (fee_id) WITH same date filters ----------
    // fees টেবিলে নাম থাকলে দেখাবে, না থাকলে "Fee #<id>"
    $incomeSubRows = (clone $base)
        ->where('fd.payment_type', 1)
        ->whereNotNull('fd.fee_id')
        ->when($start_date && $end_date, fn($q)=>$q->whereBetween('fd.date', [$start_date, $end_date]))
        ->when($start_date && !$end_date, fn($q)=>$q->where('fd.date', '>=', $start_date))
        ->when(!$start_date && $end_date, fn($q)=>$q->where('fd.date', '<=', $end_date))
        ->leftJoin('student_fees as fe', 'fe.id', '=', 'fd.fee_id') // থাকলে
        ->groupBy('fd.type', 'fd.fee_id', 'fe.name')
        ->selectRaw('fd.type, fd.fee_id, COALESCE(fe.name, CONCAT("Fee #", fd.fee_id)) as fee_name, SUM(fd.amount) as amount')
        ->get();

    $expenseSubRows = (clone $base)
        ->where('fd.payment_type', 2)
        ->whereNotNull('fd.fee_id')
        ->when($start_date && $end_date, fn($q)=>$q->whereBetween('fd.date', [$start_date, $end_date]))
        ->when($start_date && !$end_date, fn($q)=>$q->where('fd.date', '>=', $start_date))
        ->when(!$start_date && $end_date, fn($q)=>$q->where('fd.date', '<=', $end_date))
        ->leftJoin('student_fees as fe', 'fe.id', '=', 'fd.fee_id')
        ->groupBy('fd.type', 'fd.fee_id', 'fe.name')
        ->selectRaw('fd.type, fd.fee_id, COALESCE(fe.name, CONCAT("Fee #", fd.fee_id)) as fee_name, SUM(fd.amount) as amount')
        ->get();

    // ---------- Build nested trees: [type => ['total'=>x, 'subtypes'=>[...]]] ----------
    $incomeTree = [];
    foreach ($incomeDetails as $t => $total) {
        $incomeTree[$t] = ['total' => (float)$total, 'subtypes' => []];
    }
    foreach ($incomeSubRows as $r) {
        if (!isset($incomeTree[$r->type])) {
            $incomeTree[$r->type] = ['total' => 0.0, 'subtypes' => []];
        }
        $incomeTree[$r->type]['subtypes'][] = [
            'fee_id' => $r->fee_id,
            'name'   => $r->fee_name,
            'amount' => (float)$r->amount,
        ];
    }

    $expenseTree = [];
    foreach ($expenseDetails as $t => $total) {
        $expenseTree[$t] = ['total' => (float)$total, 'subtypes' => []];
    }
    foreach ($expenseSubRows as $r) {
        if (!isset($expenseTree[$r->type])) {
            $expenseTree[$r->type] = ['total' => 0.0, 'subtypes' => []];
        }
        $expenseTree[$r->type]['subtypes'][] = [
            'fee_id' => $r->fee_id,
            'name'   => $r->fee_name,
            'amount' => (float)$r->amount,
        ];
    }

    // ---------- Balances ----------
    $current_balance = $previous_balance + ((float)$periodIncome - (float)$periodExpense);
    $incomeWithPreviousBalance = $previous_balance + $periodIncome;
    $expenseWithCurrentBalance = $current_balance + $periodExpense;

    return [
        'invoice' => $invoice,
        'fundType' => $fundType,
        'fundLabel' => $fundLabel,

        // type totals
        'incomeDetails' => $incomeDetails,
        'expenseDetails' => $expenseDetails,

        // NEW: subtype trees
        'incomeTree' => $incomeTree,
        'expenseTree' => $expenseTree,

        'previous_balance' => $previous_balance,
        'periodIncome' => $periodIncome,
        'periodExpense' => $periodExpense,
        'current_balance' => $current_balance,
        'start_date' => $start_date,
        'end_date' => $end_date,
        'incomeWithPreviousBalance' => $incomeWithPreviousBalance,
        'expenseWithCurrentBalance' => $expenseWithCurrentBalance,
    ];
}


   public function auditAll(Request $request)
{
    $start_date = $request->start_date;
    $end_date   = $request->end_date;

    $fundTypes = [1 => 'General', 2 => 'Guraba', 3 => 'Mahfil', 4 => 'Other'];

    $pages = [];
    $summary = [];

    foreach ($fundTypes as $ft => $label) {
        $data = $this->buildAuditData($ft, $start_date, $end_date);
        $pages[] = $data;

        $summary[] = [
            'label'                        => $label,
            'previous_balance'             => $data['previous_balance'],
            'period_income'                => $data['periodIncome'],
            'period_expense'               => $data['periodExpense'],
            'current_balance'              => $data['current_balance'],
            'incomeWithPreviousBalance'    => $data['incomeWithPreviousBalance'],
            'expenseWithCurrentBalance'    => $data['expenseWithCurrentBalance'],
        ];
    }

    return view('dashboard.funds.auditAll', compact('pages', 'summary', 'start_date', 'end_date'));
}

}
