<?php

namespace App\Http\Controllers\BackEnd;

use Carbon\Carbon;
use App\Traits\SmsTrait;
use App\Traits\DeleteTrait;
use Illuminate\Http\Request;
use App\Models\BackEnd\GurabaFund;
use Illuminate\Support\Facades\DB;
use App\Models\BackEnd\GeneralFund;
use App\Models\BackEnd\Transaction;
use App\Http\Controllers\Controller;
use Yajra\DataTables\Facades\DataTables;

class GeneralExpenseController extends Controller
{
    use SmsTrait;
    use DeleteTrait;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {

        if ($request->ajax()) {

            $data = DB::table('general_funds')->where('type', 2)->whereNull('deleted_at')->orderBy('id','DESC');

            return Datatables::of($data)

                ->addColumn('action', function ($data) {

                        $result = '';

                        $details = '<li><a class="dropdown-item" href="' . route('general-expense.show', $data->id) . ' " ><i class="fa fa-eye text-primary" ></i> Details</a></li>';

                        $edit = '<li><a class="dropdown-item" href="' . route('general-expense.edit', $data->id) . ' "><i class="fa fa-edit text-success"></i> Edit</a></li>';

                        $delete = '<li><a class="dropdown-item btn-delete" href="#" data-remote=" ' . route('general-expense.destroy', $data->id) . ' "><i class="fa fa-trash text-red"></i> Delete</a></li>';

                        $result = $details. $edit. $delete;

                        return '<div class="btn-group open">
                        <a class="badge badge-primary dropdown-toggle" href="#" role="button"  data-toggle="dropdown">Actions<i class="ik ik-chevron-down mr-0 align-middle"></i></a>
                        <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">'.$result.'
                        </ul>
                    </div>';
                })

                ->addIndexColumn()
                ->rawColumns(['action'])
                ->toJson();
            }
         return view('backend.general_expense.index');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];
        $id = GeneralFund::count();
        $totalFund = GeneralFund::where('type', 1)->sum('amount');
        $totalExpense = GeneralFund::where('type', 2)->sum('amount');
        $sum = $totalFund - $totalExpense;
        $currentBalance = str_replace($en_numbers, $bn_numbers, $sum);
        return view('backend.general_expense.create', compact('id','currentBalance'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $messages = array(
            'receipt_no.required' => 'Enter receipt no',
            'name.required' => 'Enter name',
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
            'purpose.required' => 'Enter on account of',
        );

        $this->validate($request, array(
            'receipt_no' => 'required',
            'date' => 'required',
            'name' => 'required',
            'amount' => 'required',
            'purpose' => 'required',
        ), $messages);

        DB::beginTransaction();

        try {

            $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
            $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

            $data = new GeneralFund();
            $data->receipt_no         = $request->receipt_no;
            $data->date               = Carbon::parse($request->date)->format('Y-m-d');
            $data->name               = $request->name;
            $data->amount             = str_replace($bn_numbers, $en_numbers, $request->amount);
            $data->purpose            = $request->purpose;
            $data->type               = 2;
            $data->note               = $request->note;
            $data->save();

            $transaction = new Transaction();
            $transaction->general_expense_id    = $data->id;
            $transaction->date                  = Carbon::parse($request->date)->format('Y-m-d');
            $transaction->transaction_purpose   = $request->purpose;
            $transaction->transaction_type      = 2;
            $transaction->payment_type          = 7;
            $transaction->amount                = str_replace($bn_numbers, $en_numbers, $request->amount);
            $transaction->save();

            DB::commit();

            return redirect()->route('general-expense.index')
                ->with('success', 'General expense created successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $data = GurabaFund::findOrFail($id);
        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];
        $date = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->date)->format('d/m/Y'));
        return view('backend.general_expense.show', compact('data', 'date'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $data = GeneralFund::findOrFail($id);
        $date = Carbon::parse($data->date)->format('d-F-Y');
        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];
        $totalFund = GeneralFund::where('type', 1)->sum('amount');
        $totalExpense = GeneralFund::where('type', 2)->sum('amount');
        $sum = ($totalFund - $totalExpense) + $data->amount;
        $currentBalance = str_replace($en_numbers, $bn_numbers, $sum);
        $amount = str_replace($en_numbers, $bn_numbers, $data->amount);
        return view('backend.general_expense.edit', compact('data','date','currentBalance','amount'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $messages = array(
            'receipt_no.required' => 'Enter receipt no',
            'name.required' => 'Enter name',
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
            'purpose.required' => 'Enter on account of',
        );

        $this->validate($request, array(
            'receipt_no' => 'required',
            'date' => 'required',
            'name' => 'required',
            'amount' => 'required',
            'purpose' => 'required',
        ), $messages);

        DB::beginTransaction();

        try {
            $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
            $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

            $data = GeneralFund::findOrFail($id);
            $data->date               = Carbon::parse($request->date)->format('Y-m-d');
            $data->name               = $request->name;
            $data->amount             = str_replace($bn_numbers, $en_numbers, $request->amount);
            $data->purpose            = $request->purpose;
            $data->type               = 2;
            $data->note               = $request->note;
            $data->update();

            $transaction = Transaction::where('general_expense_id', $id)->first();
            $transaction->general_expense_id    = $data->id;
            $transaction->date                  = Carbon::parse($request->date)->format('Y-m-d');
            $transaction->transaction_purpose   = $request->purpose;
            $transaction->transaction_type      = 2;
            $transaction->payment_type          = 7;
            $transaction->amount                = str_replace($bn_numbers, $en_numbers, $request->amount);
            $transaction->update();

            DB::commit();

            return redirect()->route('general-expense.index')
                ->with('success', 'General expense updated successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try{
            $data = GeneralFund::findOrFail($id);
            if($data){
                $transaction = Transaction::where('general_expense_id', $id)->first();
                $transaction->delete();
                $data->delete();
            }
            return response()->json([
                'success' => true,
                'message' => 'General expense deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'General expense delete failed',
            ]);
        }
    }
    public function balanceSheet(Request $request)
    {
        try {

        if ($request->ajax()) {

            $data = DB::table('general_funds')->whereNull('deleted_at')->get();

            $totalCredit = DB::table('general_funds')->where('type', 1)->whereNull('deleted_at')->sum('amount');

            $totalDebit = DB::table('general_funds')->where('type', 2)->whereNull('deleted_at')->sum('amount');

            $currentBalance = $totalCredit - $totalDebit;

            return Datatables::of($data)

            ->addColumn('date', function ($data) {
                $date = Carbon::parse($data->date)->format('d F, Y');
                return $date;
            })

            ->addColumn('credit_amount', function ($data) {
                if ($data->type == 1 ) {
                    $total_amount = $data->amount;
                    return $total_amount;
                }else{
                    return '--';
                }

            })

            ->addColumn('debit_amount', function ($data) {
                if ($data->type == 2 ) {
                    $total_amount = $data->amount;
                    return $total_amount;
                }else{
                    return '--';
                }
            })

            ->addColumn('current_balance', function ($data) use (&$current_balance) {
                if ($data->type == 1) {
                    $current_balance = $current_balance + $data->amount;
                    return (number_format($current_balance));
                }

                if ($data->type == 2) {
                    $current_balance = $current_balance - $data->amount;
                    return (number_format($current_balance));
                }
            })

            ->with('totalCredit', $totalCredit)
            ->with('totalDebit', $totalDebit)
            ->with('currentBalance', $currentBalance)
            ->addIndexColumn()
            ->rawColumns(['date', 'debit_amount', 'credit_amount', 'current_balance'])
                ->toJson();
            }

         return view('backend.general_fund.balance');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
}

