<?php
namespace App\Http\Controllers;

use App\Models\Result;
use App\Models\Student;
use App\Models\ResultDetail;
use App\Models\SubjectGrade;
use Illuminate\Http\Request;
use App\Models\InvoiceDesign;
use App\Models\ExamWiseGrades;
use App\Helpers\StudentFinance;
use App\Models\SubjectWiseGrades;
use Illuminate\Support\Facades\DB;

class ParentsController extends Controller
{
    public function studentInfo()
    {
        $invoice = InvoiceDesign::first();
        return view('parents.student_info', compact('invoice'));
    }
    public function snapshotByReg(Request $request)
    {
        $reg = trim($request->query('reg', ''));

        if ($reg === '') {
            return response()->json(['error' => 'registration number required'], 422);
        }

        // Student basic
        $student = Student::with(['stuclass:id,name', 'stusession:id,name'])
            ->where('register_no', $reg)
            ->first();

        if (! $student) {
            return response()->json(['error' => 'student not found'], 404);
        }

        // -------- বেতন আদায় বাকি (fund_details ভিত্তিক) --------

          $monthly = StudentFinance::monthlyPaidDueSummary($student->id);

        // dd($monthlyPaidDueSummary);
        // ধরে নিচ্ছি fund_details টেবিলে: id, student_id, month, total, paid, status
        // due = total - paid (NULL-safe)

        // -------- পরীক্ষার রেজাল্ট (সারাংশ) --------
        $globalGrades = SubjectGrade::all();

        $examRows = Result::with(['exam:id,name', 'stusession:id,name', 'stuclass:id,name'])
            ->where('student_id', $student->id)
            ->orderByDesc('id')
            ->limit(100)
            ->get()
            ->map(function ($r) use ($globalGrades) {

                // এই Result-এর সব subject marks
                $details = ResultDetail::where('result_id', $r->id)
                    ->get(['subject_id', 'mark']);

                if ($details->isEmpty()) {
                    return [
                        'exam'       => $r->exam->name ?? '',
                        'session'    => $r->stusession->name ?? '',
                        'class'      => $r->stuclass->name ?? '',
                        'total_mark' => 0.00,
                        'total_exam' => 0,
                        'percent'    => number_format(0, 2),
                        'grade'      => 'অনুত্তীর্ণ',
                        'gpa'        => number_format(0, 2),
                    ];
                }

                $subjectIds = $details->pluck('subject_id')->unique()->values();

                // Preload: subject-wise grades (এই exam+class+session+subject সেটের জন্য)
                $subjectWiseMap = SubjectWiseGrades::where('exam_id', $r->exam_id)
                    ->where('class_id', $r->class_id)
                    ->where('session_id', $r->session_id)
                    ->whereIn('subject_id', $subjectIds)
                    ->whereNull('deleted_at')
                    ->get()
                    ->keyBy('subject_id'); // subject_id => row (exam_mark সহ)

                // Preload: exam-wise (single row)
                $examWise = ExamWiseGrades::where('exam_id', $r->exam_id)
                    ->where('session_id', $r->session_id)
                    ->whereNull('deleted_at')
                    ->first();

                $totalMark = 0.0;
                $totalExam = 0;

                foreach ($details as $d) {
                    $obtained = is_numeric($d->mark) ? (float) $d->mark : 0.0;

                    // per-subject exam_mark priority: subject-wise -> exam-wise -> 100
                    if ($subjectWiseMap->has($d->subject_id)) {
                        $exMark = (int) ($subjectWiseMap[$d->subject_id]->exam_mark ?? 100);
                    } elseif ($examWise) {
                        $exMark = (int) ($examWise->exam_mark ?? 100);
                    } else {
                        $exMark = 100;
                    }

                    $totalMark += $obtained;
                    $totalExam += $exMark;
                }

                $percent = $totalExam > 0 ? ($totalMark / $totalExam) * 100 : 0.0;

                // global grade band by percentage
                $band = $globalGrades->first(function ($g) use ($percent) {
                    $range = str_replace(' ', '', $g->mark);
                    if (strpos($range, '-') !== false) {
                        [$s, $e] = explode('-', $range, 2);
                        return $percent >= (float) $s && $percent <= (float) $e;
                    }
                    // single-threshold band like "80"
                    return $percent >= (float) $range;
                });

                return [
                    'exam'       => $r->exam->name ?? '',
                    'session'    => $r->stusession->name ?? '',
                    'class'      => $r->stuclass->name ?? '',
                    'total_mark' => round($totalMark, 2),
                    'total_exam' => (int) $totalExam,
                    'percent'    => number_format($percent, 2),
                    'grade'      => $band->name ?? 'অনুত্তীর্ণ',
                    'gpa'        => number_format((float) ($band->gpa ?? 0), 2),
                ];
            });

        // -------- হাজিরা (শেষ ৩০ দিন) --------
        $attendanceSummary = DB::table('attendances')
            ->where('student_id', $student->id)
            ->whereNull('deleted_at')
            ->selectRaw("DATE_FORMAT(date, '%Y-%m') as month")
            ->selectRaw("COUNT(*) as total_days")
            ->selectRaw("SUM(CASE WHEN LOWER(status) IN ('present','p','1') THEN 1 ELSE 0 END) as present_days")
            ->selectRaw("SUM(CASE WHEN LOWER(status) IN ('absent','a','0') THEN 1 ELSE 0 END) as absent_days")
            ->groupBy(DB::raw("DATE_FORMAT(date, '%Y-%m')"))
            ->orderBy('month', 'asc')
            ->get()
            ->map(function ($row) {
                $row->present_pct = $row->total_days > 0 ? round(($row->present_days / $row->total_days) * 100, 2) : 0;
                $row->absent_pct  = $row->total_days > 0 ? round(($row->absent_days / $row->total_days) * 100, 2) : 0;
                return $row;
            });

            // dd($attendanceSummary);

        // Summary card
        $summary = [
            'name'        => $student->name,
            'register_no' => $student->register_no,
            'roll_no'     => $student->roll_no,
            'class'       => $student->stuclass->name ?? '',
            'session'     => $student->stusession->name ?? '',
            'image'       => $student->image ? asset('backend/img/student/' . $student->image) : null,
        ];

        return response()->json([
            'student'    => $summary,
            'dues'       => $monthly,
            'results'    => $examRows,
            'attendance' => $attendanceSummary,
        ]);
    }

}
