<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\BackEnd\StaffHistory;
use App\Models\Fund;
use App\Models\FundDetail;
use App\Models\InvoiceDesign;
use App\Models\SmsStatus;
use App\Models\Staff;
use App\Traits\DateFormatter;
use App\Traits\DeleteTrait;
use App\Traits\SmsTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class StaffController extends Controller
{
    use SmsTrait;
       use DeleteTrait;
    use DateFormatter;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = DB::table('staff')->whereNull('deleted_at')->orderByDesc('id')->get();

                return Datatables::of($data)

                    ->addColumn('status', function ($data) {
                        $button = '<label class="switch">';
                        $button .= ' <input type="checkbox" class="changeStatus" id="customSwitch' . $data->id . '" getId="' . $data->id . '" name="status"';

                        if ($data->status == 1) {
                            $button .= "checked";
                        }
                        $button .= ' ><span class="slider round"></span>';
                        $button .= '</label>';
                        return $button;
                    })

                    ->addColumn('action', function ($data) {

                        $result = '';

                        $profile = '<li><a class="dropdown-item" href="' . route('staff.show', $data->id) . ' " ><i class="fa fa-user" ></i> Profile</a></li>';

                        $report = ' <li><a class="dropdown-item" href="' . route('staff.show', $data->id) . ' " ><i class="fa fa-file" ></i> Report</a></li>';

                        $details = '<li><a class="dropdown-item" href="' . route('staff.show', $data->id) . ' " ><i class="fa fa-eye" ></i> Details</a></li>';

                        $edit = '<li><a class="dropdown-item" href="' . route('staff.edit', $data->id) . ' "><i class="fa fa-edit"></i> Edit</a></li>';

                        $delete = '<li><a class="dropdown-item btn-delete" href="#" data-remote=" ' . route('staff.destroy', $data->id) . ' "><i class="fa fa-trash"></i> Delete</a></li>';

                        $result = $profile . $report . $details . $edit . $delete;

                        return '<div class="btn-group open">
                        <a class="badge badge-primary dropdown-toggle" href="#" role="button"  data-toggle="dropdown">Actions<i class="ik ik-chevron-down mr-0 align-middle"></i></a>
                        <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">' . $result . '
                        </ul>
                    </div>';

                    })

                    ->addIndexColumn()
                    ->rawColumns(['status', 'action'])
                    ->toJson();
            }
            return view('dashboard.staffs.staff.index');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $staff = Staff::count();
        return view('dashboard.staffs.staff.create', compact('staff'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        // Debugging the input data
        // dd($request->all());

        DB::beginTransaction();

        try {

            // Staff creation
            $data = new Staff();

            // Handle file upload for card_photo
            // $card_photo = [];
            // if($request->hasFile('card_photo')) {
            //     foreach($request->file('card_photo') as $image) {
            //         $filename = time() . $image->getClientOriginalName();
            //         $image->move(public_path('/backend/img/staff/'), $filename);
            //         $card_photo[] = $filename;
            //     }
            // }

            // Handle image upload for staff photo
            if ($request->hasFile('image')) {
                $file     = $request->file('image');
                $filename = time() . $file->getClientOriginalName();
                $file->move(public_path('/backend/img/staff/'), $filename);
                $data->image = $filename;
            }

            // Set staff properties, handling null values for optional fields

            $data->staff_user_id = $request->staff_user_id ?? ''; // Use empty string if null
            $data->name          = $request->name;
            $data->join_date     = $request->join_date;
            $data->gender        = $request->gender;
            $data->designation   = $request->designation;
            $data->salary        = $request->salary;
            $data->phone         = $request->phone;
            $data->email         = $request->email;
            $data->blood_group   = strtoupper($request->blood_group ?? ''); // Default to empty string if null
            $data->status        = $request->status;
            $data->address       = $request->address ?? '';     // Default to empty string if null
            $data->description   = $request->description ?? ''; // Default to empty string if null
                                                                // $data->card_name     = json_encode($request->card_name ?? []);  // Default to empty array if null
                                                                // $data->card_no       = json_encode($request->card_no ?? []);  // Default to empty array if null
                                                                // $data->card_photo    = json_encode($card_photo);
            $data->created_by = 1;                              // Use a static value or get from auth
            $data->save();

            // dd($data);

            DB::commit();

            return redirect()->route('staff.index')->with('success', 'Staff created successfully');
        } catch (\Exception $exception) {
            dd($exception);
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $data = Staff::findOrFail($id);
        return view('dashboard.staffs.staff.show', compact('data'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $staff = Staff::findOrFail($id);
        return view('dashboard.staffs.staff.edit', compact('staff'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        // Debugging the input data
        // dd($request->all());

        DB::beginTransaction();

        try {
            // Check if it's an update or create operation
            if ($id) {
                // Update existing staff
                $data = Staff::findOrFail($id);
            } else {
                // Create new staff
                $data = new Staff();
            }

            // Handle image upload for staff photo (update or create)
            if ($request->hasFile('image')) {
                // Remove old image if updating
                if (isset($data->image) && file_exists(public_path('/backend/img/staff/') . $data->image)) {
                    unlink(public_path('/backend/img/staff/') . $data->image);
                }
                $file     = $request->file('image');
                $filename = time() . $file->getClientOriginalName();
                $file->move(public_path('/backend/img/staff/'), $filename);
                $data->image = $filename;
            }

                                                                  // Set staff properties, handling null values for optional fields
            $data->staff_user_id = $request->staff_user_id ?? ''; // Use empty string if null
            $data->name          = $request->name;
            $data->join_date     = $request->join_date;
            $data->gender        = $request->gender;
            $data->designation   = $request->designation;
            $data->salary        = $request->salary;
            $data->phone         = $request->phone;
            $data->email         = $request->email;
            $data->blood_group   = strtoupper($request->blood_group ?? ''); // Default to empty string if null
            $data->status        = $request->status;
            $data->address       = $request->address ?? '';     // Default to empty string if null
            $data->description   = $request->description ?? ''; // Default to empty string if null
            $data->created_by    = 1;                           // Use a static value or get from auth
            $data->save();

            DB::commit();

            return redirect()->route('staff.index')->with('success', 'Staff ' . ($id ? 'updated' : 'created') . ' successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $data = Staff::findOrFail($id);
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Staff deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Staff deleted failed',
            ]);
        }
    }

    public function StatusChange(Request $request)
    {
        $data         = Staff::findOrFail($request->id);
        $data->status = $data->status == 1 ? 0 : 1;
        $data->update();

        if ($data->status == 1) {
            return response()->json([
                'success' => true,
                'message' => 'Status activated successfully',
            ]);
        } else {
            return response()->json([
                'success' => false,
                'message' => 'Status inactivated successfully',
            ]);
        }
    }

    public function salary($id)
    {
        $data        = Staff::findOrFail($id);
        $banks       = BankAccount::with('banks')->where('status', 1)->get();
        $mobilebanks = MobileBankingAccount::with('mobileBankings')->where('status', 1)->get();
        return view('backend.staffs.salary_pay.create', compact('data', 'banks', 'mobilebanks'));
    }

    public function salaryIncrement($id)
    {
        $data        = Staff::findOrFail($id);
        $banks       = BankAccount::with('banks')->where('status', 1)->get();
        $mobilebanks = MobileBankingAccount::with('mobileBankings')->where('status', 1)->get();
        return view('backend.staffs.salary_increment.create', compact('data', 'banks', 'mobilebanks'));
    }

    public function StaffHistory(Request $request, $id)
    {
        try {
            if ($request->ajax()) {

                $data = StaffHistory::with('staffs')->where('staff_id', $id)->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        $date = date('F d, Y', strtotime($data->date));
                        return $date;
                    })

                    ->addColumn('previousSalary', function ($data) {
                        $salary = $data->salary - $data->increment_amount;
                        return number_format($salary, 2);
                    })

                    ->addColumn('incrementAmount', function ($data) {
                        $salary = $data->increment_amount;
                        return number_format($salary, 2);
                    })

                    ->addColumn('currentSalary', function ($data) {
                        $salary = $data->salary;
                        return number_format($salary, 2);
                    })

                    ->addColumn('incrementReason', function ($data) {
                        $value = $data->increment_reason ?? '--';
                        return $value;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['date', 'previousSalary', 'incrementAmount', 'currentSalary', 'incrementReason'])
                    ->toJson();
            }
            return view('backend.staffs.staff.history');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

//type 20 for staff salary
    public function StaffSalary(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = FundDetail::with('staff:id,name,salary') // Ensure that 'staff' relationship is loaded properly
                    ->where('type', 20)
                    ->selectRaw('month, staff_id, SUM(amount) as total_amount')
                    ->groupBy('month', 'staff_id')
                    ->get();

                return Datatables::of($data)
                    ->addColumn('date', function ($data) {
                        return \Carbon\Carbon::createFromFormat('Y-m', trim($data->month))->format('F Y');
                    })
                    ->addColumn('staff', function ($data) {
                        return isset($data->staff) ? $data->staff->name : '--';
                    })
                    ->addColumn('salary', function ($data) {
                        return isset($data->staff) ? $data->staff->salary : '--';
                    })
                    ->addColumn('paid', function ($data) {
                        return $data->total_amount ?? '--';
                    })
                    ->addColumn('due', function ($data) {
                        $salary = isset($data->staff) ? $data->staff->salary : 0;
                        $due    = $salary - $data->total_amount;
                        return $due;
                    })
                    ->addColumn('action', function ($data) {
                        return '<a type="button" class="btn btn-sm btn-primary print mr-1" href="' . route('staff.salary.details', ['month' => $data->month, 'staff_id' => $data->staff_id]) . '" title="Print"><i class="fa fa-eye"></i></a>';
                    })
                    ->addIndexColumn()
                    ->rawColumns(['date', 'paid', 'staff', 'salary', 'action'])
                    ->toJson();
            }
            return view('dashboard.staffs.salary_pay.index');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StaffSalaryCreate()
    {
        $staffs = Staff::get();
        $funds  = Fund::all();

        return view('dashboard.staffs.salary_pay.create', compact('staffs', 'funds'));
    }

    public function StaffSalaryFind(Request $request)
    {
        $salary = Staff::findOrFail($request->staff_id);
        // $monthString = $request->month;
        // $carbonDate = Carbon::createFromFormat('F-Y', $monthString);
        $month = $request->month;

        $total = FundDetail::where('staff_id', $request->staff_id)
            ->where('month', $month)
            ->where('type', 20) //staff type 20 for salary
            ->where('payment_type', 2)
            ->select(
                DB::raw('SUM(amount) as total_paid'),
            )
            ->first();

        return response()->json([
            'salary'     => $salary->salary,
            'total_paid' => $total->total_paid ?? 0,
        ]);
    }

    public function StaffSalaryStore(Request $request)
    {
        $messages = [
            'staff_id.required' => 'Select Staff',
        ];

        $request->validate([
            'staff_id' => 'required',
        ], $messages);

        try {

            $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                ->orderByDesc('receipt_no')
                ->first();
            $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;

            $totalReceived = FundDetail::where('staff_id', $request->staff_id)
                ->where('type', 20) //staff type 20 for salary
                ->where('month', Carbon::parse($request->month)->format('Y-m'))
                ->sum('amount');

            $totalAmount   = (float) $request->total_amount;
            $totalReceived = (float) $totalReceived + (float) $request->amount;

            if ($totalReceived > $totalAmount) {
                return 'Salary already paid for ' . Carbon::parse($request->month)->format('F Y');
            }

            $data               = new FundDetail();
            $data->fund_id      = $request->fund_id;
            $data->staff_id     = $request->staff_id;
            $data->receipt_no   = $receipt_no;
            $data->purpose      = 'Staff salary pay';
            $data->date         = now()->format('Y-m-d');
            $data->type         = 20;
            $data->payment_type = 2;
            $data->fund_type    = 1;
            $data->month        = Carbon::parse($request->month)->format('Y-m');
            $data->amount       = $request->amount;
            $data->save();

            $staff = Staff::findOrFail($request->staff_id);
            $site  = DB::table('settings')->first();

            $to   = $staff->phone;
            $text = $staff->name . ', ' . $data->month . ', Salary Paid : ' . $data->amount . ' TK';

            $status = optional(SmsStatus::first())->staff_salary;
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            return redirect()->route('staff.salary.print', ['id' => $data->id])
                ->with('success', 'Salary paid successfully and ready for print.');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

        public function StaffSalaryEdit($id)
    {
        $data = FundDetail::with('staff')->findOrFail($id);
        $staffs = Staff::get();
        $funds = Fund::all();
        return view('dashboard.staffs.salary_pay.edit', compact('data','staffs','funds'));
    }


      public function StaffSalaryUpdate(Request $request, $id)
    {
        $messages = [
            'staff_id.required' => 'Select Staff',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'staff_id' => 'required',
            'amount' => 'required',
        ], $messages);

        try {

            $totalReceived = FundDetail::where('staff_id', $request->staff_id)
                ->where('type', 20)
                ->where('month', Carbon::parse($request->month)->format('Y-m'))
                ->sum('amount');

            $totalAmount = (float)$request->total_amount;
            $totalReceived = (float)$totalReceived + (float)$request->amount;

            if ($totalReceived > $totalAmount) {
                return 'Salary already paid for ' . Carbon::parse($request->month)->format('F Y');
            }

            $data = FundDetail::findOrFail($id);
            $data->fund_id        = $request->fund_id;
            $data->staff_id     = $request->staff_id;
            $data->purpose        = 'Staff salary pay';
            $data->date           = $data->date;
            $data->type           = 20;  //staff type 20 for salary
            $data->payment_type   = 2;
            $data->fund_type      = 1;
            $data->month          = Carbon::parse($request->month)->format('Y-m');
            $data->amount         = $request->amount;
            $data->save();

            $staff = Staff::findOrFail($request->staff_id);
            $site = DB::table('settings')->first();

            $to = $staff->phone;
            $text = $staff->name . ', ' .  $data->month . ', Salary Paid : '. $data->amount . ' TK' ;

            $status = optional(SmsStatus::first())->staff_salary;
            if($status == 1){
                $this->SendSms($to, $text);
            }

            return redirect()->route('staff.salary.print', ['id' => $data->id])
                         ->with('success', 'Salary updated successfully and ready for print.');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }


    public function StaffSalaryPrint($id)
    {
        try {
            $data      = FundDetail::with('staff:id,name,phone,salary')->findOrFail($id);
            $totalPaid = FundDetail::where('staff_id', $data->staff_id)->where('type', 20) //staff type 20 for salary
                ->where('month', $data->month)
                ->sum('amount');
            $design = InvoiceDesign::first();
            return view('dashboard.staffs.salary_pay.salary_print', compact('data', 'design', 'totalPaid'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('success', $exception);
        }
    }

    public function StaffSalaryDetails(Request $request, $month, $staff_id)
    {
        try {
            if ($request->ajax()) {

                $data = FundDetail::where('month', $month)
                    ->where('staff_id', $staff_id)
                    ->where('type', 20) // type 20 for staff salary
                    ->get();

                if ($data->isEmpty()) {
                    return response()->json([
                        'error' => 'No payments found for this student and month.',
                    ]);
                }

                $lastRecord = FundDetail::where('month', $month)
                    ->where('staff_id', $staff_id)
                    ->where('type', 20) //type 20 for staff salary
                    ->orderBy('id', 'desc')
                    ->first();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d/m/Y');
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? '';
                    })

                    ->addColumn('action', function ($data) use ($lastRecord) {

                        $edit = '';

                        if ($data->id == $lastRecord->id) {
                            $edit = '<a id="edit" href="' . route('staff.salary.edit', $data->id) . '" class="btn btn-sm btn-info edit" title="Edit"><i class="fa fa-edit"></i></a> ';
                        }

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('staff.salary.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }

                        $print = '<a type="button" id="print" class="btn btn-sm btn-primary print mr-1" href="' . route('staff.salary.print', $data->id) . ' " title="a6 page"><i class="fa fa-file"></i></a>';

                        return $edit . $print . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action', 'date', 'amount', 'discount'])
                    ->toJson();
            }

            $staff = Staff::find($staff_id);
            if (! $staff) {
                return redirect()->back()->with('error', 'Staff not found.');
            }

            return view('dashboard.staffs.salary_pay.details', compact('staff', 'month', 'staff', 'staff_id'));

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

}
