<?php

namespace App\Helpers;

use App\Models\Student;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class StudentFinance
{
    /**
     * ভর্তি-মাস (join_date) এবং প্রথম assign-মাস দেখে effective start month বের করে
     * return: 'YYYY-MM' | null
     */
    public static function effectiveStartMonth(Student $student): ?string
    {
        $joinMonth = $student->join_date ? Carbon::parse($student->join_date)->format('Y-m') : null;

        // প্রথম class/session ভিত্তিক (type != 5) assign মাস
        $firstOther = DB::table('fee_assign_details as fad')
            ->whereNull('fad.deleted_at')
            ->where('fad.class_id',  $student->class_id)
            ->where('fad.session_id',$student->session_id)
            ->where('fad.type', '!=', 5)
            ->min('fad.month'); // 'YYYY-MM' বা DATE

        // প্রথম per-student special (type = 5) assign মাস
        $firstSpecial = DB::table('fee_assign_details as fad')
            ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fad.id')
            ->whereNull('fad.deleted_at')
            ->whereNull('fds.deleted_at')
            ->where('fad.class_id',  $student->class_id)
            ->where('fad.session_id',$student->session_id)
            ->where('fad.type', 5)
            ->where('fds.student_id', $student->id)
            ->min('fad.month');

        $firstAssignedMonth = $firstOther && $firstSpecial
            ? (Carbon::parse($firstOther) < Carbon::parse($firstSpecial) ? $firstOther : $firstSpecial)
            : ($firstOther ?: $firstSpecial);

        // কার্যকর শুরু-মাস = max(joinMonth, firstAssignedMonth)
        if ($joinMonth && $firstAssignedMonth) {
            return Carbon::parse($joinMonth) > Carbon::parse($firstAssignedMonth)
                ? $joinMonth
                : $firstAssignedMonth;
        }
        return $joinMonth ?: $firstAssignedMonth; // যেটা আছে সেটাই
    }

    /**
     * মাসভিত্তিক summary: total_amount, total_paid, total_discount, total_due, status
     * optional: $fromMonth, $toMonth ('YYYY-MM') রেঞ্জ দিতে পারো
     */
    public static function monthlyPaidDueSummary(int $studentId, ?string $fromMonth = null, ?string $toMonth = null): array
    {
        $student = Student::findOrFail($studentId);

        // effective start month যদি না দাও, Helper নিজে বের করবে
        $effectiveStart = $fromMonth ?: self::effectiveStartMonth($student);

        // ---------- 1) other (type != 5) ----------
        $other = DB::table('fee_assign_details as fad')
            ->whereNull('fad.deleted_at')
            ->where('fad.class_id',  $student->class_id)
            ->where('fad.session_id',$student->session_id)
            ->where('fad.type', '!=', 5)
            ->when($effectiveStart, fn($q) => $q->where('fad.month', '>=', $effectiveStart))
            ->when($toMonth, fn($q) => $q->where('fad.month', '<=', $toMonth))
            ->select('fad.month', DB::raw('SUM(COALESCE(fad.amount,0)) as total_amount'))
            ->groupBy('fad.month')
            ->get();

        // ---------- 2) special (type = 5) ----------
        $special = DB::table('fee_assign_details as fad')
            ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fad.id')
            ->whereNull('fad.deleted_at')->whereNull('fds.deleted_at')
            ->where('fad.class_id',  $student->class_id)
            ->where('fad.session_id',$student->session_id)
            ->where('fad.type', 5)
            ->where('fds.student_id', $student->id)
            ->when($effectiveStart, fn($q) => $q->where('fad.month', '>=', $effectiveStart))
            ->when($toMonth, fn($q) => $q->where('fad.month', '<=', $toMonth))
            ->select('fad.month', DB::raw('SUM(COALESCE(fds.amount,0)) as total_amount'))
            ->groupBy('fad.month')
            ->get();

        // ---------- 3) received + discount ----------
        $received = DB::table('fund_details')
            ->whereNull('deleted_at')
            ->where('student_id', $student->id)
            ->when($student->etim == 1, fn($q) => $q->where('type', 16), fn($q) => $q->where('type', 6))
            ->when($effectiveStart, fn($q) => $q->where('month', '>=', $effectiveStart))
            ->when($toMonth, fn($q) => $q->where('month', '<=', $toMonth))
            ->select(
                'month',
                DB::raw('SUM(COALESCE(amount,0))   as total_paid'),
                DB::raw('SUM(COALESCE(discount,0)) as total_discount')
            )
            ->groupBy('month')
            ->get();

        // ---------- Merge ----------
        $byMonth = [];

        $touch = function (&$arr, $month) {
            if (!isset($arr[$month])) {
                $arr[$month] = [
                    'month'          => $month,
                    'total_amount'   => 0.0,
                    'total_paid'     => 0.0,
                    'total_discount' => 0.0,
                    'total_due'      => 0.0,
                    'status'         => 'N/A',
                ];
            }
        };

        foreach ($other as $r) {
            $m = $r->month;
            $touch($byMonth, $m);
            $byMonth[$m]['total_amount'] += (float) $r->total_amount;
        }

        foreach ($special as $r) {
            $m = $r->month;
            $touch($byMonth, $m);
            $byMonth[$m]['total_amount'] += (float) $r->total_amount;
        }

        foreach ($received as $r) {
            $m = $r->month;
            $touch($byMonth, $m);
            $byMonth[$m]['total_paid']     += (float) $r->total_paid;
            $byMonth[$m]['total_discount'] += (float) $r->total_discount;
        }

        // compute due + status
        foreach ($byMonth as $m => &$v) {
            $v['total_due'] = max(0, $v['total_amount'] - ($v['total_paid'] + $v['total_discount']));
            if ($v['total_amount'] <= 0 && $v['total_paid'] <= 0 && $v['total_discount'] <= 0) {
                $v['status'] = 'N/A';
            } elseif ($v['total_due'] == 0) {
                $v['status'] = 'Paid';
            } elseif ($v['total_paid'] > 0) {
                $v['status'] = 'Partially Paid';
            } else {
                $v['status'] = 'Due';
            }
        }
        unset($v);

        ksort($byMonth); // YYYY-MM asc
        return array_values($byMonth);
    }

    /**
     * সামগ্রিক টোটাল (রেঞ্জ সহ): total_amount, total_paid, total_discount, total_due
     */
    public static function overallTotals(int $studentId, ?string $fromMonth = null, ?string $toMonth = null): array
    {
        $student = Student::findOrFail($studentId);

        $monthly = self::monthlyPaidDueSummary($studentId, $fromMonth, $toMonth);

        $totals = [
            'total_amount'   => 0.0,
            'total_paid'     => 0.0,
            'total_discount' => 0.0,
            'total_due'      => 0.0,
        ];

        foreach ($monthly as $m) {
            $totals['total_amount']   += (float) $m['total_amount'];
            $totals['total_paid']     += (float) $m['total_paid'];
            $totals['total_discount'] += (float) $m['total_discount'];
            $totals['total_due']      += (float) $m['total_due'];
        }

        // ইচ্ছে করলে round/format করতে পারো
        foreach ($totals as $k => $v) {
            $totals[$k] = round($v, 2);
        }

        return $totals;
    }
}
