<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Fund;
use App\Models\Loan;
use App\Models\Shop;
use App\Models\User;
use App\Models\Donar;
use App\Models\Setting;
use App\Models\Student;
use App\Models\Teacher;
use App\Traits\SmsTrait;
use App\Models\FeeAssign;
use App\Models\HostelDue;
use App\Models\SmsStatus;
use App\Models\FundDetail;
use App\Models\Properties;
use App\Models\RoshidStock;
use App\Traits\DeleteTrait;
use Illuminate\Support\Str;
use Illuminate\Http\Request;
use App\Models\InvoiceDesign;
use App\Traits\DateFormatter;
use App\Models\StudentSession;
use App\Models\StudentSponsor;
use App\Models\FeeAssignDetail;
use App\Models\RoshidDistribute;
use App\Models\StudentFeeDetail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Session;
use Yajra\DataTables\Facades\DataTables;
use Illuminate\Support\Facades\Validator;

class AdministrationController extends Controller
{

    use SmsTrait;
    use DeleteTrait;
    use DateFormatter;

    public function RoshidStock(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = RoshidStock::select(
                    'name',
                    'slug',
                    DB::raw('MIN(CAST(serial_no AS UNSIGNED)) as first_serial'),
                    DB::raw('MAX(CAST(serial_no AS UNSIGNED)) as last_serial'),
                    DB::raw('COUNT(serial_no) as total_books')
                )
                ->whereNull('deleted_at')
                ->groupBy('slug')
                ->orderBy('slug', 'desc')
                ->get();

                return DataTables::of($data)
                    ->addColumn('name', function ($data) {
                        return $data->name ?? '';
                    })
                    ->addColumn('first_serial', function ($data) {
                        return $data->first_serial;
                    })
                    ->addColumn('last_serial', function ($data) {
                        return $data->last_serial;
                    })
                    ->addColumn('total_books', function ($data) {
                        return $data->total_books;
                    })
                    ->addColumn('action', function ($data) {
                        $edit = '<a id="edit" href="' . route('roshid.stock.edit', ['slug' => $data->slug]) . '" class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editClass"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="show" href="' . route('roshid.stock.show', ['slug' => $data->slug]) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('roshid.stock.destroy', $data->slug) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return $edit . $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name', 'first_serial', 'last_serial', 'total_books', 'action'])
                    ->toJson();
            }

            return view('dashboard.roshid.roshid_stock');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function RoshidStockStore(Request $request)
    {
        if ($request->ajax()) {
            try {
                $request->validate([
                    'name' => 'required|string|max:255',
                ]);

                $firstSerial = (int)$request->first_serial;
                $lastSerial = (int)$request->last_serial;
                if ($firstSerial > $lastSerial) {
                    return response()->json([
                        'success' => false,
                        'message' => 'First serial must be less than or equal to last serial.',
                    ]);
                }

                $id = mt_rand(1000, 9999);
                $slug = Str::slug($request->name) . '-' . $id;

                $dataToInsert = [];
                for ($serial = $firstSerial; $serial <= $lastSerial; $serial++) {
                    $dataToInsert[] = [
                        'name' => $request->name,
                        'slug' => $slug,
                        'serial_no' => $serial,
                        'status' => 2,
                        'created_by' => Auth::user()->id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }

                RoshidStock::insert($dataToInsert);
                return response()->json([
                    'success' => true,
                    'message' => 'Roshid stock created successfully',
                ]);

            } catch (\Exception $exception) {
                return response()->json([
                    'success' => false,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }


    public function RoshidStockShow($slug)
    {
        $items = RoshidStock::where('slug', $slug)->get();
        $name = RoshidStock::where('slug', $slug)->first();
        return view('dashboard.roshid.roshid_show', compact('items','name'));
    }

    public function RoshidStockEdit($slug)
    {
        $data = DB::table('roshid_stocks')
            ->where('slug', $slug)
            ->first();

        $first_serial = DB::table('roshid_stocks')
            ->where('slug', $data->slug)
            ->orderByRaw('CAST(serial_no AS UNSIGNED) ASC')
            ->value('serial_no');

        $last_serial = DB::table('roshid_stocks')
            ->where('slug', $data->slug)
            ->orderByRaw('CAST(serial_no AS UNSIGNED) DESC')
            ->value('serial_no');

        return response()->json([
            'name' => $data->name,
            'first_serial' => $first_serial,
            'last_serial' => $last_serial,
        ]);
    }

    public function RoshidStockUpdate(Request $request, $slug)
    {
        if ($request->ajax()) {
            try {
                $request->validate([
                    'name' => 'required|string|max:255',
                    'first_serial' => 'required|integer',
                    'last_serial' => 'required|integer|gte:first_serial',
                ]);

                $firstSerial = (int) $request->first_serial;
                $lastSerial = (int) $request->last_serial;
                RoshidStock::where('slug', $slug)->delete();
                $id = mt_rand(1000, 9999);
                $slug = Str::slug($request->name) . '-' . $id;
                $dataToUpdate = [];
                for ($serial = $firstSerial; $serial <= $lastSerial; $serial++) {
                    $dataToUpdate[] = [
                        'name' => $request->name,
                        'serial_no' => $serial,
                        'slug' => $slug,
                        'status' => 2,
                        'created_by' => Auth::user()->id,
                        'created_at' => now(),
                        'updated_at' => now(),
                    ];
                }

                RoshidStock::insert($dataToUpdate);

                return response()->json([
                    'success' => true,
                    'message' => 'Roshid stock updated successfully',
                ]);
            } catch (\Exception $exception) {
                return response()->json([
                    'success' => false,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }

    public function RoshidStockDestroy($slug)
    {
        try {
            $deletedCount = RoshidStock::where('slug', $slug)->delete();

            if ($deletedCount > 0) {
                return response()->json([
                    'success' => true,
                    'message' => 'All records of "' . $slug . '" deleted successfully.',
                ]);
            } else {
                return response()->json([
                    'success' => false,
                    'message' => 'No records found for "' . $slug . '".',
                ]);
            }
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Failed to delete records: ' . $exception->getMessage(),
            ]);
        }
    }
    public function RoshidDistribute(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = RoshidDistribute::with('roshid:id,name,serial_no')->where('type', 1)->get();

                return DataTables::of($data)

                    ->addColumn('roshidName', function ($data) {
                        return $data->roshid->name ?? '';
                    })

                    ->addColumn('name', function ($data) {
                        $name = $data->name ?? '';
                        $phone = $data->phone ?? '';
                        return $name . '<br>' . $phone;
                    })

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('serial', function ($data) {
                        return $data->roshid->serial_no ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $edit = '<a id="edit" href="' . route('roshid.distribute.edit', $data->id) . '" class="btn btn-sm btn-primary edit" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="show" href="' . route('roshid.distribute.show', $data->id) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('roshid.distribute.destroy', $data->id) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        return $edit . $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name', 'roshidName', 'date', 'serial', 'action'])
                    ->toJson();
            }

            return view('dashboard.roshid.roshid_distribute');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function RoshidDistributeStore(Request $request)
    {
        $messages = [
            'date.required' => 'Enter date',
            'roshid_stock_id.required' => 'Select roshid',
            'name.required' => 'Enter name',
        ];

        $request->validate([
            'date' => 'required',
            'roshid_stock_id' => 'required',
            'name' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {
            $data = new RoshidDistribute();
            $data->date = $this->formatAnyDateToYmd($request->date);;
            $data->roshid_stock_id = $request->roshid_stock_id;
            $data->name = $request->name;
            $data->phone = $request->phone;
            $data->page = $request->page;
            $data->note = $request->note;
            $data->type = 1;
            $data->created_by = Auth::user()->id;
            $data->save();

            $data->roshid_taken_by = $data->id;
            $data->save();

            $roshid = RoshidStock::findOrFail($request->roshid_stock_id);
            $roshid->status = 1;
            $roshid->roshid_taken_by = $data->id;
            $roshid->save();

            DB::commit();

            return redirect()->route('roshid.distribute')->with('success', 'Roshid distributed successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function RoshidDistributeCreate()
    {
        $items = RoshidStock::with('takenby')->where('status', 2)->select('name','slug','id','serial_no','roshid_taken_by')->get();
        return view('dashboard.roshid.roshid_distribute_create', compact('items'));
    }
    public function RoshidDistributeShow($id)
    {
        $data = RoshidDistribute::with('roshid:id,name,serial_no')->findOrFail($id);
        $invoice = InvoiceDesign::first();
        return view('dashboard.roshid.roshid_distribute_show', compact('data','invoice'));
    }
    public function RoshidDistributeEdit($id)
    {
        $data = RoshidDistribute::with('roshid:id,name,serial_no')->findOrFail($id);
        $items = RoshidStock::with('takenby')->select('name','slug','id','serial_no','roshid_taken_by')->get();
        return view('dashboard.roshid.roshid_distribute_edit', compact('data','items'));
    }
    public function RoshidDistributeUpdate(Request $request, $id)
    {
        $messages = [
            'date.required' => 'Enter date',
            'roshid_stock_id.required' => 'Select roshid',
            'name.required' => 'Enter name',
        ];

        $request->validate([
            'date' => 'required',
            'roshid_stock_id' => 'required',
            'name' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {
                $data = RoshidDistribute::findOrFail($id);
                $data->roshid_stock_id = $request->roshid_stock_id;
                $data->name = $request->name;
                $data->phone = $request->phone;
                $data->date = $this->formatAnyDateToYmd($request->date);;
                $data->page = $request->page;
                $data->note = $request->note;
                $data->type = 1;
                $data->created_by = Auth::user()->id;
                $data->save();

                $data->roshid_taken_by = $data->id;
                $data->save();

                if ($request->pre_roshid_stock_id == $request->roshid_stock_id) {
                    $roshid = RoshidStock::findOrFail($request->roshid_stock_id);
                    $roshid->status = 1;
                    $roshid->roshid_taken_by = $data->id;
                    $roshid->save();
                } else {
                    $oldRoshidStock = RoshidStock::findOrFail($request->pre_roshid_stock_id);
                    $oldRoshidStock->status = 2;
                    $oldRoshidStock->roshid_taken_by = $data->id;
                    $oldRoshidStock->save();

                    $newRoshidStock = RoshidStock::findOrFail($request->roshid_stock_id);
                    $newRoshidStock->status = 1;
                    $newRoshidStock->roshid_taken_by = $data->id;
                    $newRoshidStock->save();
                }

            DB::commit();

            return redirect()->route('roshid.distribute')->with('success', 'Roshid distributed successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function RoshidDistributeDestroy($id)
    {
        try {
            $data = RoshidDistribute::findOrFail($id);

            $oldRoshidStock = RoshidStock::findOrFail($data->roshid_stock_id);
            $oldRoshidStock->status = 2;
            $oldRoshidStock->save();

            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function RoshidTaken(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = RoshidDistribute::with('roshid:id,name,serial_no','takenby','refundby')->where('type', 2)->orderBy('id','DESC')->get();

                return DataTables::of($data)

                    ->addColumn('roshidName', function ($data) {
                        $name = $data->roshid->name ?? '';
                        $serial = $data->roshid->serial_no ?? '';
                        return $name . '<br>' . $serial;
                    })

                    ->addColumn('name', function ($data) {
                        $name = $data->refundby->name ?? '';
                        $phone = $data->refundby->phone ?? '';
                        return $name . '<br>' . $phone;
                    })

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $edit = '<a id="edit" href="' . route('roshid.taken.edit', $data->id) . '" class="btn btn-sm btn-primary edit" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="show" href="' . route('roshid.taken.show', $data->id) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('roshid.taken.destroy', $data->id) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return $edit . $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name', 'roshidName', 'date', 'amount', 'action'])
                    ->toJson();
            }

            return view('dashboard.roshid.roshid_taken');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function RoshidTakenStore(Request $request)
    {
        $messages = [
            'date.required' => 'Enter date',
            'roshid_refund_by.required' => 'Select name',
        ];

        $request->validate([
            'date' => 'required',
            'roshid_refund_by' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {
            $data = new RoshidDistribute();
            $data->date =  $this->formatAnyDateToYmd($request->date);
            $data->roshid_refund_by = $request->roshid_refund_by;
            $data->fund_id = $request->fund_id;
            $data->roshid_stock_id = $request->roshid_stock_id;
            $data->phone = $request->phone;
            $data->amount = $request->amount;
            $data->page = $request->page;
            $data->note = $request->note;
            $data->type = 2;
            $data->created_by = Auth::user()->id;
            $data->save();

            $roshid = RoshidStock::where('roshid_taken_by', $request->roshid_refund_by)->first();
            $roshid->status = 2;
            $roshid->save();

            $fund = new FundDetail();
            $fund->date = $this->formatAnyDateToYmd($request->date);;
            $fund->fund_id = $request->fund_id;
            $fund->roshid_refund_id = $data->id;
            $fund->name = $request->name;
            $fund->phone = $request->phone;
            $fund->amount = $request->amount;
            $fund->purpose = 'Roshid Taken';
            $fund->note = $request->note;
            $fund->type = 14;
            $fund->payment_type = 1;
            $fund->fund_type = 1;
            $fund->created_by = Auth::user()->id;
            $fund->save();

            DB::commit();

            return redirect()->route('roshid.taken')->with('success', 'Roshid taken successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function RoshidTakenCreate()
    {
        $users = RoshidDistribute::with('roshid')->where('type', 1)->get();
        $funds = Fund::all();
        return view('dashboard.roshid.roshid_taken_create', compact('users','funds'));
    }
    public function RoshidTakenShow($id)
    {
        $data = RoshidDistribute::with('roshid:id,name,serial_no','refundby')->where('type', 2)->findOrFail($id);
        $invoice = InvoiceDesign::first();
        return view('dashboard.roshid.roshid_taken_show', compact('data','invoice'));
    }
    public function RoshidTakenEdit($id)
    {
        $data = RoshidDistribute::with('roshid:id,name,serial_no','refundby:id,name,phone')->findOrFail($id);
        $users = RoshidDistribute::with('roshid')->where('type', 1)->get();
        $funds = Fund::all();
        return view('dashboard.roshid.roshid_taken_edit', compact('data','users','funds'));
    }
    public function RoshidTakenUpdate(Request $request, $id)
    {

        $messages = [
            'date.required' => 'Enter date',
            'roshid_refund_by.required' => 'Select name',
        ];

        $request->validate([
            'date' => 'required',
            'roshid_refund_by' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {
            $data = RoshidDistribute::findOrFail($id);
            $data->date = $this->formatAnyDateToYmd($request->date);;
            $data->roshid_refund_by = $request->roshid_refund_by;
            $data->roshid_stock_id = $request->roshid_stock_id;
            $data->fund_id = $request->fund_id;
            $data->phone = $request->phone;
            $data->amount = $request->amount;
            $data->page = $request->page;
            $data->note = $request->note;
            $data->type = 2;
            $data->updated_by = Auth::user()->id;
            $data->save();

            $roshid = RoshidStock::findOrFail($request->roshid_refund_by);
            $roshid->status = 2;
            $roshid->save();

            $fund = FundDetail::where('roshid_refund_id', $id)->firstOrFail();
            $fund->date = $this->formatAnyDateToYmd($request->date);;
            $fund->fund_id = $request->fund_id;
            $fund->roshid_refund_id = $data->id;
            $fund->name = $request->name;
            $fund->phone = $request->phone;
            $fund->amount = $request->amount;
            $fund->purpose = 'Roshid Taken';
            $fund->note = $request->note;
            $fund->type = 14;
            $fund->payment_type = 1;
            $fund->fund_type = 1;
            $fund->updated_by = Auth::user()->id;
            $fund->save();

            DB::commit();

            return redirect()->route('roshid.taken')->with('success', 'Roshid updated successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }
    public function RoshidTakenDestroy($id)
    {
        try {
            $data = RoshidDistribute::findOrFail($id);
            $oldRoshidStock = RoshidStock::findOrFail($data->roshid_stock_id);
            $oldRoshidStock->status = 2;
            $oldRoshidStock->save();
            FundDetail::where('roshid_refund_id', $data->id)->delete();
            $data->delete();

            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }

    public function Properties(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = Properties::select('id','product_name','price','type')->get();

                return DataTables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('name', function ($data) {
                        $name = $data->product_name ?? '';
                        return $name;
                    })

                    ->addColumn('type', function ($data) {
                        $type = '';
                        if($data->type== 1){
                            $type = 'Fixed Asset';
                        }else{
                            $type = 'Current Asset';
                        }
                        return $type;
                    })

                    ->addColumn('price', function ($data) {
                        return $data->price ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $edit = '<a id="edit" href="' . route('properties.edit', $data->id) . '" class="btn btn-sm btn-primary edit" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="show" href="' . route('properties.show', $data->id) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('properties.destroy', $data->id) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        return $edit . $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name', 'type', 'date', 'price', 'action'])
                    ->toJson();
            }

            return view('dashboard.roshid.properties');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function PropertiesStore(Request $request)
    {
        $messages = [
            'type.required' => 'Choose type',
            'product_name.required' => 'Enter product name',
        ];

        $request->validate([
            'type' => 'required',
            'product_name' => 'required',
        ], $messages);

        try {
            $data = new Properties();
            $data->date = now()->format('Y-m-d');
            $data->product_name = $request->product_name;
            $data->price = $request->price;
            $data->note = $request->note;
            $data->type = $request->type;
            $data->created_by = Auth::user()->id;
            $data->save();

            return redirect()->route('properties')->with('success', 'Properties created successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function PropertiesCreate()
    {
        return view('dashboard.roshid.properties_create');
    }
    public function PropertiesShow($id)
    {
        $data = Properties::findOrFail($id);
        $invoice = InvoiceDesign::first();
        return view('dashboard.roshid.properties_show', compact('data', 'invoice'));
    }
    public function PropertiesEdit($id)
    {
        $data = Properties::findOrFail($id);
        return view('dashboard.roshid.properties_edit', compact('data'));
    }
    public function PropertiesUpdate(Request $request, $id)
    {
        $messages = [
            'type.required' => 'Choose type',
            'product_name.required' => 'Enter product name',
        ];

        $request->validate([
            'type' => 'required',
            'product_name' => 'required',
        ], $messages);

        try {
            $data = Properties::findOrFail($id);
            $data->date = now()->format('Y-m-d');
            $data->product_name = $request->product_name;
            $data->price = $request->price;
            $data->type = $request->type;
            $data->note = $request->note;
            $data->updated_by = Auth::user()->id;
            $data->save();

            return redirect()->route('properties')->with('success', 'Properties updated successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }
    public function PropertiesDestroy($id)
    {
        try {
            $data = Properties::findOrFail($id);
            $data->delete();

            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function StudentSponsor(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = StudentSponsor::with('student:id,name,roll_no,register_no,class_id,session_id','donar:id,name,phone')->get();

                return DataTables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('donar', function ($data) {
                        $name = $data->donar->name ?? '';
                        $phone = $data->donar->phone ?? '';
                        return $name . '<br>' . $phone;
                    })

                    ->addColumn('student', function ($data) {
                        $name = $data->student->name ?? '';
                        $roll = $data->student->register_no ?? '';
                        $class = $data->student->stuclass->name ?? '';
                        return $name . '<br>' . $roll . '<br>' . $class;
                    })

                    ->addColumn('action', function ($data) {
                        $del = '<button id="remove" data-remote="' . route('student.sponsor.remove', $data->id) . '" class="btn btn-sm btn-danger btn-remove mr-1" title="remove"><i class="fas fa-times"></i></button>';

                        $edit = '<a id="edit" href="' . route('student.sponsor.edit', $data->id) . '" class="btn btn-sm btn-primary edit" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="show" href="' . route('student.sponsor.show', $data->id) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('student.sponsor.destroy', $data->id) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        return  $edit . $show .' '  .$del;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['donar', 'student', 'date', 'action'])
                    ->toJson();
            }

            return view('dashboard.roshid.student_sponsor');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentSponsorStore(Request $request)
    {
        $messages = [
            'student_id.required' => 'Choose student',
            'donar_id.required' => 'Enter product name',
        ];

        $request->validate([
            'student_id' => 'required',
            'donar_id' => 'required',
        ], $messages);

        $existing = StudentSponsor::where('student_id', $request->student_id)
                    ->where('donar_id', $request->donar_id)
                    ->exists();

        if ($existing) {
            return back()->with('error', 'This student is already sponsored ');
        }

        DB::beginTransaction();

        try {

            $data = new StudentSponsor();
            $data->date = now()->format('Y-m-d');
            $data->donar_id = $request->donar_id;
            $data->student_id = $request->student_id;
            // $data->amount = $request->amount;
            $data->note = $request->note;
            $data->created_by = Auth::user()->id;
            $data->save();

            $donar = Donar::findOrFail($data->donar_id);
            $student = Student::findOrFail($data->student_id);
            $student->etim_status = 1;
            $student->save();

            $status = optional(SmsStatus::first())->student_sponsor;

            $site = DB::table('settings')->first();
            $to = $donar->phone;
            $text = $site->name . ', '. $donar->name . ', আপনি এতিম ' . $student->name . ' এর দায়িত্ব নিয়েছেন, যার অনুমানিক মাসিক খরচ : ' . $request->amount . 'TK';
            $status = optional(SmsStatus::first())->student_sponsor;

            if($status == 1){
                $this->SendSms($to, $text);
            }

            DB::commit();

            return redirect()->route('student.sponsor')->with('success', 'Data created successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function StudentSponsorCreate()
    {
        $students = Student::with('stuclass')->where('etim', 1)->where('etim_status', 2)
        ->select('id','name','roll_no','register_no','class_id','session_id') ->where(function ($query) {
            $query->where('status', 1)
                ->orWhereNull('status');
        })->get();
        $donars = Donar::select('id','name','phone')->get();
        return view('dashboard.roshid.student_sponsor_create', compact('donars','students'));
    }
    public function StudentSponsorShow($id)
    {
        $data = StudentSponsor::with('student:id,name,roll_no,class_id,session_id,register_no','donar:id,name,phone')
        ->findOrFail($id);
        $invoice = InvoiceDesign::first();
        return view('dashboard.roshid.student_sponsor_show', compact('data','invoice'));
    }

    public function StudentSponsorEdit($id)
    {
        $data = StudentSponsor::findOrFail($id);
        $students = Student::with('stuclass')->where('etim', 1)
                    ->select('id','name','roll_no','register_no','class_id','session_id')
                    ->where(function ($query) {
                        $query->where('status', 1)
                            ->orWhereNull('status');
                    })->get();
        $donars = Donar::select('id','name','phone')->get();
        return view('dashboard.roshid.student_sponsor_edit', compact('data','students','donars'));
    }

    public function StudentSponsorUpdate(Request $request, $id)
    {
        $messages = [
            'student_id.required' => 'Choose student',
            'donar_id.required' => 'Enter product name',
        ];

        $request->validate([
            'student_id' => 'required',
            'donar_id' => 'required',
        ], $messages);

        $existing = StudentSponsor::where('student_id', $request->student_id)
                    ->where('donar_id', $request->donar_id)
                    ->exists();

        if ($existing) {
            return back()->with('error', 'This student is already sponsored ');
        }

        DB::beginTransaction();

        try {
            $data = StudentSponsor::findOrFail($id);
            $data->date = now()->format('Y-m-d');
            $data->donar_id = $request->donar_id;
            $data->student_id = $request->student_id;
            // $data->amount = $request->amount;
            $data->note = $request->note;
            $data->updated_by = Auth::user()->id;
            $data->save();

            $donar = Donar::findOrFail($data->donar_id);
            $student = Student::findOrFail($data->student_id);
            $student->etim_status = 1;
            $student->save();

            $site = DB::table('settings')->first();
            $to = $donar->phone;
            $text = $site->name . ', '. $donar->name . ', আপনি এতিম ' . $student->name . ' এর দায়িত্ব নিয়েছেন, যার অনুমানিক মাসিক খরচ : ' . $request->amount . 'TK';
            $status = optional(SmsStatus::first())->student_sponsor;

            if($status == 1){
                $this->SendSms($to, $text);
            }

            DB::commit();

            return redirect()->route('student.sponsor')->with('success', 'Data updated successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }
    public function StudentSponsorRemove($id)
    {
        try {
            $data = StudentSponsor::findOrFail($id);
            $student = Student::findOrFail($data->student_id);
            $student->etim_status = 2;
            $student->save();
            $data->delete();

            return response()->json([
                'success' => true,
                'message' => 'Student release successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function StudentSponsorDestroy($id)
    {
        try {
            $data = StudentSponsor::findOrFail($id);
            $student = Student::findOrFail($data->student_id);
            $student->etim_status = 2;
            $student->save();
            $data->delete();

            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }

    public function EtimStudentPayment(Request $request)
    {
        $studentFees = StudentFeeDetail::where('student_id', $request->student_id)->get();
        $feeAssign = FeeAssign::where('id', $studentFees->fee_assign_id)->get();
        $feeDeatils = FeeAssignDetail::where('fee_assign_id', $feeAssign->id)->get();
    }

    public function LoanGiven(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = Loan::with('returnby')->where('type', 1)->get();

                return DataTables::of($data)

                    ->addColumn('name', function ($data) {
                        $name = $data->name ?? '';
                        $phone = $data->phone ?? '';
                        return $name . '<br>' . $phone;
                    })

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? '';
                    })

                    ->addColumn('paid', function ($data) {
                        $paid = Loan::where('return_by', $data->id)->where('type', 2)->sum('amount');
                        return $paid;
                    })
                    ->addColumn('due', function ($data) {
                        $paid = Loan::where('return_by', $data->id)->where('type', 2)->sum('amount');
                        return $data->amount - $paid;
                    })

                    ->addColumn('purpose', function ($data) {
                        return $data->purpose ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $edit = '<a id="edit" href="' . route('loan.given.edit', $data->id) . '" class="btn btn-sm btn-primary edit" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="show" href="' . route('loan.given.show', $data->id) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('loan.given.destroy', $data->id) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return $edit . $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name', 'purpose', 'date', 'amount', 'action','paid','due'])
                    ->toJson();
            }

            return view('dashboard.roshid.loan_given');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function LoanGivenCreate()
    {
        $funds = Fund::all();
        return view('dashboard.roshid.loan_given_create', compact('funds'));
    }

    public function LoanGivenStore(Request $request)
    {
        $messages = [
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
            'name.required' => 'Enter name',
        ];

        $request->validate([
            'date' => 'required',
            'amount' => 'required',
            'name' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {
            $data = new Loan();
            $data->date = Carbon::parse($request->date)->format('Y-m-d');
            $data->fund_id = $request->fund_id;
            $data->amount = $request->amount;
            $data->name = $request->name;
            $data->phone = $request->phone;
            $data->purpose = $request->purpose;
            $data->note = $request->note;
            $data->type = 1;
            $data->status = 2;
            $data->created_by = Auth::user()->id;
            $data->save();

            $data->taken_by = $data->id;
            $data->save();

            $details = new FundDetail();
            $details->loan_id = $data->id;
            $details->date = $data->date;
            $details->fund_id = $request->fund_id;
            $details->amount = $request->amount;
            $details->name = $request->name;
            $details->phone = $request->phone;
            $details->purpose = $request->purpose;
            $details->note = $request->note;
            $details->type = 11;
            $details->payment_type = 2;
            $details->fund_type = 1;
            $details->created_by = Auth::user()->id;
            $details->save();

            DB::commit();

            return redirect()->route('loan.given')->with('success', 'Loan given successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function LoanGivenShow($id)
    {
        $data = Loan::where('type', 1)->findOrFail($id);
        $receipt_no = FundDetail::where('loan_id', $data->id)->first();
        $invoice = InvoiceDesign::first();
        return view('dashboard.roshid.loan_given_show', compact('data','receipt_no','invoice'));
    }
    public function LoanGivenEdit($id)
    {
        $data = Loan::where('type', 1)->findOrFail($id);
        $funds = Fund::all();
        return view('dashboard.roshid.loan_given_edit', compact('data','funds'));
    }
    public function LoanGivenUpdate(Request $request, $id)
    {
        $messages = [
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
            'name.required' => 'Enter name',
        ];

        $request->validate([
            'date' => 'required',
            'amount' => 'required',
            'name' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {
            $data = Loan::findOrFail($id);
            $data->date = Carbon::parse($request->date)->format('Y-m-d');
            $data->fund_id = $request->fund_id;
            $data->amount = $request->amount;
            $data->name = $request->name;
            $data->phone = $request->phone;
            $data->purpose = $request->purpose;
            $data->note = $request->note;
            $data->type = 1;
            $data->status = 2;
            $data->created_by = Auth::user()->id;
            $data->save();

            $data->taken_by = $data->id;
            $data->save();

            $details = FundDetail::where('loan_id', $data->id)->firstOrFail();
            $details->loan_id = $data->id;
            $details->date = $data->date;
            $details->fund_id = $request->fund_id;
            $details->amount = $request->amount;
            $details->name = $request->name;
            $details->phone = $request->phone;
            $details->purpose = $request->purpose;
            $details->note = $request->note;
            $details->type = 11;
            $details->payment_type = 2;
            $details->fund_type = 1;
            $details->created_by = Auth::user()->id;
            $details->save();

            DB::commit();

            return redirect()->route('loan.given')->with('success', 'Loan updated successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function LoanGivenDestroy($id)
    {
        try {
            $data = Loan::findOrFail($id);
            FundDetail::where('loan_id', $id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function LoanTaken(Request $request)
    {
        try {
            if ($request->ajax()) {

                $data = Loan::with('returnby')
                        ->where('type', 2)
                        ->groupBy('return_by')
                        ->selectRaw('return_by, sum(amount) as total_amount')
                        ->get();

                return DataTables::of($data)

                    ->addColumn('name', function ($data) {
                        $name = $data->returnby->name ?? '';
                        $phone = $data->returnby->phone ?? '';
                        return $name . '<br>' . $phone;
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->total_amount ?? '';
                    })

                    ->addColumn('total', function ($data) {
                        $total = Loan::where('taken_by', $data->return_by)->where('type', 1)->sum('amount');
                        return $total;
                    })

                    ->addColumn('due', function ($data) {
                        $total = Loan::where('taken_by', $data->return_by)->where('type', 1)->sum('amount');
                        return $total - $data->total_amount;
                    })

                    ->addColumn('action', function ($data) {

                        $show = '<a id="show" href="' . route('loan.taken.details', $data->return_by) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '';
                            if ($this->DeleteData()) {
                                $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('loan.taken.destroy', $data->return_by) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                            }
                        return $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name', 'amount', 'action' ,'total' ,'due'])
                    ->toJson();
            }

            return view('dashboard.roshid.loan_taken');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function LoanTakenDetails(Request $request, $id)
    {
        try {

            if ($request->ajax()) {

                $data = Loan::with('returnby')->where('type', 2)->where('return_by', $id)->get();

                return DataTables::of($data)

                    ->addColumn('name', function ($data) {
                        $name = $data->returnby->name ?? '';
                        $phone = $data->returnby->phone ?? '';
                        return $name . '<br>' . $phone;
                    })

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $edit = '<a id="edit" href="' . route('loan.taken.edit', $data->id) . '" class="btn btn-sm btn-primary edit" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="show" href="' . route('loan.taken.show', $data->id) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('loan.taken.destroy', $data->id) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return $edit . $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name', 'date', 'amount', 'action'])
                    ->toJson();
            }
            $invoice = InvoiceDesign::first();
            return view('dashboard.roshid.loan_taken_details', compact('invoice'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function LoanTakenCreate()
    {
        $users = Loan::where('type', 1)->where('status', 2)->select('id','name','phone','taken_by','amount')->get();
        $funds = Fund::all();
        return view('dashboard.roshid.loan_taken_create', compact('users','funds'));
    }

    public function LoanTakenStore(Request $request)
    {
        $messages = [
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date' => 'required',
            'amount' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {

            $data = new Loan();
            $data->date = Carbon::parse($request->date)->format('Y-m-d');
            $data->fund_id = $request->fund_id;
            $data->amount = $request->amount;
            $data->return_by = $request->return_by;
            $data->taken_by = $request->return_by;
            $data->name = $request->name;
            $data->phone = $request->phone;
            $data->note = $request->note;
            $data->type = 2;
            $data->created_by = Auth::user()->id;
            $data->save();

            $totalAmount = Loan::findOrFail($request->return_by);
            $paidAmount = Loan::where('type', 2)
                            ->where('return_by', $totalAmount->id)
                            ->sum('amount');

            if ($paidAmount >= $totalAmount->amount) {
                $totalAmount->status = 1;
                $totalAmount->save();
            }
            $details = new FundDetail();
            $details->loan_id = $data->id;
            $details->date = $data->date;
            $details->fund_id = $request->fund_id;
            $details->amount = $request->amount;
            $details->name = $request->name;
            $details->phone = $request->phone;
            $details->purpose = $request->purpose;
            $details->note = $request->note;
            $details->type = 12;
            $details->payment_type = 1;
            $details->fund_type = 1;
            $details->created_by = Auth::user()->id;
            $details->save();

            $to = $request->phone;
            $text = $request->name . ', your loan amount : ' . $data->amount . ' tk taken on ' . $data->date ;
            $this->SendSms($to, $text);

            DB::commit();

            return redirect()->route('loan.taken')->with('success', 'Loan taken successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function LoanTakenShow($id)
    {
        $data = Loan::with('returnby')->findOrFail($id);
        $invoice = InvoiceDesign::first();
        return view('dashboard.roshid.loan_taken_show', compact('data','invoice'));
    }
    public function LoanTakenEdit($id)
    {
        $data = Loan::with('returnby')->findOrFail($id);
        $paidAmount = Loan::where('type', 2)->where('taken_by', $data->taken_by)->sum('amount');
        $loanAmount = Loan::where('type', 1)->where('taken_by', $data->taken_by)->sum('amount');
        $users = Loan::where('type', 1)->with('takenby')->select('id','name','phone','taken_by','amount')->get();
        $funds = Fund::all();
        return view('dashboard.roshid.loan_taken_edit', compact('data','users','paidAmount','loanAmount','funds'));
    }
    public function LoanTakenUpdate(Request $request, $id)
    {
        $messages = [
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date' => 'required',
            'amount' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {

            $data = Loan::findOrFail($id);
            $data->date = Carbon::parse($request->date)->format('Y-m-d');
            $data->fund_id = $request->fund_id;
            $data->amount = $request->amount;
            $data->return_by = $request->return_by;
            $data->taken_by = $request->return_by;
            $data->name = $request->name;
            $data->phone = $request->phone;
            $data->note = $request->note;
            $data->type = 2;
            $data->created_by = Auth::user()->id;
            $data->save();

            $totalAmount = Loan::findOrFail($request->return_by);
            $paidAmount = Loan::where('type', 2)
                            ->where('return_by', $totalAmount->id)
                            ->sum('amount');

            if ($paidAmount + $request->amount >= $totalAmount->amount) {
                $totalAmount->status = 1;
                $totalAmount->save();
            }

            $details = new FundDetail();
            $details->loan_id = $data->id;
            $details->date = $data->date;
            $details->fund_id = $request->fund_id;
            $details->amount = $request->amount;
            $details->name = $request->name;
            $details->phone = $request->phone;
            $details->purpose = $request->purpose;
            $details->note = $request->note;
            $details->type = 12;
            $details->payment_type = 1;
            $details->fund_type = 1;
            $details->created_by = Auth::user()->id;
            $details->save();

            $to = $request->phone;
            $text = $request->name . ', your loan updated amount : ' . $data->amount . ' tk taken on ' . $data->date ;
            $this->SendSms($to, $text);

            DB::commit();

            return redirect()->route('loan.taken')->with('success', 'Loan updated successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function LoanTakenDestroy($id)
    {
        try {
            FundDetail::where('loan_id', $id)->delete();
            Loan::where('return_by', $id)->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function LoanTakenDetailsDestroy($id)
    {
        try {
            $data = Loan::findOrFail($id);
            FundDetail::where('loan_id', $id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }

    public function LoanPaidAmount(Request $request)
    {
        $amount = Loan::where('type', 2)->where('return_by', $request->return_by)->sum('amount');
        return response()->json($amount);
    }

    public function DueShopAmount(Request $request)
    {
        $dueAmount = HostelDue::where('type', 2)->where('shop_id', $request->shop_id)->sum('amount');
        $paidAmount = HostelDue::where('type', 1)->where('shop_id', $request->shop_id)->sum('amount');
        $amount = $dueAmount - $paidAmount;
        return response()->json($amount);
    }

    public function DueTaken(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = HostelDue::with('shop')
                    ->where('type', 2)
                    ->groupBy('shop_id')
                    ->selectRaw('shop_id, sum(amount) as amount')
                    ->get();

                return DataTables::of($data)

                    ->addColumn('name', function ($data) {
                        $name = $data->shop->name ?? '';
                        return $name ;
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? '';
                    })

                    ->addColumn('paid', function ($data) {
                        $paid = HostelDue::with('shop')->where('type', 1)->where('shop_id', $data->shop_id)->sum('amount');
                        return $paid ?? '';
                    })

                    ->addColumn('due', function ($data) {
                        $paid = HostelDue::with('shop')->where('type', 1)->where('shop_id', $data->shop_id)->sum('amount');
                        $due = $data->amount - $paid;
                        return $due ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $show = '<a id="show" href="' . route('due.taken.details', $data->shop_id) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('due.taken.destroy', $data->shop_id) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return  $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name','amount', 'action','paid' ,'due'])
                    ->toJson();
            }

            return view('dashboard.roshid.due_taken');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function DueTakenDetails(Request $request, $id)
    {
        try {

            if ($request->ajax()) {

                $data = HostelDue::with('shop')->where('type', 2)->where('shop_id', $id)->get();

                return DataTables::of($data)

                    ->addColumn('name', function ($data) {
                        $name = $data->shop->name ?? '';
                        return $name;
                    })

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $edit = '<a id="edit" href="' . route('due.taken.edit', $data->id) . '" class="btn btn-sm btn-primary edit" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="show" href="' . route('due.taken.show', $data->id) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('due.taken.details.destroy', $data->id) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return $edit . $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name', 'date', 'amount', 'action'])
                    ->toJson();
            }

            return view('dashboard.roshid.due_taken_details');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function DueGivenDetails(Request $request, $id)
    {
        try {

            if ($request->ajax()) {

                $data = HostelDue::with('shop','fund')->where('type', 1)->where('shop_id', $id)->get();

                return DataTables::of($data)

                    ->addColumn('name', function ($data) {
                        $name = $data->shop->name ?? '';
                        return $name;
                    })

                    ->addColumn('fund', function ($data) {
                        $name = $data->fund->name ?? '';
                        return $name;
                    })

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $edit = '<a id="edit" href="' . route('due.given.edit', $data->id) . '" class="btn btn-sm btn-primary edit" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="show" href="' . route('due.given.show', $data->id) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('due.given.details.destroy', $data->id) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return $edit . $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name', 'date', 'amount', 'action','fund'])
                    ->toJson();
            }

            return view('dashboard.roshid.due_given_details');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function DueTakenCreate()
    {
        $shops = Shop::all();
        return view('dashboard.roshid.due_taken_create', compact('shops'));
    }

    public function DueTakenStore(Request $request)
    {
        $messages = [
            'shop_id.required' => 'Select shop',
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date' => 'required',
            'amount' => 'required',
            'shop_id' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {

            $data = new HostelDue();
            $data->date = Carbon::parse($request->date)->format('Y-m-d');
            $data->fund_id = 1;
            $data->amount = $request->amount;
            $data->shop_id = $request->shop_id;
            $data->name = $request->name;
            $data->purpose = $request->purpose;
            $data->note = $request->note;
            $data->type = 2;
            $data->status = 2;
            $data->created_by = Auth::user()->id;
            $data->save();

            // $details = new FundDetail();
            // $details->due_id = $data->id;
            // $details->date = $data->date;
            // $details->fund_id = 1;
            // $details->amount = $request->amount;
            // $details->name = $request->name;
            // $details->purpose = $request->purpose;
            // $details->note = $request->note;
            // $details->type = 10;
            // $details->created_by = Auth::user()->id;
            // $details->save();

            DB::commit();

            return redirect()->route('due.taken')->with('success', 'Due created successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function DueTakenShow($id)
    {
        $data = HostelDue::with('shop')->findOrFail($id);
        $invoice = InvoiceDesign::first();
        return view('dashboard.roshid.due_taken_show', compact('data','invoice'));
    }
    public function DueTakenEdit($id)
    {
        $data = HostelDue::findOrFail($id);
        $shops = Shop::all();
        return view('dashboard.roshid.due_taken_edit', compact('data','shops'));
    }
    public function DueTakenUpdate(Request $request, $id)
    {
        $messages = [
            'shop_id.required' => 'Select shop',
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date' => 'required',
            'amount' => 'required',
            'shop_id' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {

            $data = HostelDue::findOrFail($id);
            $data->date = Carbon::parse($request->date)->format('Y-m-d');
            $data->fund_id = 1;
            $data->amount = $request->amount;
            $data->shop_id = $request->shop_id;
            $data->name = $request->name;
            $data->purpose = $request->purpose;
            $data->note = $request->note;
            $data->type = 2;
            $data->status = 2;
            $data->updated_by = Auth::user()->id;
            $data->save();

            // $details = FundDetail::where('due_id', $id)->first();
            // $details->due_id = $data->id;
            // $details->date = $data->date;
            // $details->fund_id = 1;
            // $details->amount = $request->amount;
            // $details->name = $request->name;
            // $details->purpose = $request->purpose;
            // $details->note = $request->note;
            // $details->type = 10;
            // $details->updated_by = Auth::user()->id;
            // $details->save();

            DB::commit();

            return redirect()->route('due.taken')->with('success', 'Due updated successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function DueTakenDestroy($id)
    {
        try {
            FundDetail::where('shop_id', $id)->delete();
            HostelDue::where('shop_id', $id)->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }

    public function DueTakenDetailsDestroy($id)
    {
        try {
            $data = HostelDue::findOrFail($id);
            FundDetail::where('due_id', $id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function DueGiven(Request $request)
    {
        try {

            if ($request->ajax()) {
                $data = HostelDue::with('shop')
                    ->where('type', 1)
                    ->groupBy('shop_id')
                    ->selectRaw('shop_id, sum(amount) as amount')
                    ->get();

                return DataTables::of($data)

                    ->addColumn('name', function ($data) {
                        $name = $data->shop->name ?? '';
                        return $name ;
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? '';
                    })

                    ->addColumn('total', function ($data) {
                        $total = HostelDue::with('shop')->where('type', 2)->where('shop_id', $data->shop_id)->sum('amount');
                        return $total ?? '';
                    })

                    ->addColumn('due', function ($data) {
                        $total = HostelDue::with('shop')->where('type', 2)->where('shop_id', $data->shop_id)->sum('amount');
                        $due = $total - $data->amount;
                        return $due ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $show = '<a id="show" href="' . route('due.given.details', $data->shop_id) . '" class="btn btn-sm btn-info show ml-1" title="show" ><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote="' . route('due.given.destroy', $data->shop_id) . '" title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return  $show . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['name','amount', 'action' , 'total' , 'due'])
                    ->toJson();
            }

            return view('dashboard.roshid.due_given');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function DueGivenCreate()
    {
        $shops = HostelDue::with('shop')->where('status' , 2)->where('type', 2)->groupBy('shop_id')->get();
        $funds = Fund::all();
        return view('dashboard.roshid.due_given_create', compact('shops', 'funds'));
    }

    public function DueGivenStore(Request $request)
    {
        $messages = [
            'shop_id.required' => 'Select shop',
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date' => 'required',
            'amount' => 'required',
            'shop_id' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {

            $shops = HostelDue::where('type', 2)
            ->where('shop_id', $request->shop_id)
            ->get();

            $totalAmount = HostelDue::where('type', 2)
            ->where('shop_id', $request->shop_id)
            ->sum('amount');

            foreach ($shops as $shop) {
                $paidAmount = HostelDue::where('type', 1)
                    ->where('shop_id', $shop->shop_id)
                    ->sum('amount');
                if ($paidAmount + $request->amount >= $totalAmount) {
                    $shop->status = 1;
                    $shop->save();
                }
            }

            $data = new HostelDue();
            $data->date = Carbon::parse($request->date)->format('Y-m-d');
            $data->fund_id = $request->fund_id;
            $data->amount = $request->amount;
            $data->shop_id = $request->shop_id;
            $data->purpose = 'Shop Due paid';
            $data->name = $request->name;
            $data->note = $request->note;
            $data->type = 1;
            $data->created_by = Auth::user()->id;
            $data->save();

            $details = new FundDetail();
            $details->due_id = $data->id;
            $details->date = $data->date;
            $details->fund_id = $request->fund_id;;
            $details->amount = $request->amount;
            $details->name = $request->name;
            $details->note = $request->note;
            $details->purpose = 'Shop Due paid';
            $details->type = 9;
            $details->fund_type = 1;
            $details->payment_type = 2;
            $details->created_by = Auth::user()->id;
            $details->save();

            DB::commit();

            return redirect()->route('due.given')->with('success', 'Due created successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function DueGivenShow($id)
    {
        $data = HostelDue::with(['shop','fund'])->findOrFail($id);
        $invoice = InvoiceDesign::first();
        return view('dashboard.roshid.due_given_show', compact('data','invoice'));
    }
    public function DueGivenEdit($id)
    {
        $data = HostelDue::findOrFail($id);
        $shops = HostelDue::with('shop')->where('type', 2)->get();
        $dueAmount = HostelDue::where('type', 2)->where('shop_id', $data->shop_id)->sum('amount');
        $paidAmount = HostelDue::where('type', 1)->where('shop_id', $data->shop_id)->sum('amount');
        $amount = $dueAmount - $paidAmount;
        $funds = Fund::all();
        return view('dashboard.roshid.due_given_edit', compact('data','shops','amount','funds'));
    }
    public function DueGivenUpdate(Request $request, $id)
    {
        $messages = [
            'shop_id.required' => 'Select shop',
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date' => 'required',
            'amount' => 'required',
            'shop_id' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {

            $shops = HostelDue::where('type', 2)
            ->where('shop_id', $request->shop_id)
            ->get();

            $totalAmount = HostelDue::where('type', 2)
            ->where('shop_id', $request->shop_id)
            ->sum('amount');

            $prePaid = $paidAmount = HostelDue::where('type', 1)
                    ->where('shop_id', $request->shop_id)
                    ->where('id', $id)
                    ->sum('amount');

            foreach ($shops as $shop) {
                $paidAmount = HostelDue::where('type', 1)
                    ->where('shop_id', $shop->shop_id)
                    ->sum('amount');

                if (($paidAmount + $request->amount -  $prePaid) >= $totalAmount) {
                    $shop->status = 1;
                    $shop->save();
                }
            }

            $data = HostelDue::findOrFail($id);
            $data->date = Carbon::parse($request->date)->format('Y-m-d');
            $data->fund_id = $request->fund_id;;
            $data->amount = $request->amount;
            $data->shop_id = $request->shop_id;
            $data->name = $request->name;
            $data->purpose = 'Shop Due paid';
            $data->note = $request->note;
            $data->type = 1;
            $data->updated_by = Auth::user()->id;
            $data->save();

            $details = FundDetail::where('due_id', $id)->first();
            $details->due_id = $data->id;
            $details->date = $data->date;
            $details->fund_id = $request->fund_id;;
            $details->amount = $request->amount;
            $details->name = $request->name;
            $details->purpose = 'Shop Due paid';
            $details->note = $request->note;
            $details->type = 9;
            $details->fund_type = 1;
            $details->payment_type = 2;
            $details->updated_by = Auth::user()->id;
            $details->save();

            DB::commit();

            return redirect()->route('due.given')->with('success', 'Due updated successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function DueGivenDestroy($id)
    {
        try {

            $data = HostelDue::findOrFail($id);
            FundDetail::where('due_id', $id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }

    public function GroupNotice()
    {
        return view('dashboard.settings.notice');
    }

    public function GroupNoticeStore(Request $request)
    {
        $students = Student::select('id', 'name', 'gurdian_mobile')  ->where(function ($query) {
            $query->where('status', 1)
                ->orWhereNull('status');
        })->get();
        $teachers = Teacher::whereNull('discharge_status')->select('id', 'name', 'phone')->get();
        $donars = Donar::select('id', 'name', 'phone')->get();

        $setting = Setting::first();

        if (empty($request->student_text) && empty($request->teacher_text) && empty($request->donar_text)) {
            return redirect()->back()->with('error', 'Please write at least one message.');
        }

        if (!empty($request->student_text)) {
            foreach ($students as $student) {
                $to = $student->gurdian_mobile;
                $text =  $setting->name .' '. $request->student_text;
                $this->SendSms($to, $text);
            }
        }

        if (!empty($request->teacher_text)) {
            foreach ($teachers as $teacher) {
                $to = $teacher->phone;
                $text = $setting->name .' '. $request->teacher_text;
                $this->SendSms($to, $text);
            }
        }

        if (!empty($request->donar_text)) {
            foreach ($donars as $donar) {
                $to = $donar->phone;
                $text = $setting->name .' '. $request->donar_text;
                $this->SendSms($to, $text);
            }
        }

        return redirect()->back()->with('success' ,' Sent notice successfully');
    }
    public function SingleNotice()
    {
        $students = Student::select('id','name','gurdian_mobile')->where(function ($query) {
            $query->where('status', 1)
                ->orWhereNull('status');
        })->get();
        $donars = Donar::select('id','name')->get();
        $teachers = Teacher::whereNull('discharge_status')->select('id','name')->get();
        $sessions = StudentSession::select('id','name')->get();
        return view('dashboard.settings.single_notice', compact('teachers','donars','students','sessions'));
    }

    public function SingleNoticeStore(Request $request)
    {
        $student = $request->student_id ? Student::select('gurdian_mobile', 'id')->findOrFail($request->student_id) : null;
        $teacher = $request->teacher_id ? Teacher::select('id', 'phone')->findOrFail($request->teacher_id) : null;
        $donar = $request->donar_id ? Donar::findOrFail($request->donar_id) : null;
        $setting = Setting::first();

        $type = $request->type;

        if (empty($student) && empty($teacher) && empty($donar)) {
            return redirect()->back()->with('error', 'Please select at least one user');
        }

        if (!empty($type) && $type == 1) {
            $to = $student->gurdian_mobile;
            $text = $setting->name .' '. $request->notice;
            $this->SendSms($to, $text);
        }

        if (!empty($type) && $type == 2) {
            $to = $teacher->phone;
            $text = $setting->name .' '. $request->notice;
            $this->SendSms($to, $text);
        }

        if (!empty($type) && $type == 3) {
            $to = $donar->phone;
            $text = $setting->name .' '. $request->notice;
            $this->SendSms($to, $text);
        }

        return redirect()->back()->with('success' ,' Sent notice successfully');
    }

    public function NoticePrint()
    {
        $invoice = InvoiceDesign::first();
        return view('dashboard.settings.notice_print', compact('invoice'));
    }

}
