<?php
namespace App\Http\Controllers;

use App\Models\Attendance;
use App\Models\AttendanceSetting;
use App\Models\InvoiceDesign;
use App\Models\SmsStatus;
use App\Models\Student;
use App\Models\StudentClass;
use App\Models\StudentSession;
use App\Models\Teacher;
use App\Models\User;
use App\Traits\DateFormatter;
use App\Traits\DeleteTrait;
use App\Traits\SmsTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Validator;
use Rats\Zkteco\Lib\ZKTeco;
use Yajra\DataTables\Facades\DataTables;

class AttendanceController extends Controller
{
    use SmsTrait;
    use DeleteTrait;
    use DateFormatter;
    public function TeacherAttendance(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = Attendance::with('teacher:id,name,reg_no')->where('type', 1)->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('status', function ($data) {
                        $status = '';
                        if ($data->status == 1) {
                            $status = '<span class="badge badge-primary">Present</span>';
                        } elseif ($data->status == 2) {
                            $status = '<span class="badge badge-danger">Absent</span>';
                        }
                        return $status;
                    })

                    ->addColumn('teacher', function ($data) {
                        $name = isset($data->teacher) ? $data->teacher->name : '--';
                        $roll = isset($data->teacher) ? $data->teacher->reg_no : '--';
                        return $name . '<br>' . $roll;
                    })

                    ->addColumn('action', function ($data) {

                        $result = '';

                        $details = '<li><a class="dropdown-item" href="' . route('teacher.attendance.show', $data->id) . ' " ><i class="fa fa-eye text-primary" ></i> Details</a></li>';

                        $edit = '<li><a class="dropdown-item" href="' . route('teacher.attendance.edit', $data->id) . ' "><i class="fa fa-edit text-success"></i> Edit</a></li>';

                        $delete = '<li><a class="dropdown-item btn-delete" href="#" data-remote=" ' . route('teacher.attendance.destroy', $data->id) . ' "><i class="fa fa-trash text-red"></i> Delete</a></li>';

                        $result = $details . $edit . $delete;

                        return '<div class="btn-group open">
                            <a class="badge badge-primary dropdown-toggle" href="#" role="button"  data-toggle="dropdown">Actions<i class="ik ik-chevron-down mr-0 align-middle"></i></a>
                            <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">' . $result . '
                            </ul>
                        </div>';
                    })

                    ->addIndexColumn()
                    ->rawColumns(['date', 'status', 'teacher', 'action'])
                    ->toJson();
            }
            return view('dashboard.attendance.teacher_attendance');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
public function TeacherAllAttendance(Request $request)
{
    try {
        if ($request->ajax()) {
            // Fetch distinct months for the dropdown
            $distinctMonths = Attendance::where('type', 1)->selectRaw('DISTINCT DATE_FORMAT(date, "%Y-%m") as month')->get();

            $data = Attendance::with('teacher:id,name,reg_no')->where('type', 1)->groupBy('date')->get();

            return Datatables::of($data)
                ->addColumn('date', function ($data) {
                    return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                })
                ->addColumn('teacher', function ($data) {
                    return Attendance::where('type', 1)->where('date', $data->date)->count();
                })
                ->addColumn('action', function ($data) {
                    $details = '<a href="' . route('teacher.attendance.show', $data->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i> Details</a> ';
                    $edit    = '<a href="' . route('teacher.attendance.edit', $data->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i> Edit</a> ';
                    $delete  = '<a href="#" data-remote="' . route('teacher.attendance.destroy', $data->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i> Delete</a>';

                    return $details . $edit . $delete;
                })
                ->addIndexColumn()
                ->rawColumns(['date', 'action', 'teacher'])
                ->toJson();
        }

        // Fetch distinct months for the dropdown
        $distinctMonths = Attendance::where('type', 1)->selectRaw('DISTINCT DATE_FORMAT(date, "%Y-%m") as month')->get();
        return view('dashboard.attendance.teacher_all_attendance', compact('distinctMonths'));
    } catch (\Exception $exception) {
        return redirect()->back()->with('error', $exception->getMessage());
    }
}

public function printAttendance($month)
{
    // Fetch attendance for the selected month
    $attendanceData = Attendance::where('type', 1)->where('date', 'like', $month . '%')->groupBy('teacher_id')->with('teacher:id,name,reg_no')->get();
    // dd($attendanceData);
    return view('dashboard.attendance.print', compact('attendanceData', 'month'));
}


    // teacher attendence bangla update by ashiq start 19.5.25
    public function TeacherAttendanceTake(Request $request)
    {
        try {
            $date             = Carbon::createFromFormat('d/m/Y', $request->date)->format('Y-m-d');
            $attendanceExists = Attendance::where('date', $date)->where('type', 1)->exists();
            $invoiceDesign    = InvoiceDesign::first();
            if ($attendanceExists) {
                return redirect()->back()->with('error', 'Attendance has already been taken for this date.');
            }
            $teachers = Teacher::whereNull('discharge_status')->select('id', 'name', 'reg_no', 'phone')->get();
            return view('dashboard.attendance.teacher_attendance_take', compact('teachers', 'date', 'invoiceDesign'));

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    // teacher attendence bangla update by ashiq end 19.5.25

    public function TeacherAttendanceCreate()
    {
        try {
            return view('dashboard.attendance.teacher_attendance_create');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function TeacherAttendanceStore(Request $request)
    {
        $messages = [
            'status.required' => 'Select attendance status',
            'date.required'   => 'Enter date',
        ];

        $request->validate([
            'status' => 'required|array',
            'date'   => 'required',
        ], $messages);

        try {

            $date   = Carbon::parse($request->date)->format('Y-m-d');
            $exists = Attendance::where('type', 1)
                ->whereDate('date', $date)
                ->exists();
            if ($exists) {
                return redirect()->back()->with('error', 'Attendance already taken for this date');
            }

            foreach ($request->teacher_id as $key => $teacherId) {
                $data             = new Attendance();
                $data->teacher_id = $teacherId;
                $data->type       = 1;
                $data->date       = Carbon::parse($request->date)->format('Y-m-d');
                $data->status     = $request->status[$teacherId];
                $data->created_by = Auth::user()->id;
                $data->save();

                $attenstatus = '';
                if ($data->status == 1) {
                    $attenstatus = 'উপস্থিত';
                } elseif ($data->status == 2) {
                    $attenstatus = 'অনুপস্থিত';
                } elseif ($data->status == 3) {
                    $attenstatus = 'দেরিতে উপস্থিত';
                } elseif ($data->status == 4) {
                    $attenstatus = 'ছুটি';
                } elseif ($data->status == 5) {
                    $attenstatus = 'অর্ধেক দিন';
                }

                $teacher = Teacher::findOrFail($teacherId);
                $status  = optional(SmsStatus::first())->teacher_attendance;
                $to      = $teacher->phone;
                $text    = $teacher->name . ', ' . $data->date . ' এ ' . $attenstatus;
                if ($status == 1) {
                    $this->SendSms($to, $text);
                }
            }

            return redirect()->route('teacher.all.attendance')->with('success', 'Attendance created successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function TeacherAttendanceShow($id)
    {
        try {
            $data     = Attendance::with('createdBy:id,name')->where('type', 1)->where('id', $id)->first();
            $teachers = Attendance::with('teacher:id,name,reg_no,phone')
                ->where('type', 1)
                ->where('date', $data->date)
                ->get();
            return view('dashboard.attendance.teacher_attendance_show', compact('data', 'teachers'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    // teacher attendence bangla update by ashiq start 19.5.25
    public function TeacherAttendanceEdit($id)
    {
        try {
            $data          = Attendance::findOrFail($id);
            $teachers      = Attendance::where('date', $data->date)->where('type', 1)->with('teacher:id,name,reg_no,phone')->get();
            $invoiceDesign = InvoiceDesign::first();
            return view('dashboard.attendance.teacher_attendance_edit', compact('teachers', 'data', 'invoiceDesign'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    // teacher attendence bangla update by ashiq end 19.5.25

    public function TeacherAttendanceUpdate(Request $request, $id)
    {
        $messages = [
            'status.required' => 'Select attendance status',
            'date.required'   => 'Enter date',
        ];

        $validator = Validator::make($request->all(), [
            'status' => 'required|array',
            'date'   => 'required',
        ], $messages);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors'  => $validator->errors(),
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::beginTransaction();

            foreach ($request->teacher_id as $attendanceId => $teacherId) {
                if (isset($request->status[$attendanceId])) {
                    $existingAttendance             = Attendance::findOrFail($attendanceId);
                    $existingAttendance->date       = Carbon::parse($request->date)->format('Y-m-d');
                    $existingAttendance->status     = $request->status[$attendanceId];
                    $existingAttendance->updated_by = Auth::id();
                    $existingAttendance->save();

                    $teacher = Teacher::find($teacherId);
                    if ($teacher && $teacher->phone) {
                        $status = optional(SmsStatus::first())->teacher_attendance;

                        $attenstatus = '';
                        if ($existingAttendance->status == 1) {
                            $attenstatus = 'উপস্থিত';
                        } elseif ($existingAttendance->status == 2) {
                            $attenstatus = 'অনুপস্থিত';
                        } elseif ($existingAttendance->status == 3) {
                            $attenstatus = 'দেরিতে উপস্থিত';
                        } elseif ($existingAttendance->status == 4) {
                            $attenstatus = 'ছুটি';
                        } elseif ($existingAttendance->status == 5) {
                            $attenstatus = 'অর্ধেক দিন';
                        }

                        if ($status == 1) {
                            $text = $teacher->name . ', ' . $existingAttendance->date . ' এ ' . $attenstatus;
                            $this->SendSms($teacher->phone, $text);
                        }
                    }
                }
            }

            DB::commit();

            if ($request->ajax()) {
                return response()->json([
                    'success'  => true,
                    'message'  => 'Attendance updated successfully',
                    'redirect' => route('teacher.all.attendance'),
                ]);
            }

            return redirect()->route('teacher.all.attendance')
                ->with('success', 'Attendance updated successfully');

        } catch (\Exception $exception) {
            DB::rollBack();

            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error: ' . $exception->getMessage(),
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'Error: ' . $exception->getMessage())
                ->withInput();
        }
    }

    // Helper method to get status text
    private function getStatusText($status)
    {
        $statuses = [
            1 => 'Present',
            2 => 'Absent',
            3 => 'Late',
            4 => 'Holiday',
            5 => 'Half Day',
        ];

        return $statuses[$status] ?? 'Unknown';
    }

    public function TeacherAttendanceDestroy($id)
    {
        try {
            // dd($id);
            $data = Attendance::findOrFail($id);
            Attendance::where('date', $data->date)->where('type', 1)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Attendance deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Attendance delete failed',
            ]);
        }
    }

    public function StudentAttendance(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = Attendance::with('students:id,name,roll_no,register_no', 'stuclass')
                    ->where('type', 2)
                    ->groupBy('date')
                    ->groupBy('class_id')
                    ->get();

                 $distinctMonths = Attendance::where('type', 2)->selectRaw('DISTINCT DATE_FORMAT(date, "%Y-%m") as month')->get();
                 $distinctClasses = Attendance::where('type', 2)->selectRaw('DISTINCT class_id')->with('stuclass')->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('stuclass', function ($data) {
                        return $data->stuclass->name ? $data->stuclass->name : '';
                    })

                    ->addColumn('student', function ($data) {
                        return Attendance::where('type', 2)->where('date', $data->date)->where('class_id', $data->class_id)->count();;
                    })

                    ->addColumn('action', function ($data) {

                        $result = '';

                        $details = '<li><a class="dropdown-item" href="' . route('student.attendance.show', $data->id) . ' " ><i class="fa fa-eye text-primary" ></i> Details</a></li>';

                        $edit = '<li><a class="dropdown-item" href="' . route('student.attendance.edit', $data->id) . ' "><i class="fa fa-edit text-success"></i> Edit</a></li>';

                        $delete = '<li><a class="dropdown-item btn-delete" href="#" data-remote=" ' . route('student.attendance.destroy', $data->id) . ' "><i class="fa fa-trash text-red"></i> Delete</a></li>';

                        $result = $details . $edit . $delete;

                        return '<div class="btn-group open">
                            <a class="badge badge-primary dropdown-toggle" href="#" role="button"  data-toggle="dropdown">Actions<i class="ik ik-chevron-down mr-0 align-middle"></i></a>
                            <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">' . $result . '
                            </ul>
                        </div>';
                    })

                    ->addIndexColumn()
                    ->rawColumns(['date', 'action', 'stuclass', 'student'])
                    ->toJson();
            }
                $distinctMonths = Attendance::where('type', 2)->selectRaw('DISTINCT DATE_FORMAT(date, "%Y-%m") as month')->get();

                $distinctClasses = Attendance::where('type', 2)->selectRaw('DISTINCT class_id')->with('stuclass')->get();
                // dd($distinctClasses);
            return view('dashboard.attendance.student_attendance', compact('distinctMonths','distinctClasses'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function StudentAttendanceCreate(Request $request)
    {
        try {
            $students = Student::select('id', 'name', 'roll_no', 'register_no')->get();
            $classes  = StudentClass::all();
            $sessions = StudentSession::all();
            return view('dashboard.attendance.student_attendance_create', compact('students', 'classes', 'sessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function printStudentAttendance($month, $class)
{
    // Fetch attendance for the selected month
    $attendanceData = Attendance::where('type', 2)->where('date', 'like', $month . '%')->where('class_id', $class)->groupBy('student_id')->with('students:id,name,register_no')->get();

    // dd($attendanceData);
    // dd($attendanceData);
    return view('dashboard.attendance.print_student', compact('attendanceData', 'month'));
}

    public function FindStudent(Request $request)
    {
        try {

            $date = Carbon::parse($request->date)->format('Y-m-d');

            $exists = Attendance::where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->whereDate('date', $date)
                ->exists();

            if ($exists) {
                return response()->json([
                    'success' => false,
                    'message' => 'Attendance already taken for this date.',
                ]);
            }

            $students = Student::where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->select('id', 'name', 'roll_no', 'register_no', 'status')
                ->where(function ($query) {
                    $query->where('status', 1)
                        ->orWhereNull('status');
                })
                ->get();

            $html = '';

            foreach ($students as $key => $student) {
                $html .= '<tr>';
                $html .= '<td>' . ($key + 1) . '</td>';
                $html .= '<td>' . $student->name . ' <br> ' . $student->roll_no . '</td>';
                $html .= '<td>' . $student->register_no . '</td>';
                $html .= '<td>';
                $html .= '<input type="hidden" name="student_id[]" value="' . $student->id . '">';
                $html .= '<div class="form-group d-flex align-items-center">';

                $html .= '<div class="radio radio-info radio-inline mx-3 mt-2">';
                $html .= '<input type="radio" name="status[' . $student->id . ']" class="present status-green" value="1" id="status_present_' . $student->id . '" data-status="' . $student->attendance_status . '" autocomplete="off" required>';
                $html .= '<label for="status_present_' . $student->id . '" style="font-weight: normal;"> Present</label>';
                $html .= '</div>';

                $html .= '<div class="radio radio-info radio-inline mx-3 mt-2">';
                $html .= '<input type="radio" name="status[' . $student->id . ']" class="absent status-red" value="2" id="status_absent_' . $student->id . '" data-status="' . $student->attendance_status . '" autocomplete="off">';
                $html .= '<label for="status_absent_' . $student->id . '" style="font-weight: normal;"> Absent</label>';
                $html .= '</div>';

                $html .= '<div class="radio radio-info radio-inline mx-3 mt-2">';
                $html .= '<input type="radio" name="status[' . $student->id . ']" class="late" value="3" id="status_late_' . $student->id . '" data-status="' . $student->attendance_status . '" autocomplete="off">';
                $html .= '<label for="status_late_' . $student->id . '" style="font-weight: normal;"> Late</label>';
                $html .= '</div>';

                $html .= '<div class="radio radio-info radio-inline mx-3 mt-2">';
                $html .= '<input type="radio" name="status[' . $student->id . ']" class="holiday" value="4" id="status_holiday_' . $student->id . '" data-status="' . $student->attendance_status . '" autocomplete="off">';
                $html .= '<label for="status_holiday_' . $student->id . '" style="font-weight: normal;"> Holiday</label>';
                $html .= '</div>';

                $html .= '<div class="radio radio-info radio-inline mx-3 mt-2">';
                $html .= '<input type="radio" name="status[' . $student->id . ']" class="halfday" value="5" id="status_halfday_' . $student->id . '" data-status="' . $student->attendance_status . '" autocomplete="off">';
                $html .= '<label for="status_halfday_' . $student->id . '" style="font-weight: normal;"> Half Day</label>';
                $html .= '</div>';

                $html .= '</div>';
                $html .= '</td>';
                $html .= '</tr>';
            }

            return response()->json(['html' => $html]);
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentAttendanceStore(Request $request)
    {
        $messages = [
            'status.required'     => 'Select attendance status',
            'date.required'       => 'Enter date',
            'session_id.required' => 'Select session',
            'class_id.required'   => 'Select class',
        ];

        $request->validate([
            'status'     => 'required',
            'date'       => 'required',
            'class_id'   => 'required',
            'session_id' => 'required',
        ], $messages);

        try {

            $date = Carbon::parse($request->date)->format('Y-m-d');

            $exists = Attendance::where('class_id', $request->class_id)
                ->whereDate('date', $date)
                ->exists();
            if ($exists) {
                return redirect()->back()->with('error', 'Attendance already taken for this date');
            }

            foreach ($request->student_id as $key => $studentId) {
                $data             = new Attendance();
                $data->student_id = $studentId;
                $data->session_id = $request->session_id;
                $data->class_id   = $request->class_id;
                $data->type       = 2;
                $data->date       = Carbon::parse($request->date)->format('Y-m-d');
                $data->status     = $request->status[$studentId];
                $data->created_by = Auth::user()->id;
                $data->save();

                $attenstatus = '';
                if ($data->status == 1) {
                    $attenstatus = 'উপস্থিত';
                } elseif ($data->status == 2) {
                    $attenstatus = 'অনুপস্থিত';
                } elseif ($data->status == 3) {
                    $attenstatus = 'দেরিতে উপস্থিত';
                } elseif ($data->status == 4) {
                    $attenstatus = 'ছুটি';
                } elseif ($data->status == 5) {
                    $attenstatus = 'অর্ধেক দিন ';
                }

                $student   = Student::findOrFail($studentId);
                $shiftName = optional($data->shift)->name ?? '';
                $status    = optional(SmsStatus::first())->student_attendance;
                $to        = $student->gurdian_mobile;

                $text = "{$student->name}, " . Carbon::parse($data->date)->format('d F') . ", এ {$attenstatus}";

                if ($status == 1) {
                    $this->SendSms($to, $text);
                }
            }

            return redirect()->route('student.attendance')->with('success', 'Attendance created successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function StudentAttendanceShow($id)
    {
        try {
            $data = Attendance::with('students:id,name,register_no,roll_no,class_id', 'createdBy:id,name')
                ->where('type', 2)->findOrFail($id);
            $students = Attendance::with('students:id,name,register_no,roll_no', 'createdBy:id,name', 'shift:shift_name,id')
                ->where('type', 2)
                ->where('class_id', $data->class_id)
                ->where('date', $data->date)
                ->get();
            return view('dashboard.attendance.student_attendance_show', compact('data', 'students'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentAttendanceEdit($id)
    {
        try {
            $classes  = StudentClass::all();
            $data     = Attendance::findOrFail($id);
            $students = Attendance::where('date', $data->date)
                ->where('class_id', $data->class_id)
                ->where('session_id', $data->session_id)
                ->where('type', 2)->with('students:id,name,register_no,roll_no,status')
                ->whereHas('students', function ($q) {
                    $q->where('status', 1)->orWhereNull('status');
                })
                ->get();
            $sessions = StudentSession::all();
            return view('dashboard.attendance.student_attendance_edit', compact('students', 'data', 'classes', 'sessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentAttendanceUpdate(Request $request, $id)
    {
        $messages = [
            'status.required' => 'Select attendance status',
            'date.required'   => 'Enter date',
        ];

        $validator = Validator::make($request->all(), [
            'status' => 'required|array',
            'date'   => 'required',
        ], $messages);

        if ($validator->fails()) {
            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'errors'  => $validator->errors(),
                ], 422);
            }
            return redirect()->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            DB::beginTransaction();

            foreach ($request->student_id as $attendanceId => $studentId) {
                if (isset($request->status[$attendanceId])) {
                    $existingAttendance             = Attendance::findOrFail($attendanceId);
                    $existingAttendance->class_id   = $request->class_id;
                    $existingAttendance->date       = Carbon::parse($request->date)->format('Y-m-d');
                    $existingAttendance->status     = $request->status[$attendanceId];
                    $existingAttendance->updated_by = Auth::id();
                    $existingAttendance->save();

                    $attenstatus = '';
                    if ($existingAttendance->status == 1) {
                        $attenstatus = 'উপস্থিত';
                    } elseif ($existingAttendance->status == 2) {
                        $attenstatus = 'অনুপস্থিত';
                    } elseif ($existingAttendance->status == 3) {
                        $attenstatus = 'দেরিতে উপস্থিত';
                    } elseif ($existingAttendance->status == 4) {
                        $attenstatus = 'ছুটি';
                    } elseif ($existingAttendance->status == 5) {
                        $attenstatus = 'অর্ধেক দিন ';
                    }

                    $student = Student::find($studentId);
                    if ($student && $student->gurdian_mobile) {
                        $status = optional(SmsStatus::first())->student_attendance;
                        if ($status == 1) {
                            $text = "{$student->name}, " . Carbon::parse($existingAttendance->date)->format('d F') . ", এ {$attenstatus}";
                            $this->SendSms($student->gurdian_mobile, $text);
                        }
                    }
                }
            }

            DB::commit();

            if ($request->ajax()) {
                return response()->json([
                    'success'  => true,
                    'message'  => 'Attendance updated successfully',
                    'redirect' => route('student.attendance'),
                ]);
            }

            return redirect()->route('student.attendance')
                ->with('success', 'Attendance updated successfully');

        } catch (\Exception $exception) {
            DB::rollBack();

            if ($request->ajax()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Error: ' . $exception->getMessage(),
                ], 500);
            }

            return redirect()->back()
                ->with('error', 'Error: ' . $exception->getMessage())
                ->withInput();
        }
    }

    public function StudentAttendanceDestroy($id)
    {
        try {
            $data = Attendance::findOrFail($id);
            Attendance::where('date', $data->date)->where('type', 2)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Attendance deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Attendance delete failed',
            ]);
        }
    }

    public function ExamFees()
    {
        return view('dashboard.students.student.lession');
    }
    public function AnnualAccounting()
    {
        return view('dashboard.students.student.lession');
    }
    public function KuturKhana()
    {
        return view('dashboard.students.student.lession');
    }
    public function Library()
    {
        return view('dashboard.students.student.lession');
    }
    public function Mosjid()
    {
        return view('dashboard.students.student.lession');
    }
    public function ImamSalary()
    {
        return view('dashboard.students.student.lession');
    }
    public function ImamLeave()
    {
        return view('dashboard.students.student.lession');
    }
    public function AreaSalary()
    {
        return view('dashboard.students.student.lession');
    }
    public function Grave()
    {
        return view('dashboard.students.student.lession');
    }
    public function Construction()
    {
        return view('dashboard.students.student.lession');
    }

    public function Attendance(Request $request)
    {
        $zk = new ZKTeco('192.168.0.173', 4370);
        $zk->connect();
        $zk->enableDevice();
        $zk->serialNumber();
        $attendances = $zk->getAttendance();
        $user        = $zk->getUser();
        $currentDate = now()->toDateString();

        $filteredAttendances = array_merge(
            array_filter($attendances, fn($attendance) => explode(' ', $attendance['timestamp'])[0] == $currentDate),
        );

        $start = Carbon::parse($request->start_date)->format('Y-m-d');
        $end   = Carbon::parse($request->end_date)->format('Y-m-d');

        $query = Attendance::with('user:id,name,fingerprint_id,phone,email')
            ->orderBy('id', 'desc');

        if ($request->has('start_date') && $start) {
            $query->whereDate('date', '>=', $start);
        }

        if ($request->has('end_date') && $end) {
            $query->whereDate('date', '<=', $end);
        }

        if ($request->has('user_id') && $request->user_id) {
            $query->where('user_id', $request->user_id);
        }

        $records = $query->get();
        $setting = AttendanceSetting::first();
        $users   = User::all();

        return view('dashboard.attendance.attendance', compact('setting', 'records', 'users'));
    }

    public function AttendanceCreate(Request $request)
    {
        DB::beginTransaction();

        try {
            $setting        = AttendanceSetting::first();
            $entryTimeStart = $setting->entry_start_time;
            $entryTimeEnd   = $setting->entry_end_time;
            $exitTimeStart  = $setting->exit_start_time;
            $exitTimeEnd    = $setting->exit_end_time;

            $zk = new ZKTeco('192.168.0.173', 4370);
            $zk->connect();
            $zk->enableDevice();
            $zk->serialNumber();
            $attendances = $zk->getAttendance();
            $users       = $zk->getUser();
            $currentDate = now()->toDateString();

            foreach ($attendances as $user) {

                $userId = User::where('fingerprint_id', $user['id'])->first();
                if ($userId) {
                    $attendanceDate = Carbon::parse($user['timestamp'])->format('Y-m-d');
                    $timestamp      = Carbon::parse($user['timestamp'])->format('H:i:s');

                    $attendance = Attendance::where('user_id', $userId->id)
                        ->where('date', $attendanceDate)
                        ->first();
                    if ($attendance) {
                        if ($timestamp >= $exitTimeStart && $timestamp <= $exitTimeEnd) {
                            $attendance->exit_time   = $timestamp;
                            $attendance->exit_status = 1;
                            $attendance->save();
                        }
                    } else {
                        $attendance                 = new Attendance();
                        $attendance->user_id        = $userId->id;
                        $attendance->fingerprint_id = $user['id'];
                        $attendance->date           = $attendanceDate;
                        if ($timestamp >= $entryTimeStart && $timestamp <= $entryTimeEnd) {
                            $attendance->entry_time   = $timestamp;
                            $attendance->status       = 1;
                            $attendance->entry_status = 1;
                        }
                        if ($timestamp >= $exitTimeStart && $timestamp <= $exitTimeEnd) {
                            $attendance->exit_time   = $timestamp;
                            $attendance->exit_status = 1;
                            $attendance->status      = 1;
                        }

                        $attendance->save();
                    }
                }
            }

            DB::commit();
            return redirect()->route('student.attendance')->with('success', 'Attendance saved successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    public function DeviceUser()
    {
        DB::beginTransaction();

        try {
            $zk = new ZKTeco('192.168.0.173', 4370);
            $zk->connect();
            $zk->enableDevice();
            $users = $zk->getUser();
            $zk->disableDevice();
            $zk->disconnect();

            DB::commit();

            return view('users.device_user', compact('users'));
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Error syncing to device: ' . $e->getMessage());
            throw $e;
        }
    }
    public function DeviceUserEdit($id)
    {
        try {
            $zk = new ZKTeco('192.168.0.173', 4370);
            $zk->connect();
            $zk->enableDevice();
            $users = $zk->getUser();
            $user  = collect($users)->firstWhere('uid', $id);
            return view('users.device_user_edit', compact('user'));
        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function DeviceUserUpdate(Request $request, $id)
    {
        try {
            $zk = new ZKTeco('192.168.0.173', 4370);
            $zk->connect();
            $zk->enableDevice();
            $users = $zk->getUser();
            $user  = collect($users)->firstWhere('uid', $id);
            $zk->setUser($id, $request->userid, $request->name, $request->role);

            return redirect()->route('device.user')->with('success', 'User updated successfull');

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }

    public function DeviceUserDelete($id)
    {
        try {
            $zk = new ZKTeco('192.168.0.173', 4370);
            $zk->connect();
            $zk->enableDevice();
            $users = $zk->getUser();

            $zk->removeUser($id);
            return redirect()->route('device.user')->with('success', 'User updated successfull');

        } catch (\Exception $e) {
            DB::rollBack();
            throw $e;
        }
    }
}
