<?php
namespace App\Http\Controllers;

use App\Models\FundDetail;
use App\Models\Role;
use App\Models\Student;
use App\Models\Teacher;
use App\Models\User;
use Illuminate\Database\Schema\Blueprint;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Schema;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\Facades\DataTables;

class DashboardController extends Controller
{
    public function dashboard()
    {
        // Cache frequently accessed data
        $settings = Cache::remember('settings', 3600, function () {
            return DB::table('settings')->first();
        });

        // Optimize student count query
        $totalStudents = Cache::remember('total_students', 3600, function () {
            return Student::where(function ($query) {
                $query->where('status', 1)
                    ->orWhereNull('status');
            })->count();
        });

        // Optimize fund calculations
        $fundStats = Cache::remember('fund_stats', 3600, function () {
            $totalIncome    = FundDetail::where('payment_type', 1)->sum('amount');
            $totalExpense   = FundDetail::where('payment_type', 2)->sum('amount');
            $totalTeacher   = Teacher::whereNull('discharge_status')->count();
            $currentBalance = $totalIncome - $totalExpense;

            return [
                'totalIncome'    => $totalIncome,
                'totalExpense'   => $totalExpense,
                'totalTeacher'   => $totalTeacher,
                'currentBalance' => $currentBalance,
            ];
        });

        $totalIncome    = FundDetail::where('payment_type', 1)->sum('amount');
        $totalExpense   = FundDetail::where('payment_type', 2)->sum('amount');
        $currentBalance = $totalIncome - $totalExpense;
        $totalTeacher   = \App\Models\Teacher::whereNull('discharge_status')->count();

        // Get all funds
        $allFunds = \App\Models\Fund::select('id', 'name', 'amount', 'fund_type')->get();

        $fundBalances = $allFunds->map(function ($fund) {
            $income = FundDetail::where('fund_id', $fund->id)
                ->where('payment_type', 1)
                ->sum('amount');
            $expense = FundDetail::where('fund_id', $fund->id)
                ->where('payment_type', 2)
                ->sum('amount');
            $hasTransactions = ($income > 0 || $expense > 0);
            $balance         = $hasTransactions ? ($income - $expense) : ($fund->amount ?? 0);
            // Attach a pseudo funds relationship for compatibility with the view
            $fund->balance = $balance;
            $fund->funds   = (object) [
                'name'      => $fund->name,
                'fund_type' => $fund->fund_type,
            ];
            return $fund;
        });

        // Calculate current and previous month for balance trend
        $currentMonth  = now()->format('Y-m');
        $previousMonth = now()->subMonth()->format('Y-m');

        // Current month income/expense
        $currentIncome = FundDetail::where('payment_type', 1)
            ->whereRaw("DATE_FORMAT(date, '%Y-%m') = ?", [$currentMonth])
            ->sum('amount');
        $currentExpense = FundDetail::where('payment_type', 2)
            ->whereRaw("DATE_FORMAT(date, '%Y-%m') = ?", [$currentMonth])
            ->sum('amount');
        $currentBalanceMonth = $currentIncome - $currentExpense;

        // Previous month income/expense
        $prevIncome = FundDetail::where('payment_type', 1)
            ->whereRaw("DATE_FORMAT(date, '%Y-%m') = ?", [$previousMonth])
            ->sum('amount');
        $prevExpense = FundDetail::where('payment_type', 2)
            ->whereRaw("DATE_FORMAT(date, '%Y-%m') = ?", [$previousMonth])
            ->sum('amount');
        $prevBalanceMonth = $prevIncome - $prevExpense;

        // Calculate percentage change
        $balanceTrend = $prevBalanceMonth != 0 ? (($currentBalanceMonth - $prevBalanceMonth) / abs($prevBalanceMonth)) * 100 : 0;

        // Student trend
        $currentStudents = Student::where(function (
            $query) {
            $query->where('status', 1)
                ->orWhereNull('status');
        })->whereRaw("DATE_FORMAT(created_at, '%Y-%m') = ?", [$currentMonth])->count();
        $prevStudents = Student::where(function ($query) {
            $query->where('status', 1)
                ->orWhereNull('status');
        })->whereRaw("DATE_FORMAT(created_at, '%Y-%m') = ?", [$previousMonth])->count();
        $studentTrend = $prevStudents != 0 ? (($currentStudents - $prevStudents) / abs($prevStudents)) * 100 : 0;

        // Income trend
        $currentIncomeMonth = FundDetail::where('payment_type', 1)
            ->whereRaw("DATE_FORMAT(date, '%Y-%m') = ?", [$currentMonth])
            ->sum('amount');
        $prevIncomeMonth = FundDetail::where('payment_type', 1)
            ->whereRaw("DATE_FORMAT(date, '%Y-%m') = ?", [$previousMonth])
            ->sum('amount');
        $incomeTrend = $prevIncomeMonth != 0 ? (($currentIncomeMonth - $prevIncomeMonth) / abs($prevIncomeMonth)) * 100 : 0;

        // Teacher trend
        $currentTeachers = Teacher::whereNull('discharge_status')
            ->whereRaw("DATE_FORMAT(created_at, '%Y-%m') = ?", [$currentMonth])
            ->count();
        $prevTeachers = Teacher::whereNull('discharge_status')
            ->whereRaw("DATE_FORMAT(created_at, '%Y-%m') = ?", [$previousMonth])
            ->count();
        $teacherTrend = $prevTeachers != 0 ? (($currentTeachers - $prevTeachers) / abs($prevTeachers)) * 100 : 0;

        // Monthly collections for the last 6 months
        $monthlyCollections = [];
        for ($i = 5; $i >= 0; $i--) {
            $month                           = now()->subMonths($i)->format('Y-m');
            $monthLabel                      = now()->subMonths($i)->format('M Y');
            $monthlyCollections[$monthLabel] = FundDetail::where('payment_type', 1)
                ->whereRaw("DATE_FORMAT(date, '%Y-%m') = ?", [$month])
                ->sum('amount');
        }

        //monthly due for collection for the last 6 months
        // Last 6 months labels & keys
        $months      = [];
        $monthLabels = [];
        for ($i = 36; $i >= 0; $i--) {
            $months[]      = now()->subMonths($i)->format('Y-m');
            $monthLabels[] = now()->subMonths($i)->format('M Y');
        }

// Students (status: 1 or NULL), counts per class
        $studentsByClass = DB::table('students')
            ->where(function ($q) {$q->where('status', 1)->orWhereNull('status');})
            ->select('id', 'class_id', 'student_type')
            ->get()->groupBy('class_id');
        $studentCount   = $studentsByClass->map->count();
        $hostelCount    = $studentsByClass->map(fn($c) => $c->where('student_type', 2)->count());
        $onabashikCount = $studentsByClass->map(fn($c) => $c->where('student_type', 1)->count());

// Classes present in fee table for these months
        $classIds = DB::table('fee_assign_details')
            ->whereIn('month', $months)->whereNull('deleted_at')
            ->distinct()->pluck('class_id');

// Type 1..4 fee sum per (class,month,type)
        $feeType = DB::table('fee_assign_details')
            ->whereIn('class_id', $classIds)
            ->whereIn('month', $months)
            ->whereIn('type', [1, 2, 3, 4])
            ->whereNull('deleted_at')
            ->select('class_id', 'month', 'type', DB::raw('SUM(amount) as total'))
            ->groupBy('class_id', 'month', 'type')->get()
            ->groupBy(fn($r) => $r->month . '|' . $r->class_id);

// Special (type=5) per month
        $specialByMonth = DB::table('fee_assign_details as fd')
            ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id')
            ->whereNull('fd.deleted_at')->whereNull('fds.deleted_at')
            ->where('fd.type', 5)->whereIn('fd.month', $months)
            ->select('fd.month', DB::raw('SUM(fds.amount) as total'))
            ->groupBy('fd.month')->pluck('total', 'month');

// OLD receive+discount per month
        $oldRD = DB::table('fund_details')
            ->whereIn('month', $months)
            ->whereIn('type', [6, 16])->where('payment_type', 1)
            ->whereNull('deleted_at')->whereNull('fund_detail_type_id')
            ->whereNull('fee_id')->whereNull('receipt_type')
            ->groupBy('month')
            ->select('month', DB::raw('SUM(amount) as recv'), DB::raw('SUM(discount) as disc'))
            ->get()->keyBy('month');

// NEW receive per month
        $newReceive = DB::table('money_receipts')
            ->whereIn('month', $months)->where('receipt_type', 'new')
            ->whereNull('deleted_at')
            ->groupBy('month')
            ->select('month', DB::raw('SUM(current_receive) as total'))
            ->get()->pluck('total', 'month');

// NEW discount per month (fund_details_type)
        $newDiscount = DB::table('fund_details_type')
            ->whereIn('month', $months)->where('receipt_type', 'new')
            ->whereNull('deleted_at')
            ->groupBy('month')
            ->select('month', DB::raw('SUM(discount) as total'))
            ->get()->pluck('total', 'month');

// Build monthly DUE only
        $monthlyDue = [];
        foreach ($months as $idx => $m) {
            $label = $monthLabels[$idx];

            // Assigned = (t1+t3)*all + t2*hostel + t4*onab + special
            $assigned = 0.0;
            $groups   = $feeType->filter(fn($v, $k) => str_starts_with($k, $m . '|'));
            foreach ($groups as $k => $coll) {
                [, $cid] = explode('|', $k);
                $cid     = (int) $cid;
                $t       = $coll->keyBy('type');

                $t1 = (float) ($t->get(1)->total ?? 0);
                $t2 = (float) ($t->get(2)->total ?? 0);
                $t3 = (float) ($t->get(3)->total ?? 0);
                $t4 = (float) ($t->get(4)->total ?? 0);

                $totalStd = (int) ($studentCount[$cid] ?? 0);
                $hostel   = (int) ($hostelCount[$cid] ?? 0);
                $onab     = (int) ($onabashikCount[$cid] ?? 0);

                $assigned += ($t1 + $t3) * $totalStd
                     + ($t2 * $hostel)
                     + ($t4 * $onab);
            }
            $assigned += (float) ($specialByMonth[$m] ?? 0);

            $oldR = (float) ($oldRD[$m]->recv ?? 0);
            $oldD = (float) ($oldRD[$m]->disc ?? 0);
            $nR   = (float) ($newReceive[$m] ?? 0);
            $nD   = (float) ($newDiscount[$m] ?? 0);

            $due                = max(0, $assigned - ($oldR + $oldD + $nR + $nD));
            $monthlyDue[$label] = round($due, 2);
        }

// Pass $monthlyDue & $monthLabels to view

        // Gender distribution
        $maleStudents   = \App\Models\Student::where('gender', 1)->count();
        $femaleStudents = \App\Models\Student::where('gender', 2)->count();

        // Active vs Inactive students
        $activeStudents = \App\Models\Student::where(function ($q) {$q->where('status', 1)->orWhereNull('status');})->count();
        $inactiveStudents = \App\Models\Student::where('status', 2)->count();

        // Top 3 funds by balance
        $topFunds = $fundBalances->sortByDesc('balance')->take(3);

        // Recent admissions (this month)
        $recentAdmissions = \App\Models\Student::whereMonth('created_at', now()->month)->whereYear('created_at', now()->year)->count();

        // Teacher-student ratio
        $teacherStudentRatio = $fundStats['totalTeacher'] > 0 ? round($totalStudents / $fundStats['totalTeacher'], 2) : 0;

        // Monthly expense trend (last 6 months)
        $monthlyExpenses = [];
        for ($i = 5; $i >= 0; $i--) {
            $month                        = now()->subMonths($i)->format('Y-m');
            $monthLabel                   = now()->subMonths($i)->format('M Y');
            $monthlyExpenses[$monthLabel] = FundDetail::where('payment_type', 2)
                ->whereRaw("DATE_FORMAT(date, '%Y-%m') = ?", [$month])
                ->sum('amount');
        }

        // Total funds count
        $totalFunds = \App\Models\Fund::count();
        // Total staff count
        $totalStaff = class_exists('App\\Models\\Staff') ? \App\Models\Staff::count() : 0;
        // Total classes
        $totalClasses = class_exists('App\\Models\\StudentClass') ? \App\Models\StudentClass::count() : 0;
        // Most used fund purpose
        $mostUsedPurpose = \App\Models\FundDetail::select('purpose', \DB::raw('count(*) as total'))
            ->groupBy('purpose')
            ->orderByDesc('total')
            ->value('purpose');
        // Highest single collection
        $highestCollection = \App\Models\FundDetail::where('payment_type', 1)->max('amount');
        // Highest single expense
        $highestExpense = \App\Models\FundDetail::where('payment_type', 2)->max('amount');
        // Oldest student
        $oldestStudent = \App\Models\Student::orderBy('join_date', 'asc')->first();
        // Newest student
        $newestStudent = \App\Models\Student::orderBy('join_date', 'desc')->first();

        return view('dashboard.include.main', [
            'totalStudents'       => $totalStudents,
            // 'totalIncome' => $fundStats['totalIncome'],
            'totalIncome'         => $totalIncome,
            'totalExpense'        => $fundStats['totalExpense'],
            'totalTeacher'        => $totalTeacher,
            // 'currentBalance' => $fundStats['currentBalance'],
            'currentBalance'      => $currentBalance,
            'fundBalances'        => $fundBalances,
            'site'                => $settings,
            'balanceTrend'        => $balanceTrend,
            'studentTrend'        => $studentTrend,
            'incomeTrend'         => $incomeTrend,
            'teacherTrend'        => $teacherTrend,
            'monthlyCollections'  => $monthlyCollections,
            'monthlyDue'          => $monthlyDue,
            'maleStudents'        => $maleStudents,
            'femaleStudents'      => $femaleStudents,
            'activeStudents'      => $activeStudents,
            'inactiveStudents'    => $inactiveStudents,
            'topFunds'            => $topFunds,
            'recentAdmissions'    => $recentAdmissions,
            'teacherStudentRatio' => $teacherStudentRatio,
            'monthlyExpenses'     => $monthlyExpenses,
            'totalFunds'          => $totalFunds,
            'totalStaff'          => $totalStaff,
            'totalClasses'        => $totalClasses,
            'mostUsedPurpose'     => $mostUsedPurpose,
            'highestCollection'   => $highestCollection,
            'highestExpense'      => $highestExpense,
            'oldestStudent'       => $oldestStudent,
            'newestStudent'       => $newestStudent,
        ]);
    }

    public function switchLang($lang)
    {
        if (in_array($lang, ['en', 'bn', 'ar'])) {
            Session::put('lang', $lang);
            App::setLocale($lang);
        }
        return redirect()->back();
    }

    public function User(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = DB::table('users')
                    ->leftJoin('roles', 'users.role_id', '=', 'roles.id')
                    ->whereNull('deleted_at')
                    ->whereNull('created_by')
                    ->whereNot('type', 3)
                    ->OrWhereNull('type')
                    ->select('users.*', 'roles.name as role_name')
                    ->orderBy('id', 'desc')->get();

                return Datatables::of($data)

                    ->addColumn('role', function ($data) {

                        return '<b>' . $data->role_name . '</b>';
                    })

                    ->addColumn('action', function ($data) {

                        if ($data->type == 1) {
                            $pass = '<a id="pass" href="' . route('user.password', $data->id) . ' " class="btn btn-sm btn-primary pass" title="password" data-toggle="modal" data-target="#passClass"><i class="fa fa-lock"></i></a> ';
                            return $pass;
                        } else {

                            $edit = '<a id="edit" href="' . route('user.edit', $data->id) . ' " class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editClass"><i class="fa fa-edit"></i></a> ';

                            $pass = '<a id="pass" href="' . route('user.password', $data->id) . ' " class="btn btn-sm btn-info pass" title="password" data-toggle="modal" data-target="#passClass"><i class="fa fa-lock"></i></a> ';

                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('user.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';

                            return $edit . $pass . $delete;
                        }
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action', 'role'])
                    ->toJson();
            }
            $roles = Role::all();
            return view('dashboard.settings.user', compact('roles'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function UserStore(Request $request)
    {
        if ($request->ajax()) {

            $data = Validator::make($request->all(), [
                'phone' => 'required|string|unique:users,phone,NULL,id,deleted_at,NULL',
            ]);

            if ($data->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $data->errors()->all(),
                ]);
            }

            try {
                $data            = new User();
                $data->name      = $request->name;
                $data->phone     = $request->phone;
                $data->email     = $request->email;
                $data->password  = Hash::make($request->password);
                $data->show_pass = $request->password;
                $data->role_id   = $request->role_id;
                $data->save();

                return response()->json([
                    'success' => true,
                    'message' => 'User created successfully',
                ]);

            } catch (\Exception $exception) {
                return response()->json([
                    'success' => false,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }

    public function UserEdit($id)
    {
        $data = User::findOrFail($id);
        return response()->json($data);
    }
    public function UserPassword($id)
    {
        $data = User::findOrFail($id);
        return response()->json($data);
    }

    public function UserUpdate(Request $request, $id)
    {
        $data = Validator::make($request->all(), [
            'phone' => 'required|string|unique:users,phone,' . $id . ',id,deleted_at,NULL',
        ]);

        if ($data->fails()) {
            return response()->json([
                'success' => false,
                'message' => $data->errors()->all(),
            ]);
        }

        try {
            $data            = User::findOrFail($id);
            $data->name      = $request->name;
            $data->phone     = $request->phone;
            $data->email     = $request->email;
            $data->password  = Hash::make($request->password);
            $data->show_pass = $request->password;
            $data->role_id   = $request->role_id;
            $data->save();

            return response()->json([
                'success' => true,
                'message' => 'User updated successfully',
            ]);

        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function UserDestroy($id)
    {
        try {
            $data = User::findOrFail($id);
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'User deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'User delete failed',
            ]);
        }
    }
    public function UserPasswordUpdate(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'password' => 'required|string|min:6|confirmed',
            ]);

            $data            = User::findOrFail($id);
            $data->password  = Hash::make($request->password);
            $data->email     = $request->email;
            $data->show_pass = $request->password;
            $data->save();

            return response()->json([
                'success' => true,
                'message' => 'Password updated successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Password update failed',
            ]);
        }
    }

    //user role and permission start
    public function userRoleIndex(Request $request)
    {
        if ($request->ajax()) {
            $roles = Role::select(['id', 'name', 'created_at']);
            return DataTables::of($roles)
                ->addIndexColumn()
                ->addColumn('actions', function ($row) {
                    $permBtn = '<a href="' . route('user.role.perms.edit', $row) . '" class="btn btn-sm btn-outline-secondary"><i class="fa fa-key"></i></a>';
                    $editBtn = '<button type="button" class="btn btn-sm btn-outline-primary editRoleBtn"
                    data-id="' . $row->id . '"
                    data-name="' . $row->name . '"
                    data-action="' . route('user.role.update', $row) . '">
                    <i class="fa fa-edit"></i></button>';

                    // শুধু role_id != 1 হলে delete দেখাবে
                    $deleteForm = '';
                    if ($row->id != 1) {
                        $deleteForm = '<form action="' . route('user.role.destroy', $row) . '" method="POST" style="display:inline;" onsubmit="return confirm(\'Delete?\')">'
                        . csrf_field() . method_field('DELETE') . '
            <button class="btn btn-sm btn-outline-danger"><i class="fa fa-trash"></i></button>
            </form>';
                    }

                    return $permBtn . ' ' . $editBtn . ' ' . $deleteForm;
                })

                ->rawColumns(['actions'])
                ->make(true);
        }

        return view('dashboard.settings.role');
    }

    // Store
    public function userRoleStore(Request $request)
    {
        $request->validate(['name' => 'required|string|max:100']);
        Role::create(['name' => $request->name]);
        return redirect()->route('user.role.index')->with('success', 'Role created');
    }

    // Update
    public function userRoleUpdate(Request $request, Role $role)
    {
        $request->validate(['name' => 'required|string|max:100']);
        $role->update(['name' => $request->name]);
        return redirect()->route('user.role.index')->with('success', 'Role updated');
    }

    // Delete
    public function userRoleDestroy(Role $role)
    {
        $role->delete();
        return redirect()->route('user.role.index')->with('success', 'Role deleted');
    }

    public function userRolePermissions(\App\Models\Role $role)
    {
        // permissions টেবিলে module, key, name কলাম ধরে নিলাম
        $permissions = \App\Models\Permission::orderBy('module')->orderBy('key')->get()
            ->groupBy('module'); // collection: [module => [Permission,...]]

        $rolePerms = $role->permissions()->pluck('permissions.id')->toArray(); // pivot: role_permission

        return view('dashboard.settings.permissions', compact('role', 'permissions', 'rolePerms'));
    }

    public function userRolePermissionsStore(Request $request, \App\Models\Role $role)
    {
        $ids = $request->input('permissions', []); // array of permission ids
                                                   // চাইলে validate করতে পারো: exists:permissions,id
        $role->permissions()->sync($ids);

        if ($request->ajax()) {
            return response()->json(['ok' => true, 'message' => 'Permissions updated']);
        }
        return redirect()->route('user.role.index')->with('success', 'Permissions updated');
    }

    //user role and permission end

    public function CreateTable()
    {
        $table  = DB::select('SHOW TABLES');
        $tables = array_map('current', $table);
        return view('dashboard.settings.table', compact('tables'));
    }

    public function CreateTableStore(Request $request)
    {
        $tableName = $request->table_name;
        $fields    = $request->fields;

        if (Schema::hasTable($tableName)) {
            Schema::table($tableName, function (Blueprint $table) use ($fields, $tableName) {
                foreach ($fields as $field) {
                    if (! Schema::hasColumn($tableName, $field['name'])) {
                        $this->addColumn($table, $field);
                    }
                }
            });

            return redirect()->back()->with('success', "Table `$tableName` updated successfully");

        } else {
            Schema::create($tableName, function (Blueprint $table) use ($fields) {
                $table->id();
                foreach ($fields as $field) {
                    $this->addColumn($table, $field);
                }
                $table->timestamps();
            });

            return redirect()->back()->with('success', "Table `$tableName` created successfully");
        }
    }

    private function addColumn(Blueprint $table, array $field)
    {
        $column = match ($field['type']) {
            'string'     => $table->string($field['name']),
            'integer'    => $table->integer($field['name']),
            'bigInteger' => $table->bigInteger($field['name']),
            'boolean'    => $table->boolean($field['name']),
            'text'       => $table->text($field['name']),
            'longText'   => $table->longText($field['name']),
            'date'       => $table->date($field['name']),
            'datetime'   => $table->dateTime($field['name']),
            'time'       => $table->time($field['name']),
            'timestamp'  => $table->timestamp($field['name']),
            'float'      => $table->float($field['name']),
            'decimal'    => $table->decimal($field['name'], 10, 2),
            'double'     => $table->double($field['name'], 15, 8),
            'binary'     => $table->binary($field['name']),
            'json'       => $table->json($field['name']),
            default      => $table->string($field['name']),
        };

        $column->nullable();

        if (! empty($field['default'])) {
            $column->default($field['default']);
        }
    }

    public function SettingMenu()
    {
        return view('dashboard.menu.setting');
    }
    public function StudentMenu()
    {
        return view('dashboard.menu.student');
    }
    public function HujurMenu()
    {
        return view('dashboard.menu.hujur');
    }
    public function EducationMenu()
    {
        return view('dashboard.menu.education');
    }
    public function AdministrationMenu()
    {
        return view('dashboard.menu.administration');
    }
    public function AccountingMenu()
    {
        return view('dashboard.menu.accounting');
    }
    public function LibraryMenu()
    {
        return view('dashboard.menu.library');
    }
    public function BoardingMenu()
    {
        return view('dashboard.menu.boarding');
    }
    public function MosjidMenu()
    {
        return view('dashboard.menu.mosjid');
    }
    public function RoshidMenu()
    {
        return view('dashboard.menu.roshid');
    }
    public function NoticeMenu()
    {
        return view('dashboard.menu.notice');
    }
    public function SponsorMenu()
    {
        return view('dashboard.menu.sponsor');
    }
    public function LoanMenu()
    {
        return view('dashboard.menu.loan');
    }
    public function SponsorDonationcMenu()
    {
        return view('dashboard.menu.sponsor_donation');
    }
}
