<?php
namespace App\Http\Controllers;

use App\Models\FeeAssign;
use App\Models\FeeAssignDetail;
use App\Models\FeeAssignDetailStudent;
use App\Models\Fund;
use App\Models\Student;
use App\Models\StudentClass;
use App\Models\StudentFee;
use App\Models\StudentSession;
use App\Traits\DeleteTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Yajra\DataTables\Facades\DataTables;

class FeesController extends Controller
{
    use DeleteTrait;
    public function AcademicFees(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = DB::table('student_fees')
                    ->where('type', 1)
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'desc')->get();

                return Datatables::of($data)

                    ->addColumn('action', function ($data) {

                        $edit = '<a id="edit" href="' . route('academic.fees.edit', $data->id) . ' " class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editClass"><i class="fa fa-edit"></i></a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('academic.fees.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }

                        return $edit . ' ' . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action'])
                    ->toJson();
            }
            return view('dashboard.fees.academic_fees');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function AcademicFeesStore(Request $request)
    {
        if ($request->ajax()) {

            $data = Validator::make($request->all(), [
                'name'   => 'required|string|unique:student_fees,name,NULL,id,deleted_at,NULL',
                'amount' => 'required|numeric',
            ]);

            if ($data->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $data->errors()->all(),
                ]);
            }

            try {
                $data             = new StudentFee();
                $data->name       = $request->name;
                $data->type       = $request->type;
                $data->created_by = Auth::user()->id;
                $data->save();

                return response()->json([
                    'success' => true,
                    'message' => 'Fee created successfully',
                ]);

            } catch (\Exception $exception) {
                return response()->json([
                    'success' => true,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }
    public function AcademicFeesEdit($id)
    {
        $data = StudentFee::findOrFail($id);
        return response()->json($data);
    }

    public function AcademicFeesUpdate(Request $request, $id)
    {
        $data = Validator::make($request->all(), [
            'name'   => 'required|string|unique:student_fees,name,' . $id . ',id,deleted_at,NULL',
            'amount' => 'required|numeric',
        ]);

        if ($data->fails()) {
            return response()->json([
                'success' => false,
                'message' => $data->errors()->all(),
            ]);
        }

        try {
            $data             = StudentFee::findOrFail($id);
            $data->name       = $request->name;
            $data->amount     = $request->amount;
            $data->type       = $request->type;
            $data->created_by = Auth::user()->id;
            $data->update();

            return response()->json([
                'success' => true,
                'message' => 'Fees updated successfully',
            ]);

        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function AcademicFeesDestroy($id)
    {
        try {
            $data   = StudentFee::findOrFail($id);
            $tables = DB::select("
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE COLUMN_NAME = 'fee_id'
            AND TABLE_SCHEMA = DATABASE()
            ");

            foreach ($tables as $table) {
                if ($table->TABLE_NAME !== 'student_fees') {
                    DB::table($table->TABLE_NAME)->where('fee_id', $data->id)->delete();
                }
            }
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Fees deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Fees delete failed',
            ]);
        }
    }

    public function HostelFees(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = DB::table('student_fees')
                    ->whereNull('deleted_at')
                    ->groupBy('type')
                    ->where('type', '!=', 5)
                    ->orderBy('id', 'desc')->get();

                return Datatables::of($data)
                    ->addColumn('type', function ($data) {
                        return $data->type;
                    })
                    ->editColumn('type', function ($data) {
                        $type = '';
                        if ($data->type == 1) {
                            $type = 'একাডেমিক ';
                        } elseif ($data->type == 2) {
                            $type = 'আবাসিক';
                        } elseif ($data->type == 3) {
                            $type = 'পরীক্ষা ফি';
                        } elseif ($data->type == 4) {
                            $type = 'অনাবাসিক';
                        }
                        return $type;
                    })
                    ->addColumn('details', function ($data) {
                        $details = '';
                        if ($data->type == 1) {
                            $details = 'নির্ধারিত ক্লাসের সকল শিক্ষার্থীদের জন্য';
                        } elseif ($data->type == 2) {
                            $details = 'নির্ধারিত ক্লাসের সকল আবাসিক শিক্ষার্থীদের জন্য';
                        } elseif ($data->type == 3) {
                            $details = 'নির্ধারিত ক্লাসের সকল শিক্ষার্থীদের জন্য';
                        } elseif ($data->type == 4) {
                            $details = 'নির্ধারিত ক্লাসের সকল অনাবাসিক শিক্ষার্থীদের জন্য';
                        }
                        return $details;
                    })
                    ->addColumn('action', function ($data) {
                        $show = '<a href="' . route('hostel.fees.showByType', $data->type) . ' " class="btn btn-sm btn-primary" ><i class="fa fa-eye"></i></a> ';
                        return $show;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action', 'type', 'details'])
                    ->toJson();
            }

            $funds = Fund::select('id', 'name')
                ->whereNull('deleted_at')
                ->get();

            return view('dashboard.fees.hostel_fees', compact('funds'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function HostelFeesShowByType(Request $request, $type)
    {
        try {
            if ($request->ajax()) {
               
                 $data = DB::table('student_fees')
                    ->whereNull('deleted_at')
                  
                    ->where('type', $type)
                   
                    ->orderBy('id', 'desc')
                    ->get();

                return Datatables::of($data)
                    ->addColumn('fund', function ($data) {
                        // dd($data);
                        $fundName = '';
                        if($data->fund_id){
                            $fundName = Fund::where('id', $data->fund_id)->first();
                            return $fundName->name ?? 'N/A';
                        }
                        return 'not set';
                    })

                    ->addColumn('action', function ($data) {
                        $edit = '<a id="edit" href="' . route('hostel.fees.edit', $data->id) . ' " class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editClass"><i class="fa fa-edit"></i></a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('hostel.fees.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }

                        return $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action', 'fund'])
                    ->toJson();
            }
            $funds = Fund::select('id', 'name')
                ->whereNull('deleted_at')
                ->get();
            return view('dashboard.fees.hostel_fees_show_by_type', compact('type', 'funds'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function HostelFeesStore(Request $request)
    {
        // dd($request->all());
        if ($request->ajax()) {

            $data = Validator::make($request->all(), [
                'name'   => 'required|string|unique:student_fees,name,NULL,id,deleted_at,NULL',
                'amount' => 'required|numeric',
                'type'   => 'required',
            ]);

            if ($data->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $data->errors()->all(),
                ]);
            }

            try {
                $data             = new StudentFee();
                $data->name       = $request->name;
                $data->amount     = $request->amount;
                $data->type       = $request->type;
                $data->fund_id    = $request->fund;
                $data->created_by = Auth::user()->id;
                $data->save();

                return response()->json([
                    'success' => true,
                    'message' => 'Fee created successfully',
                ]);

            } catch (\Exception $exception) {
                return response()->json([
                    'success' => true,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }
    public function HostelFeesEdit($id)
    {
        $data = StudentFee::findOrFail($id);
        return response()->json($data);
    }

    public function HostelFeesUpdate(Request $request, $id)
    {
        $data = Validator::make($request->all(), [
            'name'   => 'required|string|unique:student_fees,name,' . $id . ',id,deleted_at,NULL',
            'amount' => 'required|numeric',
            'type'   => 'required',
        ]);
        // dd($request->all());

        if ($data->fails()) {
            return response()->json([
                'success' => false,
                'message' => $data->errors()->all(),
            ]);
        }

        try {
            $data             = StudentFee::findOrFail($id);
            $data->name       = $request->name;
            $data->amount     = $request->amount;
            $data->type       = $request->type;
            $data->fund_id    = $request->fund;
            $data->created_by = Auth::user()->id;
            $data->update();

            return response()->json([
                'success' => true,
                'message' => 'Fees updated successfully',
            ]);

        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function HostelFeesDestroy($id)
    {
        try {
            $data   = StudentFee::findOrFail($id);
            $tables = DB::select("
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE COLUMN_NAME = 'fee_id'
            AND TABLE_SCHEMA = DATABASE()
            ");

            foreach ($tables as $table) {
                if ($table->TABLE_NAME !== 'student_fees') {
                    DB::table($table->TABLE_NAME)->where('fee_id', $data->id)->delete();
                }
            }

            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Fees deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Fees delete failed',
            ]);
        }
    }

    //onabashik fee
    public function OnabashikFees(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = DB::table('student_fees')
                    ->where('type', 4)
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'desc')->get();

                return Datatables::of($data)

                    ->addColumn('action', function ($data) {

                        $edit = '<a id="edit" href="' . route('onabashik.fees.edit', $data->id) . ' " class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editClass"><i class="fa fa-edit"></i></a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('onabashik.fees.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }

                        return $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action'])
                    ->toJson();
            }
            return view('dashboard.fees.onabashik');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function OnabashikFeesStore(Request $request)
    {
        if ($request->ajax()) {

            $data = Validator::make($request->all(), [
                'name'   => 'required|string|unique:student_fees,name,NULL,id,deleted_at,NULL',
                'amount' => 'required|numeric',
            ]);

            if ($data->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $data->errors()->all(),
                ]);
            }

            try {
                $data             = new StudentFee();
                $data->name       = $request->name;
                $data->amount     = $request->amount;
                $data->type       = 4;
                $data->created_by = Auth::user()->id;
                $data->save();

                return response()->json([
                    'success' => true,
                    'message' => 'Fee created successfully',
                ]);

            } catch (\Exception $exception) {
                return response()->json([
                    'success' => true,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }

    public function OnabashikFeesEdit($id)
    {
        $data = StudentFee::findOrFail($id);
        return response()->json($data);
    }

    public function OnabashikFeesUpdate(Request $request, $id)
    {
        $data = Validator::make($request->all(), [
            'name'   => 'required|string|unique:student_fees,name,' . $id . ',id,deleted_at,NULL',
            'amount' => 'required|numeric',
        ]);

        if ($data->fails()) {
            return response()->json([
                'success' => false,
                'message' => $data->errors()->all(),
            ]);
        }

        try {
            $data             = StudentFee::findOrFail($id);
            $data->name       = $request->name;
            $data->amount     = $request->amount;
            $data->type       = 4;
            $data->created_by = Auth::user()->id;
            $data->update();

            return response()->json([
                'success' => true,
                'message' => 'Fees updated successfully',
            ]);

        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }

    public function OnabashikFeesDestroy($id)
    {
        try {
            $data   = StudentFee::findOrFail($id);
            $tables = DB::select("
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE COLUMN_NAME = 'fee_id'
            AND TABLE_SCHEMA = DATABASE()
            ");

            foreach ($tables as $table) {
                if ($table->TABLE_NAME !== 'student_fees') {
                    DB::table($table->TABLE_NAME)->where('fee_id', $data->id)->delete();
                }
            }

            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Fees deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Fees delete failed',
            ]);
        }
    }

    public function OnabashikFeesAssign(Request $request)
    {
        try {
            if ($request->ajax()) {

                $data = FeeAssign::with('class:id,name', 'studentsession:id,name')->where('type', 4)->groupBy('class_id')->groupBy('session_id')->get();

                return Datatables::of($data)

                    ->addColumn('class', function ($data) {
                        $department = $data->class->name ?? '--';
                        return $department;
                    })

                    ->addColumn('session', function ($data) {
                        $session = $data->studentsession->name ?? '--';
                        return $session;
                    })

                    ->addColumn('action', function ($data) {

                        $edit = '<a id="details" href="' . route('onabashik.fees.assign.edit', ['class_id' => $data->class_id, 'session_id' => $data->session_id]) . '" class="btn btn-sm btn-info ml-1" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="details" href="' . route('onabashik.fees.assign.show', ['class_id' => $data->class_id, 'session_id' => $data->session_id]) . '" class="btn btn-sm btn-primary show ml-1" title="Details"><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {

                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('onabashik.fees.assign.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return $edit . $show . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['session', 'class', 'action'])
                    ->toJson();
            }
            return view('dashboard.fees.onabashik_fees_assign');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function OnabashikFeesAssignCreate()
    {
        $fees              = StudentFee::where('type', 4)->get();
        $class             = StudentClass::all();
        $sessions          = StudentSession::all();
        $lastAssignedMonth = FeeAssignDetail::orderBy('month', 'desc')->where('type', 4)->value('month');
        return view('dashboard.fees.onabashik_fees_assign_create', compact('fees', 'class', 'sessions', 'lastAssignedMonth'));
    }

    public function OnabashikFeesMonth(Request $request)
    {
        $validatedData = $request->validate([
            'type'       => 'required',
            'session_id' => 'required',
            'class_id'   => 'required',
            'month_for'  => 'required',
            'from_month' => 'required',
            'fee_name'   => 'array',
            'fee_name.*' => 'required',
        ]);

        $fromMonth = \DateTime::createFromFormat('F-Y', $validatedData['from_month']);
        if (! $fromMonth) {
            return back()->with('error', 'Invalid month format');
        }

        $data = [
            'type'          => $validatedData['type'],
            'session_id'    => $validatedData['session_id'],
            'class_id'      => $validatedData['class_id'],
            'month_for'     => $validatedData['month_for'],
            'from_month'    => $fromMonth->format('Y-m'),
            'selected_fees' => $validatedData['fee_name'] ?? [],
        ];

        $startMonth = Carbon::createFromFormat('Y-m', $data['from_month']);
        $allMonths  = [];

        for ($i = 0; $i < $data['month_for']; $i++) {
            $month       = $startMonth->copy()->addMonthsNoOverflow($i)->format('Y-m');
            $allMonths[] = $month;
        }

        $class   = StudentClass::findOrFail($data['class_id']);
        $session = StudentSession::findOrFail($data['session_id']);
        $fees    = StudentFee::whereIn('id', $data['selected_fees'])->get();
        return view('dashboard.fees.onabashik_fees_assign_create_month', compact('data', 'fees', 'class', 'session', 'allMonths'));
    }

    public function OnabashikFeesAssignStore(Request $request)
    {
        $messages = [
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'amount' => 'required|array',
        ], $messages);

        foreach ($request->month as $monthIndex => $month) {
            foreach ($request->fee_id[$monthIndex] as $key => $feeId) {
                $existingFeeAssign = FeeAssign::where('session_id', $request->session_id)
                    ->where('class_id', $request->class_id)
                    ->where('type', 4)
                    ->first();

                if ($existingFeeAssign) {
                    $existingAssignment = FeeAssignDetail::with('fees')->where('fee_assign_id', $existingFeeAssign->id)
                        ->where('month', \Carbon\Carbon::createFromFormat('Y-m', $month)->format('Y-m'))
                        ->where('fee_id', $feeId)
                        ->where('type', 4)
                        ->first();

                    if ($existingAssignment) {
                        return redirect()->back()->with('error', 'For this class & session same fees already assigned');
                    }
                }
            }
        }

        DB::beginTransaction();

        try {

            $assign             = new FeeAssign();
            $assign->session_id = $request->session_id;
            $assign->class_id   = $request->class_id;
            $assign->type       = 4;
            $assign->created_by = Auth::user()->id;
            $assign->save();

            foreach ($request->month as $monthIndex => $month) {
                foreach ($request->fee_id[$monthIndex] as $key => $value) {
                    $assignDetails                = new FeeAssignDetail();
                    $assignDetails->type          = 4;
                    $assignDetails->fee_assign_id = $assign->id;
                    $assignDetails->fee_id        = $value;
                    $assignDetails->month         = $month;
                    $assignDetails->session_id    = $assign->session_id;
                    $assignDetails->class_id      = $assign->class_id;
                    $assignDetails->amount        = $request->amount[$monthIndex][$key];
                    $assignDetails->created_by    = Auth::user()->id;
                    $assignDetails->save();
                }
            }

            DB::commit();

            return redirect()->route('onabashik.fees.assign')->with('success', 'Fees assigned successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function OnabashikFeesAssignShow($class_id, $session_id)
    {
        $data = FeeAssign::with('class:id,name', 'studentsession:id,name')
            ->where('type', 4)
            ->where('class_id', $class_id)
            ->where('session_id', $session_id)
            ->get();

        $feeAssignIds = $data->pluck('id');

        $details = FeeAssignDetail::with('fees')
            ->where('type', 4)
            ->whereIn('fee_assign_id', $feeAssignIds)
            ->get()
            ->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->month)->format('Y-m');
            });
        return view('dashboard.fees.onabashik_fees_assign_show', compact('data', 'details'));
    }

    public function OnabashikFeesAssignEdit($class_id, $session_id)
    {
        $data = FeeAssign::with('class:id,name', 'studentsession:id,name')
            ->where('type', 4)
            ->where('class_id', $class_id)
            ->where('session_id', $session_id)
            ->get();

        $feeAssignIds = $data->pluck('id');

        $details = FeeAssignDetail::with('fees')
            ->where('type', 4)
            ->whereIn('fee_assign_id', $feeAssignIds)
            ->get()
            ->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->month)->format('Y-m');
            });

        return view('dashboard.fees.onabashik_fees_assign_edit', compact('data', 'details'));
    }

    public function OnabashikFeesAssignUpdate(Request $request, $class_id, $session_id)
    {
        $messages = [
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'amount'   => 'required|array',
            'amount.*' => 'required|numeric',
        ], $messages);

        try {

            foreach ($request->id as $key => $id) {
                $assignDetails = FeeAssignDetail::where('class_id', $class_id)
                    ->where('session_id', $session_id)
                    ->where('type', 4)
                    ->findOrFail($id);

                $assignDetails->amount     = $request->amount[$key];
                $assignDetails->session_id = $request->session_id;
                $assignDetails->class_id   = $request->class_id;
                $assignDetails->created_by = Auth::user()->id;
                $assignDetails->save();
            }

            return redirect()->route('onabashik.fees.assign')->with('success', 'Fees assigned updated successfully');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function OnabashikFeesAssignDestroy($id)
    {
        try {
            $data = FeeAssign::findOrFail($id);
            // dd($data, $id);
            FeeAssignDetail::where('fee_assign_id', $id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }

    //onabashik fee end

    //specific fee
    public function SpecialFees(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = DB::table('student_fees')
                    ->where('type', 5)
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'desc')->get();
// dd($data);
                return Datatables::of($data)
                ->addColumn('fund', function ($data) {
                    // dd($data);
                    $fundName = '';
                    if($data->fund_id){
                        $fundName = Fund::where('id', $data->fund_id)->first();
                        return $fundName->name ?? 'N/A';
                    }
                    return 'not set';
                })

                    ->addColumn('action', function ($data) {

                        $edit = '<a id="edit" href="' . route('special.fees.edit', $data->id) . ' " class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editSpecialFee"><i class="fa fa-edit"></i></a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('onabashik.fees.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }

                        return $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action','fund'])
                    ->toJson();
            }

            $funds = Fund::whereNull('deleted_at')->get();
            return view('dashboard.fees.special_fees', compact('funds'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function SpecialFeesStore(Request $request)
    {
        if ($request->ajax()) {

            $data = Validator::make($request->all(), [
                'name' => 'required|string|unique:student_fees,name,NULL,id,deleted_at,NULL',
            ]);

            if ($data->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $data->errors()->all(),
                ]);
            }

            try {
                $data             = new StudentFee();
                $data->name       = $request->name;
                $data->type       = 5;
                $data->amount     = $request->amount ?? 0;
                $data->fund_id    = $request->fund;
                $data->created_by = Auth::user()->id;
                $data->save();

                return response()->json([
                    'success' => true,
                    'message' => 'Fee created successfully',
                ]);

            } catch (\Exception $exception) {
                return response()->json([
                    'success' => true,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }

    public function SpecialFeesEdit($id)
    {
        $data = StudentFee::findOrFail($id);
        return response()->json($data);
    }

    public function SpecialFeesUpdate(Request $request, $id)
    {
        $data = Validator::make($request->all(), [
            'name' => 'required|string|unique:student_fees,name,' . $id . ',id,deleted_at,NULL',
        ]);

        if ($data->fails()) {
            return response()->json([
                'success' => false,
                'message' => $data->errors()->all(),
            ]);
        }

        try {
            $data             = StudentFee::findOrFail($id);
            $data->name       = $request->name;
            $data->amount     = $request->amount ?? 0;
            $data->type       = 5;
            $data->fund_id    = $request->fund;
            $data->created_by = Auth::user()->id;
            $data->update();

            return response()->json([
                'success' => true,
                'message' => 'Fees updated successfully',
            ]);

        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }

    public function SpecialFeesDestroy($id)
    {
        try {
            $data   = StudentFee::findOrFail($id);
            $tables = DB::select("
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE COLUMN_NAME = 'fee_id'
            AND TABLE_SCHEMA = DATABASE()
            ");

            foreach ($tables as $table) {
                if ($table->TABLE_NAME !== 'student_fees') {
                    DB::table($table->TABLE_NAME)->where('fee_id', $data->id)->delete();
                }
            }

            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Fees deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Fees delete failed',
            ]);
        }
    }

    public function SpecialFeesAssign(Request $request)
    {
        try {
            if ($request->ajax()) {

                $data = FeeAssign::with('class:id,name', 'studentsession:id,name')->where('type', 5)->groupBy('class_id')->groupBy('session_id')->get();

                return Datatables::of($data)

                    ->addColumn('class', function ($data) {
                        $department = $data->class->name ?? '--';
                        return $department;
                    })

                    ->addColumn('session', function ($data) {
                        $session = $data->studentsession->name ?? '--';
                        return $session;
                    })

                    ->addColumn('action', function ($data) {

                        $show = '<a id="messageShow" class="btn btn-sm btn-info ml-1" href=" ' . route('special.fees.assign.show', ['class_id' => $data->class_id, 'session_id' => $data->session_id]) . ' " title="Show"><i class="fa fa-eye"></i></a>';

                        return $show;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['session', 'class', 'action'])
                    ->toJson();
            }
            return view('dashboard.fees.special_fees_assign');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function SpecialFeesAssignCreate()
    {
        $fees              = StudentFee::where('type', 5)->get();
        $class             = StudentClass::all();
        $sessions          = StudentSession::all();
        $specialFees       = StudentFee::where('type', 5)->whereNull('deleted_at')->get();
        $lastAssignedMonth = FeeAssignDetail::orderBy('month', 'desc')->where('type', 5)->value('month');
        return view('dashboard.fees.special_fees_assign_create', compact('fees', 'class', 'sessions', 'specialFees', 'lastAssignedMonth'));
    }

    public function getStudentBySession(Request $request)
    {
        $students = Student::where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->where(function ($query) {
                $query->where('status', 1)
                    ->orWhereNull('status');
            })
            ->select('id', 'name', 'register_no', 'roll_no', 'session_id', 'class_id', 'student_type')
            ->get();

        // dd($students, $request->all());

        $students = $students->map(function ($student) use ($request) {
            // feeassign_details_students এর সাথে join করতে হবে এবং special_fees_id মিলাতে হবে
            $assigned = \DB::table('feeassign_details_students as fds')
                ->join('fee_assign_details as fd', 'fd.id', '=', 'fds.fee_assign_details_id')
                ->where('fds.student_id', $student->id)
                ->where('fds.fee_assign_id', $request->feeAssignId)
                ->where('fds.class_id', $student->class_id)
                ->where('fds.session_id', $student->session_id)
                ->where('fd.fee_id', $request->special_fees_id)
                ->whereNull('fd.deleted_at')  // fee_assign_details এর soft delete check
                ->whereNull('fds.deleted_at') // feeassign_details_students এর soft delete check
                ->select('fds.amount')
                ->first();

            // dd($assigned);

            // amount থাকলে সেট করবে, না থাকলে null
            $student->amount = $assigned ? $assigned->amount : null;

            return $student;
        });

        return response()->json($students);
    }

    public function SpecialFeesMonth(Request $request)
    {
        dd($request->all());
        $validatedData = $request->validate([
            'type'       => 'required',
            'session_id' => 'required',
            'class_id'   => 'required',
            'month_for'  => 'required',
            'from_month' => 'required',
            'fee_name'   => 'array',
            'fee_name.*' => 'required',
        ]);

        $fromMonth = \DateTime::createFromFormat('F-Y', $validatedData['from_month']);
        if (! $fromMonth) {
            return back()->with('error', 'Invalid month format');
        }

        $data = [
            'type'          => $validatedData['type'],
            'session_id'    => $validatedData['session_id'],
            'class_id'      => $validatedData['class_id'],
            'month_for'     => $validatedData['month_for'],
            'from_month'    => $fromMonth->format('Y-m'),
            'selected_fees' => $validatedData['fee_name'] ?? [],
        ];

        $startMonth = Carbon::createFromFormat('Y-m', $data['from_month']);
        $allMonths  = [];

        for ($i = 0; $i < $data['month_for']; $i++) {
            $month       = $startMonth->copy()->addMonthsNoOverflow($i)->format('Y-m');
            $allMonths[] = $month;
        }

        $class   = StudentClass::findOrFail($data['class_id']);
        $session = StudentSession::findOrFail($data['session_id']);
        $fees    = StudentFee::whereIn('id', $data['selected_fees'])->get();
        return view('dashboard.fees.onabashik_fees_assign_create_month', compact('data', 'fees', 'class', 'session', 'allMonths'));
    }

    public function SpecialFeesAssignStore(Request $request)
    {
        $messages = [
            'months.required'   => 'Please select at least one month',
            'months.*.required' => 'Invalid month format',
        ];

        $request->validate([
            'months'   => 'required|array',
            'months.*' => 'required|date_format:Y-m',
        ], $messages);

        DB::beginTransaction();

        try {
            // dd($request->all());
            $existingFeeAssign = FeeAssign::where('session_id', $request->session_id)
                ->where('class_id', $request->class_id)
                ->where('type', 5)
                ->first();

            // Check for existing assignments for any of the selected months
            if ($existingFeeAssign) {
                $existingAssignments = FeeAssignDetail::with('fees')
                    ->where('fee_assign_id', $existingFeeAssign->id)
                    ->whereIn('month', $request->months)
                    ->where('fee_id', $request->special_fees_id)
                    ->where('type', 5)
                    ->get();

                if ($existingAssignments->isNotEmpty()) {
                    $conflictingMonths = $existingAssignments->pluck('month')->map(function ($month) {
                        return \Carbon\Carbon::parse($month)->format('F Y');
                    })->implode(', ');

                    return redirect()->back()
                        ->withInput()
                        ->with('error', "For this class & session, the same fees are already assigned for: $conflictingMonths");
                }
            }

            $assign = FeeAssign::create([
                'session_id' => $request->session_id,
                'class_id'   => $request->class_id,
                'type'       => 5,
                'created_by' => Auth::id(),
            ]);

            foreach ($request->student_id as $key => $studentId) {
                foreach ($request->months as $month) {
                    // Format month to ensure consistency
                    $formattedMonth = \Carbon\Carbon::parse($month)->format('Y-m');

                    $assignDetails = FeeAssignDetail::firstOrCreate([
                        'fee_assign_id' => $assign->id,
                        'fee_id'        => $request->special_fees_id,
                        'month'         => $formattedMonth,
                        'type'          => 5,
                    ], [
                        'session_id' => $assign->session_id,
                        'class_id'   => $assign->class_id,
                        'amount'     => $request->amount[$key],
                        'created_by' => Auth::id(),
                    ]);

                    FeeAssignDetailStudent::create([
                        'fee_assign_id'         => $assign->id,
                        'fee_assign_details_id' => $assignDetails->id,
                        'session_id'            => $request->session_id,
                        'class_id'              => $request->class_id,
                        'student_id'            => $studentId,
                        'month'                 => $formattedMonth,
                        'amount'                => $request->amount[$key],
                        'created_by'            => Auth::id(),
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('special.fees.assign')
                ->with('success', 'Fees assigned successfully for selected months');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'Error: ' . $exception->getMessage());
        }
    }
    public function SpecialFeesAssignShow(Request $request, $class_id, $session_id)
    {
        try {
            if ($request->ajax()) {

                $data = DB::table('fee_assigns')
                    ->join('student_classes', 'student_classes.id', '=', 'fee_assigns.class_id')
                    ->join('student_sessions', 'student_sessions.id', '=', 'fee_assigns.session_id')
                    ->join('fee_assign_details', 'fee_assign_details.fee_assign_id', '=', 'fee_assigns.id')
                    ->join('student_fees', 'student_fees.id', '=', 'fee_assign_details.fee_id')
                    ->where('fee_assigns.type', 5)
                    ->whereNull('fee_assigns.deleted_at')
                    ->whereNull('fee_assign_details.deleted_at')
                    ->whereNull('student_fees.deleted_at')
                    ->whereNull('student_classes.deleted_at')
                    ->whereNull('student_sessions.deleted_at')

                    ->where('fee_assigns.class_id', $class_id)
                    ->where('fee_assigns.session_id', $session_id)
                    ->select(
                        'fee_assigns.*',
                        'student_classes.name as class',
                        'student_sessions.name as session',
                        'student_fees.name as feesDetails',
                        'fee_assign_details.month as month',
                    )
                    ->groupBy('fee_assigns.id')
                    ->get();
                // dd($data);

                return Datatables::of($data)
                    ->addColumn('class', function ($data) {
                        return $data->class ?? '--';
                    })
                    ->addColumn('session', function ($data) {
                        return $data->session ?? '--';
                    })
                    ->addColumn('fees', function ($data) {
                        $months = DB::table('fee_assign_details')
                            ->where('fee_assign_id', $data->id)
                            ->where('type', 5)
                            ->whereNull('deleted_at')
                            ->pluck('month')
                            ->map(function ($month) {
                                return \Carbon\Carbon::parse($month)->format('F Y');
                            })
                            ->implode(', ');

                        // dd($data, $months);
                        $fee_details = $data->feesDetails . ' (' . $months . ')';
                        return $fee_details ?? '--';

                    })
                    ->addColumn('action', function ($data) {

                        $edit = '<a id="details" href="' . route('special.fees.assign.edit', [
                            'class_id'   => $data->class_id,
                            'session_id' => $data->session_id,
                            'id'         => $data->id,
                        ]) . '" class="btn btn-sm btn-info ml-1" title="Edit">
                        <i class="fa fa-edit"></i></a>';

                        $delete = '';
                        if (method_exists($this, 'DeleteData') && $this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete"
                            data-remote="' . route('special.fees.assign.destroy', $data->id) . '"
                            title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }

                        return $edit . ' ' . $delete;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['class', 'session', 'fees', 'action'])
                    ->toJson();
            }

            return view('dashboard.fees.special_fees_assign_show', compact('class_id', 'session_id'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function SpecialFeesAssignEdit($class_id, $session_id, $id)
    {
        $feeAssign = FeeAssign::with('class:id,name', 'studentsession:id,name')
            ->where('type', 5)
            ->where('class_id', $class_id)
            ->where('session_id', $session_id)
            ->find($id);

        // $feeAssignIds = $feeAssign->pluck('id');

        $details = FeeAssignDetail::with('fees')
            ->where('type', 5)
            ->where('fee_assign_id', $feeAssign->id)
            ->get()
            ->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->month)->format('Y-m');
            });

        $selectedFeeId = DB::table('fee_assign_details')
            ->where('fee_assign_id', $feeAssign->id)
            ->where('type', 5)
            ->whereNull('deleted_at')
            ->value('fee_id');

        $selectedMonths = DB::table('fee_assign_details')
            ->where('fee_assign_id', $feeAssign->id)
            ->where('type', 5)
            ->whereNull('deleted_at')
            ->pluck('month')
            ->toArray();

        //  dd($feeAssign, $class_id, $session_id,$details);

        $detailsStudent = FeeAssignDetailStudent::with('student:id,name,register_no,roll_no')
            ->where('fee_assign_id', $feeAssign->id)
            ->get();

        // dd($feeAssign, $details, $detailsStudent);

        $class             = StudentClass::all();
        $sessions          = StudentSession::all();
        $specialFees       = StudentFee::where('type', 5)->whereNull('deleted_at')->get();
        $lastAssignedMonth = FeeAssignDetail::orderBy('month', 'desc')->where('type', 5)->value('month');
        // dd($specialFees);

        return view('dashboard.fees.special_fees_assign_edit', compact('feeAssign', 'details', 'detailsStudent', 'class', 'sessions', 'specialFees', 'selectedFeeId', 'lastAssignedMonth', 'selectedMonths'));
    }

    public function SpecialFeesAssignUpdate(Request $request, $id)
    {

        // $messages = [
        //     'months.required'   => 'Please select at least one month',
        //     'months.*.required' => 'Invalid month format',
        // ];

        // $request->validate([
        //     'months'   => 'required|array',
        //     'months.*' => 'required|date_format:Y-m',
        // ], $messages);

        DB::beginTransaction();

        try {
            $assign = FeeAssign::findOrFail($id);
            // dd($request->all(), $assign);

            $assign->update([
                'session_id' => $request->session_id,
                'class_id'   => $request->class_id,
                'type'       => 5,
                'created_by' => Auth::id(),
            ]);

            // dd($existingAssignments, $request->all(), $id);

            // Delete previous related FeeAssignDetailStudents for this assign, fee_id and months
            FeeAssignDetailStudent::where('fee_assign_id', $assign->id)
            // ->where('class_id', $request->class_id)
            // ->where('session_id', $request->session_id)
            // ->whereIn('month', $request->months)
                ->delete();

            // Delete previous FeeAssignDetail for this assign, fee_id and months
            FeeAssignDetail::where('fee_assign_id', $assign->id)
                ->where('type', 5)
            // ->whereIn('month', $request->months)
                ->delete();

            foreach ($request->student_id as $key => $studentId) {
                foreach ($request->months as $month) {
                    $formattedMonth = \Carbon\Carbon::parse($month)->format('Y-m');

                    $assignDetails = FeeAssignDetail::firstOrCreate([
                        'fee_assign_id' => $assign->id,
                        'fee_id'        => $request->special_fees_id,
                        'month'         => $formattedMonth,
                        'type'          => 5,
                    ], [
                        'session_id' => $assign->session_id,
                        'class_id'   => $assign->class_id,
                        'amount'     => $request->amount[$key],
                        'created_by' => Auth::id(),
                    ]);

                    FeeAssignDetailStudent::create([
                        'fee_assign_id'         => $assign->id,
                        'fee_assign_details_id' => $assignDetails->id,
                        'session_id'            => $assign->session_id,
                        'class_id'              => $assign->class_id,
                        'student_id'            => $studentId,
                        'month'                 => $formattedMonth,
                        'amount'                => $request->amount[$key],
                        'created_by'            => Auth::id(),
                    ]);
                }
            }

            DB::commit();

            return redirect()->route('special.fees.assign.show', ['class_id' => $assign->class_id, 'session_id' => $assign->session_id])
                ->with('success', 'Fees assignment updated successfully for selected months');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()
                ->withInput()
                ->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function SpecialFeesAssignDestroy($id)
    {
        try {
            $data = FeeAssign::findOrFail($id);
            // dd($data, $data->id);
            FeeAssignDetailStudent::where('fee_assign_id', $data->id)->delete();
            FeeAssignDetail::where('fee_assign_id', $data->id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }

    public function getValidateStudent(Request $request)
    {
        $sessionId = $request->session_id;
        $classId   = $request->class_id;
        $month     = $request->month;
        $feeId     = $request->special_fees_id;
        // dd($request->all());

        $feeAssignDetails = FeeAssignDetail::where('session_id', $sessionId)
            ->where('class_id', $classId)
            ->where('fee_id', $feeId)
            ->whereIn('month', $month)
            ->where('type', 5)
            ->first();

        if ($feeAssignDetails) {
            return response()->json(['status' => true]);
        } else {
            return response()->json(['status' => false]);
        }
    }

    public function getValidateFees(Request $request)
    {
        $sessionId = $request->session_id;
        $classId   = $request->class_id;
        $feeId     = $request->special_fees_id;

        $feeAssignDetails = FeeAssignDetail::where('session_id', $sessionId)
            ->where('class_id', $classId)
            ->where('fee_id', $feeId)
            ->where('type', 5)
            ->whereNull('deleted_at')
            ->first();

        if ($feeAssignDetails) {
            return response()->json(['status' => true]);
        } else {
            return response()->json(['status' => false]);
        }
    }

    //special fee end

    public function OtherFees(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = DB::table('student_fees')
                    ->where('type', 3)
                    ->whereNull('deleted_at')
                    ->orderBy('id', 'desc')->get();

                return Datatables::of($data)

                    ->addColumn('action', function ($data) {

                        $edit = '<a id="edit" href="' . route('other.fees.edit', $data->id) . ' " class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editClass"><i class="fa fa-edit"></i></a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('other.fees.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }

                        return $edit . ' ' . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action'])
                    ->toJson();
            }
            return view('dashboard.fees.other_fees');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function OtherFeesStore(Request $request)
    {
        if ($request->ajax()) {

            $data = Validator::make($request->all(), [
                'name'   => 'required|string|unique:student_fees,name,NULL,id,deleted_at,NULL',
                'amount' => 'required|numeric',

            ]);

            if ($data->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $data->errors()->all(),
                ]);
            }

            try {
                $data             = new StudentFee();
                $data->name       = $request->name;
                $data->amount     = $request->amount;
                $data->type       = 3;
                $data->created_by = Auth::user()->id;
                $data->save();

                return response()->json([
                    'success' => true,
                    'message' => 'Fee created successfully',
                ]);

            } catch (\Exception $exception) {
                return response()->json([
                    'success' => true,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }
    public function OtherFeesEdit($id)
    {
        $data = StudentFee::findOrFail($id);
        return response()->json($data);
    }

    public function OtherFeesUpdate(Request $request, $id)
    {
        $data = Validator::make($request->all(), [
            'name'   => 'required|string|unique:student_fees,name,' . $id . ',id,deleted_at,NULL',
            'amount' => 'required|numeric',
        ]);

        if ($data->fails()) {
            return response()->json([
                'success' => false,
                'message' => $data->errors()->all(),
            ]);
        }

        try {
            $data             = StudentFee::findOrFail($id);
            $data->name       = $request->name;
            $data->amount     = $request->amount;
            $data->type       = 3;
            $data->created_by = Auth::user()->id;
            $data->update();

            return response()->json([
                'success' => true,
                'message' => 'Fees updated successfully',
            ]);

        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }
    public function OtherFeesDestroy($id)
    {
        try {
            $data   = StudentFee::findOrFail($id);
            $tables = DB::select("
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE COLUMN_NAME = 'fee_id'
            AND TABLE_SCHEMA = DATABASE()
            ");

            foreach ($tables as $table) {
                if ($table->TABLE_NAME !== 'student_fees') {
                    DB::table($table->TABLE_NAME)->where('fee_id', $data->id)->delete();
                }
            }
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Fees deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Fees delete failed',
            ]);
        }
    }

    public function AcademicFeesAssign(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = FeeAssign::with('class:id,name', 'studentsession:id,name')->where('type', 1)
                    ->groupBy('class_id')->groupBy('session_id')->get();

                return Datatables::of($data)

                    ->addColumn('class', function ($data) {
                        $department = $data->class->name ?? '--';
                        return $department;
                    })

                    ->addColumn('session', function ($data) {
                        $session = $data->studentsession->name ?? '--';
                        return $session;
                    })

                    ->addColumn('action', function ($data) {
                        $edit = '<a id="details" href="' . route('academic.fees.assign.edit', ['class_id' => $data->class_id, 'session_id' => $data->session_id]) . '" class="btn btn-sm btn-info ml-1" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="details" href="' . route('academic.fees.assign.show', ['class_id' => $data->class_id, 'session_id' => $data->session_id]) . '" class="btn btn-sm btn-primary show ml-1" title="Details"><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {

                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('academic.fees.assign.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }

                        return $edit . $show . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['session', 'class', 'action'])
                    ->toJson();
            }
            return view('dashboard.fees.academic_fees_assign');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function AcademicFeesAssignCreate()
    {
        $fees              = StudentFee::where('type', 1)->get();
        $class             = StudentClass::all();
        $sessions          = StudentSession::all();
        $lastAssignedMonth = FeeAssignDetail::orderBy('month', 'desc')->where('type', 1)->value('month');
        return view('dashboard.fees.academic_fees_assign_create', compact('fees', 'class', 'sessions', 'lastAssignedMonth'));
    }

    public function AcademicFeesMonth(Request $request)
    {
        $validatedData = $request->validate([
            'type'       => 'required',
            'session_id' => 'required',
            'class_id'   => 'required',
            'month_for'  => 'required',
            'from_month' => 'required',
            'fee_name'   => 'array',
            'fee_name.*' => 'required',
        ]);

        $fromMonth = \DateTime::createFromFormat('F-Y', $validatedData['from_month']);
        if (! $fromMonth) {
            return back()->with('error', 'Invalid month format');
        }

        $data = [
            'type'          => $validatedData['type'],
            'session_id'    => $validatedData['session_id'],
            'class_id'      => $validatedData['class_id'],
            'month_for'     => $validatedData['month_for'],
            'from_month'    => $fromMonth->format('Y-m'),
            'selected_fees' => $validatedData['fee_name'] ?? [],
        ];

        $startMonth = Carbon::createFromFormat('Y-m', $data['from_month']);
        $allMonths  = [];

        for ($i = 0; $i < $data['month_for']; $i++) {
            $month       = $startMonth->copy()->addMonthsNoOverflow($i)->format('Y-m');
            $allMonths[] = $month;
        }

        $class   = StudentClass::findOrFail($data['class_id']);
        $session = StudentSession::findOrFail($data['session_id']);
        $fees    = StudentFee::whereIn('id', $data['selected_fees'])->get();
        return view('dashboard.fees.academic_fees_assign_create_month', compact('data', 'fees', 'class', 'session'));
    }

    public function AcademicFeesAssignStore(Request $request)
    {
        $messages = [
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'amount' => 'required|array',
        ], $messages);

        foreach ($request->month as $monthIndex => $month) {
            foreach ($request->fee_id[$monthIndex] as $key => $feeId) {
                $existingFeeAssign = FeeAssign::where('session_id', $request->session_id)
                    ->where('class_id', $request->class_id)
                    ->where('type', 1)
                    ->first();

                if ($existingFeeAssign) {
                    $existingAssignment = FeeAssignDetail::with('fees')->where('fee_assign_id', $existingFeeAssign->id)
                        ->where('month', \Carbon\Carbon::parse($month)->format('Y-m'))
                        ->where('fee_id', $feeId)
                        ->where('type', 1)
                        ->first();

                    if ($existingAssignment) {
                        return redirect()->back()->with('error', 'For this class & session same fees already assigned');
                    }
                }
            }
        }

        DB::beginTransaction();

        try {

            $assign             = new FeeAssign();
            $assign->session_id = $request->session_id;
            $assign->class_id   = $request->class_id;
            $assign->type       = 1;
            $assign->created_by = Auth::user()->id;
            $assign->save();

            foreach ($request->month as $monthIndex => $month) {
                foreach ($request->fee_id[$monthIndex] as $key => $value) {
                    $assignDetails                = new FeeAssignDetail();
                    $assignDetails->type          = $assign->type;
                    $assignDetails->session_id    = $assign->session_id;
                    $assignDetails->class_id      = $assign->class_id;
                    $assignDetails->fee_assign_id = $assign->id;
                    $assignDetails->fee_id        = $value;
                    $assignDetails->month         = \Carbon\Carbon::parse($month)->format('Y-m');
                    $assignDetails->amount        = $request->amount[$monthIndex][$key];
                    $assignDetails->created_by    = Auth::user()->id;
                    $assignDetails->save();
                }
            }

            DB::commit();

            return redirect()->route('academic.fees.assign')->with('success', 'Fees assigned successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function AcademicFeesAssignShow($class_id, $session_id)
    {
        $data = FeeAssign::with('class:id,name', 'studentsession:id,name')
            ->where('type', 1)
            ->where('class_id', $class_id)
            ->where('session_id', $session_id)
            ->get();

        $feeAssignIds = $data->pluck('id');

        $details = FeeAssignDetail::with('fees')
            ->where('type', 1)
            ->whereIn('fee_assign_id', $feeAssignIds)
            ->get()
            ->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->month)->format('Y-m');
            });

        return view('dashboard.fees.academic_fees_assign_show', compact('data', 'details'));
    }

    public function AcademicFeesAssignEdit($class_id, $session_id)
    {
        $data = FeeAssign::with('class:id,name', 'studentsession:id,name')
            ->where('type', 1)
            ->where('class_id', $class_id)
            ->where('session_id', $session_id)
            ->get();

        $feeAssignIds = $data->pluck('id');

        $details = FeeAssignDetail::with('fees')
            ->where('type', 1)
            ->whereIn('fee_assign_id', $feeAssignIds)
            ->get()
            ->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->month)->format('Y-m');
            });

        return view('dashboard.fees.academic_fees_assign_edit', compact('data', 'details'));
    }

    public function AcademicFeesAssignUpdate(Request $request, $class_id, $session_id)
    {
        $messages = [
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'amount'   => 'required|array',
            'amount.*' => 'required|numeric',
        ], $messages);

        try {
            foreach ($request->id as $key => $id) {
                $assignDetails = FeeAssignDetail::where('class_id', $class_id)
                    ->where('session_id', $session_id)
                    ->where('type', 1)
                    ->findOrFail($id);

                $assignDetails->amount     = $request->amount[$key];
                $assignDetails->session_id = $request->session_id;
                $assignDetails->class_id   = $request->class_id;
                $assignDetails->created_by = Auth::user()->id;
                $assignDetails->save();
            }

            return redirect()->route('academic.fees.assign')->with('success', 'Fees assigned updated successfully');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function HostelFeesAssign(Request $request)
    {
        try {
            if ($request->ajax()) {

                $data = FeeAssign::with('class:id,name', 'studentsession:id,name')->whereNull('deleted_at')->where('type', '!=', '5')->groupBy('type')->get();

                return Datatables::of($data)

                    ->addColumn('type', function ($data) {
                        $type = '';
                        if ($data->type == 1) {
                            $type = 'একাডেমিক';
                        } elseif ($data->type == 2) {
                            $type = 'আবাসিক';
                        } elseif ($data->type == 3) {
                            $type = 'পরীক্ষা';
                        } elseif ($data->type == 4) {
                            $type = 'অনাবাসিক';
                        }
                        return $type;
                    })

                    ->addColumn('action', function ($data) {

                        $show = '<a id="details" href="' . route('hostel.fees.assign.showByType', ['type' => $data->type]) . '" class="btn btn-sm btn-primary show ml-1" title="Details"><i class="fa fa-eye"></i></a>';
                        return $show;

                    })

                    ->addIndexColumn()
                    ->rawColumns(['session', 'class', 'action', 'type'])
                    ->toJson();
            }
            return view('dashboard.fees.hostel_fees_assign');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function HostelFeesAssignShowByType(Request $request, $type)
    {
        try {
            if ($request->ajax()) {

                $data = FeeAssign::with('class:id,name', 'studentsession:id,name')->where('type', $type)->get();

                return Datatables::of($data)

                    ->addColumn('class', function ($data) {
                        $department = $data->class->name ?? '--';
                        return $department;
                    })

                    ->addColumn('session', function ($data) {
                        $session = $data->studentsession->name ?? '--';
                        return $session;
                    })
                    ->addColumn('details', function ($data) {
                        // Fetch all fee details for this assign
                        $assignDetails = DB::table('fee_assign_details')
                            ->join('student_fees', 'fee_assign_details.fee_id', '=', 'student_fees.id')
                            ->where('fee_assign_id', $data->id)
                            ->where('fee_assign_details.type', $data->type)
                            ->where('class_id', $data->class_id)
                            ->where('session_id', $data->session_id)
                            ->whereNull('fee_assign_details.deleted_at')
                            ->select('fee_assign_details.month', 'student_fees.name as fee_name')
                            ->get()
                            ->groupBy('month'); // Now group by month

                        $result = '';
                        foreach ($assignDetails as $month => $fees) {
                            $result .= '<strong>' . \Carbon\Carbon::parse($month . '-01')->format('F Y') . ':</strong> ';
                            $result .= $fees->pluck('fee_name')->join(', ');
                            $result .= '<br>';
                        }

                        // dd($result);
                        return $result ?? 'N/A';
                    })

                    ->addColumn('action', function ($data) {

                        $edit = '<a id="details" href="' . route('hostel.fees.assign.edit', ['class_id' => $data->class_id, 'session_id' => $data->session_id, 'type' => $data->type, 'id' => $data->id]) . '" class="btn btn-sm btn-info ml-1" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="details" href="' . route('hostel.fees.assign.show', ['class_id' => $data->class_id, 'session_id' => $data->session_id, 'type' => $data->type, 'id' => $data->id]) . '" class="btn btn-sm btn-primary show ml-1" title="Details"><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {

                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('hostel.fees.assign.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return $edit . $show . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['session', 'class', 'details', 'action'])
                    ->toJson();
            }
            return view('dashboard.fees.hostel_fees_assign_show_by_type', compact('type'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function HostelFeesAssign_old(Request $request)
    {
        try {
            if ($request->ajax()) {

                $data = FeeAssign::with('class:id,name', 'studentsession:id,name')->where('type', 2)->groupBy('class_id')->groupBy('session_id')->get();

                return Datatables::of($data)

                    ->addColumn('class', function ($data) {
                        $department = $data->class->name ?? '--';
                        return $department;
                    })

                    ->addColumn('session', function ($data) {
                        $session = $data->studentsession->name ?? '--';
                        return $session;
                    })

                    ->addColumn('action', function ($data) {

                        $edit = '<a id="details" href="' . route('hostel.fees.assign.edit', ['class_id' => $data->class_id, 'session_id' => $data->session_id]) . '" class="btn btn-sm btn-info ml-1" title="Edit"><i class="fa fa-edit"></i></a>';

                        $show = '<a id="details" href="' . route('hostel.fees.assign.show', ['class_id' => $data->class_id, 'session_id' => $data->session_id]) . '" class="btn btn-sm btn-primary show ml-1" title="Details"><i class="fa fa-eye"></i></a>';

                        $delete = '';
                        if ($this->DeleteData()) {

                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('hostel.fees.assign.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        }
                        return $edit . $show . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['session', 'class', 'action'])
                    ->toJson();
            }
            return view('dashboard.fees.hostel_fees_assign');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function HostelFeesAssignCreate()
    {
        $fees              = StudentFee::where('type', 2)->get();
        $class             = StudentClass::all();
        $sessions          = StudentSession::all();
        $lastAssignedMonth = FeeAssignDetail::orderBy('month', 'desc')->where('type', 2)->value('month');
        return view('dashboard.fees.hostel_fees_assign_create', compact('fees', 'class', 'sessions', 'lastAssignedMonth'));
    }

    public function LoadFeesByType(Request $request)
    {
        // dd($request->all());
        $fees = StudentFee::where('type', $request->fee_type)->whereNull('deleted_at')->get();
        return response()->json($fees);
    }

    public function HostelFeesMonth(Request $request)
    {
        $validatedData = $request->validate([
            'type'       => 'required',
            'session_id' => 'required',
            'class_id'   => 'required',
            'month_for'  => 'required',
            'from_month' => 'required',
            'fee_name'   => 'array',
            'fee_name.*' => 'required',
        ]);

        $fromMonth = \DateTime::createFromFormat('F-Y', $validatedData['from_month']);
        if (! $fromMonth) {
            return back()->with('error', 'Invalid month format');
        }

        $data = [
            'type'          => $validatedData['type'],
            'session_id'    => $validatedData['session_id'],
            'class_id'      => $validatedData['class_id'],
            'month_for'     => $validatedData['month_for'],
            'from_month'    => $fromMonth->format('Y-m'),
            'selected_fees' => $validatedData['fee_name'] ?? [],
        ];

        $startMonth = Carbon::createFromFormat('Y-m', $data['from_month']);
        $allMonths  = [];

        for ($i = 0; $i < $data['month_for']; $i++) {
            $month       = $startMonth->copy()->addMonthsNoOverflow($i)->format('Y-m');
            $allMonths[] = $month;
        }

        $class   = StudentClass::findOrFail($data['class_id']);
        $session = StudentSession::findOrFail($data['session_id']);
        $fees    = StudentFee::whereIn('id', $data['selected_fees'])->get();
        return view('dashboard.fees.hostel_fees_assign_create_month', compact('data', 'fees', 'class', 'session', 'allMonths'));
    }

    public function HostelFeesAssignStore(Request $request)
    {
        $messages = [
            'amount.required' => 'Enter amount',
            'fee_id'          => 'Select fee',
        ];

        $request->validate([
            'amount' => 'required|array',
            'fee_id' => 'required|array',
        ], $messages);

        foreach ($request->months as $monthIndex => $month) {
            // dd($request->fee_id[$monthIndex]);
            foreach ($request->fee_id as $feeId) {
                // $existingFeeAssign = FeeAssign::where('session_id', $request->session_id)
                //     ->where('class_id', $request->class_id)
                //     ->where('type', $request->fee_type)
                //     ->first();

                // if ($existingFeeAssign) {
                $existingAssignment = FeeAssignDetail::with('fees')
                    ->where('month', \Carbon\Carbon::createFromFormat('Y-m', $month)->format('Y-m'))
                    ->where('session_id', $request->session_id)
                    ->where('class_id', $request->class_id)
                    ->where('type', $request->fee_type)
                    ->where('fee_id', $feeId)
                    ->first();

                if ($existingAssignment) {
                    return redirect()->back()->with('error', 'same fees already assigned');
                }
                // }
            }
        }

        DB::beginTransaction();

        try {

            $assign             = new FeeAssign();
            $assign->session_id = $request->session_id;
            $assign->class_id   = $request->class_id;
            $assign->type       = $request->fee_type;
            $assign->created_by = Auth::user()->id;
            $assign->save();

            foreach ($request->months as $monthIndex => $month) {
                foreach ($request->fee_id as $value) {
                    $assignDetails                = new FeeAssignDetail();
                    $assignDetails->type          = $request->fee_type;
                    $assignDetails->fee_assign_id = $assign->id;
                    $assignDetails->fee_id        = $value;
                    $assignDetails->month         = $month;
                    $assignDetails->session_id    = $assign->session_id;
                    $assignDetails->class_id      = $assign->class_id;
                    $assignDetails->amount        = $request->amount[$value];
                    $assignDetails->created_by    = Auth::user()->id;
                    $assignDetails->save();
                }
            }

            DB::commit();

            return redirect()->route('hostel.fees.assign.showByType', ['type' => $request->fee_type])->with('success', 'Fees assigned successfully');

        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function HostelFeesAssignShow($class_id, $session_id, $type, $id)
    {
        $data = FeeAssign::with('class:id,name', 'studentsession:id,name')
            ->where('type', $type)
            ->where('class_id', $class_id)
            ->where('session_id', $session_id)
            ->find($id);
        // dd($data);

        // $feeAssignIds = $data->pluck('id');

        $details = FeeAssignDetail::with('fees')
            ->where('type', $type)
            ->where('fee_assign_id', $data->id)
            ->get()
            ->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->month)->format('Y-m');
            });
        return view('dashboard.fees.hostel_fees_assign_show', compact('data', 'details'));
    }

    public function HostelFeesAssignEdit($class_id, $session_id, $type, $id)
    {

        $data = FeeAssign::where('type', $type)
            ->where('class_id', $class_id)
            ->where('session_id', $session_id)
            ->find($id);
        // dd($data,$class_id);

        if (! $data) {
            abort(404, 'Fee Assignment not found.');
        }

        $details = FeeAssignDetail::with('fees')
            ->where('type', $type)
            ->where('fee_assign_id', $id)
            ->get()
            ->groupBy(function ($item) {
                return Carbon::parse($item->month)->format('Y-m');
            });

        // dd($details);

        $classes  = StudentClass::all();
        $sessions = StudentSession::all();
        $type     = $type;

        $selectedFees = DB::table('student_fees')
            ->join('fee_assign_details', 'student_fees.id', '=', 'fee_assign_details.fee_id')
            ->where('fee_assign_details.type', $type)
            ->where('fee_assign_details.class_id', $class_id)
            ->where('fee_assign_details.session_id', $session_id)
            ->where('fee_assign_details.fee_assign_id', $id)
            ->whereNull('student_fees.deleted_at')
            ->whereNull('fee_assign_details.deleted_at')
            ->select('student_fees.id', 'student_fees.name', 'fee_assign_details.amount', 'fee_assign_details.id as detail_id', 'fee_assign_details.month')
            ->get();
        // dd($details, $selectedFees, $data);

        return view('dashboard.fees.hostel_fees_assign_edit', compact('data', 'details', 'classes', 'sessions', 'type', 'selectedFees'));
    }

    public function HostelFeesAssignUpdate(Request $request, $class_id, $session_id, $type, $id)
    {
        // dd($request->all());
        // $messages = [
        //     'amount.required' => 'Enter amount',
        // ];

        // $request->validate([
        //     'amount'   => 'required|array',
        //     'amount.*' => 'required|numeric',
        // ], $messages);

        try {
            DB::beginTransaction();

            $feeAssign = FeeAssign::where('class_id', $class_id)
                ->where('session_id', $session_id)
                ->where('type', $type)
                ->findOrFail($id);

            $feeAssignDetailsIds = DB::table('fee_assign_details')
                ->where('fee_assign_id', $feeAssign->id)
                ->whereNull('deleted_at')
                ->pluck('id')->toArray();
            $requestedDetailsIds = $request->details_id ?? [];
            // dd($request->all(), $feeAssignDetailsIds, $requestedDetailsIds, $feeAssign->id);
            $requestedDetailsIds = array_map('intval', $requestedDetailsIds);

            // Delete missing IDs
            $toDelete = array_diff($feeAssignDetailsIds, $requestedDetailsIds);
            // dd($request->all(), $feeAssign, $feeAssignDetailsIds, $requestedDetailsIds, $toDelete);
            if (! empty($toDelete)) {
                FeeAssignDetail::whereIn('id', $toDelete)->delete();
            }

            // Update existing
            if (! empty($requestedDetailsIds)) {
                foreach ($requestedDetailsIds as $index => $detailId) {
                    FeeAssignDetail::where('id', $detailId)->update([
                        'amount'     => $request->amount[$detailId],
                        'session_id' => $request->session_id,
                        'class_id'   => $request->class_id,
                        'created_by' => Auth::user()->id,
                    ]);
                }
            }

            // Insert new
            if (! empty($request->fee_id_new)) {

                // dd($request->all(),$request->fee_id_new);

                foreach ($request->fee_id_new as $feeMonth => $feeMonthData) {
                    foreach ($feeMonthData as $key => $feeId) {
                        $existingAssignment = FeeAssignDetail::where('month', $feeMonth)
                            ->where('session_id', $request->session_id)
                            ->where('class_id', $request->class_id)
                            ->where('type', $request->fee_type)
                            ->where('fee_id', $feeId)
                            ->first();
                        if ($existingAssignment) {

                            return redirect()->back()->with('error', 'Your custom error message')->withInput();
                            // return back()->with('error', 'Same fees already assigned');
                        }
                        // Ensure amount_new[feeId] exists
                        if (isset($request->amount_new[$feeMonth][$feeId])) {

                            FeeAssignDetail::create([
                                'type'          => $request->fee_type,
                                'fee_assign_id' => $feeAssign->id,
                                'fee_id'        => $feeId,
                                'month'         => $feeMonth,
                                'session_id'    => $feeAssign->session_id,
                                'class_id'      => $feeAssign->class_id,
                                'amount'        => $request->amount_new[$feeMonth][$feeId],
                                'created_by'    => Auth::user()->id,
                            ]);
                        }
                    }
                    // dd($month, $feeMonth, $feeMonthData, $request->all());

                }

            }

            DB::commit();

            return redirect()->route('hostel.fees.assign.showByType', ['type' => $type])->with('success')->with('success', 'Fees assigned updated successfully');

        } catch (\Exception $e) {
            // dd($e);
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $e->getMessage());
        }
    }

    public function checkFeesAvailability(Request $request)
    {
        $month      = $request->month;
        $session_id = $request->session_id;
        $class_id   = $request->class_id;
        $feeType    = $request->fee_type;
        $feeId      = $request->fee_id;
        // dd($request->all(), $month, $session_id, $class_id, $feeType, $feeId);
        $existingAssignment = FeeAssignDetail::where('month', $month)
            ->where('session_id', $session_id)
            ->where('class_id', $class_id)
            ->where('type', $feeType)
            ->where('fee_id', $feeId)
            ->first();
        if ($existingAssignment) {
            return response()->json(['status' => 'error', 'message' => 'Same fees already assigned']);
        } else {
            return response()->json(['status' => 'success']);
        }

        // dd($existingAssignment);
    }

    public function HostelFeesAssignUpdate_old(Request $request, $class_id, $session_id)
    {
        $messages = [
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'amount'   => 'required|array',
            'amount.*' => 'required|numeric',
        ], $messages);

        try {

            foreach ($request->id as $key => $id) {
                $assignDetails = FeeAssignDetail::where('class_id', $class_id)
                    ->where('session_id', $session_id)
                    ->where('type', 2)
                    ->findOrFail($id);

                $assignDetails->amount     = $request->amount[$key];
                $assignDetails->session_id = $request->session_id;
                $assignDetails->class_id   = $request->class_id;
                $assignDetails->created_by = Auth::user()->id;
                $assignDetails->save();
            }

            return redirect()->route('hostel.fees.assign')->with('success', 'Fees assigned updated successfully');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function OtherFeesAssign(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = FeeAssign::with('class:id,name')->where('type', 3)->get();
                // dd($data);

                return Datatables::of($data)

                    ->addColumn('class', function ($data) {
                        $department = $data->class->name ?? '--';
                        return $department;
                    })

                    ->addColumn('month', function ($data) {
                        $monthData = DB::table('fee_assign_details')->where('fee_assign_id', $data->id)->first();
                        return $monthData && $monthData->month ? \Carbon\Carbon::parse($monthData->month)->format('F, Y') : 'N/A';
                    })

                    ->addColumn('fee', function ($data) {
                        $fee = FeeAssignDetail::with('fees')->where('fee_assign_id', $data->id)->first();
                        return $fee->fees->name ?? '';
                    })

                    ->addColumn('amount', function ($data) {
                        $amount = DB::table('fee_assign_details')->where('fee_assign_id', $data->id)->first();
                        return $amount->amount ?? '';
                    })
                    ->addColumn("session", function ($data) {
                        $session = StudentSession::find($data->session_id);
                        return $session->name ?? 'N/A';
                    })

                    ->addColumn('action', function ($data) {
                        $edit = '<a id="edit" href="' . route('other.fees.assign.edit', $data->id) . ' " class="btn btn-sm btn-primary edit" title="Edit" data-toggle="modal" data-target="#editClass"><i class="fa fa-edit"></i></a> ';

                        $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete ml-1" data-remote=" ' . route('other.fees.assign.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                        return $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['month', 'class', 'action', 'fee', 'amount', 'session'])
                    ->toJson();
            }

            $fees     = StudentFee::where('type', 3)->get();
            $class    = StudentClass::all();
            $sessions = StudentSession::all();

            return view('dashboard.fees.other_fees_assign', compact('class', 'sessions', 'fees'));

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function OtherFeesAssignStore(Request $request)
    {
        if ($request->ajax()) {
            $data = Validator::make($request->all(), [
                'amount' => 'required',
            ]);

            if ($data->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => $data->errors()->all(),
                ]);
            }

            DB::beginTransaction();

            try {
                $existingFeeAssign = FeeAssign::where('session_id', $request->session_id)
                    ->where('class_id', $request->class_id)
                    ->where('type', 3)
                    ->whereHas('feesDetails', function ($query) use ($request) {
                        $query->where('fee_id', $request->fee_id);
                        $query->where('month', Carbon::parse($request->month)->format('Y-m'));
                    })
                    ->first();
                // dd($existingFeeAssign);

                if ($existingFeeAssign) {
                    $existingAssignment = FeeAssignDetail::with('fees')->where('fee_assign_id', $existingFeeAssign->id)
                        ->where('month', Carbon::parse($request->month)->format('Y-m'))
                        ->where('fee_id', $request->fee_id)
                        ->where('type', 3)
                        ->first();

                    // dd($existingAssignment);
                    return response()->json([
                        'success' => false,
                        'message' => 'Same fee already assigned',
                    ]);
                }

                $assign             = new FeeAssign();
                $assign->class_id   = $request->class_id;
                $assign->type       = $request->type;
                $assign->session_id = $request->session_id;
                $assign->created_by = Auth::user()->id;
                $assign->save();

                $assignDetails                = new FeeAssignDetail();
                $assignDetails->type          = $assign->type;
                $assignDetails->fee_assign_id = $assign->id;
                $assignDetails->fee_id        = $request->fee_id;
                $assignDetails->session_id    = $assign->session_id;
                $assignDetails->class_id      = $assign->class_id;
                $assignDetails->month         = \Carbon\Carbon::parse($request->month)->format('Y-m');
                $assignDetails->amount        = $request->amount;
                $assignDetails->created_by    = Auth::user()->id;
                $assignDetails->save();

                DB::commit();

                return response()->json([
                    'success' => true,
                    'message' => 'Data created successfully',
                ]);
            } catch (\Exception $exception) {
                DB::rollBack();
                return response()->json([
                    'success' => false,
                    'message' => $exception->getMessage(),
                ]);
            }
        }
    }

    public function OtherFeesAssignEdit($id)
    {
        $data  = FeeAssign::findOrFail($id);
        $fee   = FeeAssignDetail::where('fee_assign_id', $id)->first();
        $month = \Carbon\Carbon::parse($fee->month)->format('M-Y');

        return response()->json([
            'id'         => $data->id,
            'class_id'   => $data->class_id,
            'session_id' => $data->session_id,
            'fee_id'     => $fee->fee_id,
            'month'      => $month,
            'amount'     => $fee->amount,
        ]);
    }

    public function OtherFeesAssignUpdate(Request $request, $id)
    {
        $data = Validator::make($request->all(), [
            'amount' => 'required',
        ]);

        if ($data->fails()) {
            return response()->json([
                'success' => false,
                'message' => $data->errors()->all(),
            ]);
        }

        DB::beginTransaction();

        try {
            $assign = FeeAssign::findOrFail($id);

            // Check for duplicates
            $exists = FeeAssign::where('id', '!=', $id)
                ->where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->where('type', 3)
                ->whereHas('feesDetails', function ($q) use ($request) {
                    $q->where('fee_id', $request->fee_id)
                        ->where('month', \Carbon\Carbon::parse($request->month)->format('Y-m'));
                })
                ->exists();

            if ($exists) {
                return response()->json([
                    'success' => false,
                    'message' => ['This fee is already assigned for the given month, session, and class.'],
                ]);
            }

            // Update main assign
            $assign->class_id   = $request->class_id;
            $assign->type       = 3;
            $assign->session_id = $request->session_id;
            $assign->created_by = Auth::user()->id;
            $assign->save();

            // Update details
            $assignDetails                = FeeAssignDetail::where('fee_assign_id', $id)->first();
            $assignDetails->type          = $assign->type;
            $assignDetails->fee_assign_id = $assign->id;
            $assignDetails->fee_id        = $request->fee_id;
            $assignDetails->month         = \Carbon\Carbon::parse($request->month)->format('Y-m');
            $assignDetails->amount        = $request->amount;
            $assignDetails->session_id    = $assign->session_id;
            $assignDetails->class_id      = $assign->class_id;
            $assignDetails->created_by    = Auth::user()->id;
            $assignDetails->save();

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Data updated successfully',
            ]);

        } catch (\Exception $exception) {
            DB::rollBack();
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }

    public function HostelFeesAssignDestroy($id)
    {
        try {
            $data = FeeAssign::findOrFail($id);
            // dd($data, $id);
            FeeAssignDetail::where('fee_assign_id', $id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }

    public function AcademicFeesAssignDestroy($id)
    {
        try {
            $data = FeeAssign::findOrFail($id);
            // dd($data, $id);
            FeeAssignDetail::where('fee_assign_id', $id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }

    public function OtherFeesAssignDestroy($id)
    {
        try {
            $data = FeeAssign::findOrFail($id);
            FeeAssignDetail::where('fee_assign_id', $id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }

}
