<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Traits\SmsTrait;
use App\Models\GurabaFund;
use App\Models\Transaction;
use App\Traits\DeleteTrait;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Yajra\DataTables\Facades\DataTables;

class GurabaFundController extends Controller
{
    use SmsTrait;
    use DeleteTrait;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {

        if ($request->ajax()) {

            $data = DB::table('guraba_funds')->where('type', 1)->whereNull('deleted_at')->orderBy('id','DESC');

            return Datatables::of($data)

                ->addColumn('action', function ($data) {

                        $result = '';

                        $details = '<li><a class="dropdown-item" href="' . route('guraba-fund.show', $data->id) . ' " ><i class="fa fa-eye" ></i> Details</a></li>';

                        $edit = '<li><a class="dropdown-item" href="' . route('guraba-fund.edit', $data->id) . ' "><i class="fa fa-edit"></i> Edit</a></li>';

                        $delete = '<li><a class="dropdown-item btn-delete" href="#" data-remote=" ' . route('guraba-fund.destroy', $data->id) . ' "><i class="fa fa-trash"></i> Delete</a></li>';

                        $result = $details. $edit. $delete;

                        return '<div class="btn-group open">
                        <a class="badge badge-primary dropdown-toggle" href="#" role="button"  data-toggle="dropdown">Actions<i class="ik ik-chevron-down mr-0 align-middle"></i></a>
                        <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">'.$result.'
                        </ul>
                    </div>';
                })

                ->addIndexColumn()
                ->rawColumns(['action'])
                ->toJson();
            }
         return view('dashboard.guraba_fund.index');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $id = GurabaFund::count();
        return view('dashboard.guraba_fund.create', compact('id'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        $messages = array(
            'receipt_no.required' => 'Enter receipt no',
            'name.required' => 'Enter name',
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
            'mobile.required' => 'Enter mobile number',
            'purpose.required' => 'Enter on account of',
            'address.required' => 'Write address',
        );

        $this->validate($request, array(
            'receipt_no' => 'required',
            'date' => 'required',
            'name' => 'required',
            'amount' => 'required',
            'mobile' => 'required',
            'purpose' => 'required',
            'address' => 'required',
        ), $messages);

        DB::beginTransaction();

        try {

            $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
            $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

            $data = new GurabaFund();
            $data->receipt_no         = $request->receipt_no ;
            $data->date               = Carbon::parse($request->date)->format('Y-m-d');
            $data->name               = $request->name;
            $data->mobile             = str_replace($bn_numbers, $en_numbers, $request->mobile);
            $data->address            = $request->address;
            $data->amount             = str_replace($bn_numbers, $en_numbers, $request->amount);
            $data->in_word            = $request->in_word;
            $data->type               = 1;
            $data->purpose            = $request->purpose;
            $data->note               = $request->note;
            $data->save();

            $transaction = new Transaction();
            $transaction->guraba_fund_id        = $data->id;
            $transaction->date                  = Carbon::parse($request->date)->format('Y-m-d');
            $transaction->transaction_purpose   = $request->purpose;
            $transaction->transaction_type      = 1;
            $transaction->payment_type          = 6;
            $transaction->amount                = str_replace($bn_numbers, $en_numbers, $request->amount);
            $transaction->save();

            $user = GurabaFund::findOrFail($data->id);

            $date = str_replace($en_numbers, $bn_numbers, Carbon::parse($user->date)->format('d/m/Y'));

            $phone = str_replace($bn_numbers, $en_numbers, $user->mobile);

            $msg = "দারুল উলুম সরকারপাড়া মাদরাসা, তারিখ : $date, নাম : $user->name, মোবাইল : $user->mobile, টাকার পরিমাণ : $user->amount , আপনার টাকা গ্রহণ করা হয়েছে, জাযাকাল্লাহ ।";

            $this->SendSms($phone, $msg);

            DB::commit();

            return redirect()->route('guraba-fund.show', $data->id)
                ->with('success', 'Money received successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $data = GurabaFund::findOrFail($id);
        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

        $receiptNo = str_replace($en_numbers, $bn_numbers, $data->receipt_no);
        $date = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->date)->format('d/m/Y'));
        return view('backend.guraba_fund.show', compact('data', 'receiptNo','date'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $data = GurabaFund::findOrFail($id);
        $date = Carbon::parse($data->date)->format('d-F-Y');
        return view('backend.guraba_fund.edit', compact('data','date'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $messages = array(
            'receipt_no.required' => 'Enter receipt no',
            'name.required' => 'Enter name',
            'date.required' => 'Enter date',
            'amount.required' => 'Enter amount',
            'mobile.required' => 'Enter mobile number',
            'purpose.required' => 'Enter on account of',
            'address.required' => 'Write address',
        );

        $this->validate($request, array(
            'receipt_no' => 'required',
            'date' => 'required',
            'name' => 'required',
            'amount' => 'required',
            'mobile' => 'required',
            'purpose' => 'required',
            'address' => 'required',
        ), $messages);

        DB::beginTransaction();

        try {
            $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
            $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

            $data = GurabaFund::findOrFail($id);
            $data->receipt_no         = $request->receipt_no ;
            $data->date               = Carbon::parse($request->date)->format('Y-m-d');
            $data->name               = $request->name;
            $data->mobile             = str_replace($bn_numbers, $en_numbers, $request->mobile);
            $data->address            = $request->address;
            $data->amount             = str_replace($bn_numbers, $en_numbers, $request->amount);
            $data->in_word            = $request->in_word;
            $data->purpose            = $request->purpose;
            $data->type               = 1;
            $data->note               = $request->note;
            $data->update();

            $transaction = Transaction::where('guraba_fund_id', $id)->first();
            $transaction->guraba_fund_id        = $data->id;
            $transaction->date                  = Carbon::parse($request->date)->format('Y-m-d');
            $transaction->transaction_purpose   = $request->purpose;
            $transaction->transaction_type      = 1;
            $transaction->amount                = str_replace($bn_numbers, $en_numbers, $request->amount);
            $transaction->update();

            $user = GurabaFund::findOrFail($data->id);
            $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
            $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

            $date = str_replace($en_numbers, $bn_numbers, Carbon::parse($user->date)->format('d/m/Y'));

            $phone = str_replace($bn_numbers, $en_numbers, $user->mobile);

            $msg = "দারুল উলুম সরকারপাড়া মাদরাসা, তারিখ : $date, নাম : $user->name, মোবাইল : $user->mobile, টাকার পরিমাণ : $user->amount , আপনার টাকা আপডেট করা হয়েছে, জাযাকাল্লাহ ।";

            $this->SendSms($phone, $msg);

            DB::commit();

            return redirect()->route('guraba-fund.show', $id)
                ->with('success', 'Money received updated successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try{
            $data = GurabaFund::findOrFail($id);
            if($data){
                $transaction = Transaction::where('guraba_fund_id', $id)->first();
                $transaction->delete();
                $data->delete();
            }
            return response()->json([
                'success' => true,
                'message' => 'Guraba fund deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Guraba fund delete failed',
            ]);
        }
    }
    public function print($id)
    {
        $data = GurabaFund::findOrFail($id);

        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

        $receiptNo = str_replace($en_numbers, $bn_numbers, $data->receipt_no);
        $date = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->date)->format('d/m/Y'));
        return view('backend.guraba_fund.print', compact('data','receiptNo','date'));
    }

    public function balanceSheet(Request $request)
    {
        try {

        if ($request->ajax()) {

            $data = DB::table('guraba_funds')->whereNull('deleted_at')->get();

            $totalCredit = DB::table('guraba_funds')->where('type', 1)->whereNull('deleted_at')->sum('amount');

            $totalDebit = DB::table('guraba_funds')->where('type', 2)->whereNull('deleted_at')->sum('amount');

            $currentBalance = $totalCredit - $totalDebit;

            return Datatables::of($data)

            ->addColumn('date', function ($data) {
                $date = Carbon::parse($data->date)->format('d F, Y');
                return $date;
            })

            ->addColumn('credit_amount', function ($data) {
                if ($data->type == 1 ) {
                    $total_amount = $data->amount;
                    return $total_amount;
                }else{
                    return '--';
                }

            })

            ->addColumn('debit_amount', function ($data) {
                if ($data->type == 2 ) {
                    $total_amount = $data->amount;
                    return $total_amount;
                }else{
                    return '--';
                }
            })

            ->addColumn('current_balance', function ($data) use (&$current_balance) {
                if ($data->type == 1) {
                    $current_balance = $current_balance + $data->amount;
                    return (number_format($current_balance));
                }

                if ($data->type == 2) {
                    $current_balance = $current_balance - $data->amount;
                    return (number_format($current_balance));
                }
            })

            ->with('totalCredit', $totalCredit)
            ->with('totalDebit', $totalDebit)
            ->with('currentBalance', $currentBalance)
            ->addIndexColumn()
            ->rawColumns(['date', 'debit_amount', 'credit_amount', 'current_balance'])
                ->toJson();
            }

         return view('backend.guraba_fund.balance');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
}
