<?php
namespace App\Http\Controllers;

use App\Models\Boarding;
use App\Models\Donar;
use App\Models\FeeAssignDetail;
use App\Models\Fund;
use App\Models\FundDetail;
use App\Models\FundDetailType;
use App\Models\InvoiceDesign;
use App\Models\Mahfil;
use App\Models\MoneyReceipt;
use App\Models\MoneyReceiptMultiple;
use App\Models\SmsStatus;
use App\Models\Student;
use App\Models\StudentClass;
use App\Models\StudentPayment;
use App\Models\StudentSponsor;
use App\Traits\DateFormatter;
use App\Traits\DeleteTrait;
use App\Traits\SmsTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\Facades\DataTables;

class PaymentController extends Controller
{
    use SmsTrait;
    use DateFormatter;
    use DeleteTrait;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function PaymentListNew(Request $request)
    {
        try {
            if ($request->ajax()) {

                $data = MoneyReceipt::with('students', 'stuclass')
                    ->where('type', 1)
                    ->where('receipt_type', 'new');

                if ($request->month) {
                    $data->where('month', $request->month);
                }

                if ($request->class_id) {
                    $data->whereHas('students', function ($q) use ($request) {
                        $q->where('class_id', $request->class_id);
                    });
                }

                $data = $data->groupBy('month', 'student_id')
                    ->orderBy('id', 'desc')
                    ->get();

                return DataTables::of($data)
                    ->addColumn('department', function ($data) {
                        return optional($data->students->stuclass)->name ?? '--';
                    })
                    ->addColumn('student', function ($data) {
                        $name   = optional($data->students)->name ?? '--';
                        $reg    = optional($data->students)->register_no ?? '--';
                        $mobile = optional($data->students)->gurdian_mobile ?? '--';
                        return $name . '<br>' . $reg . '<br>' . $mobile;
                    })
                    ->addColumn('date', function ($data) {
                        return \Carbon\Carbon::createFromFormat('Y-m', trim($data->month))->format('M-Y');
                    })
                    ->addColumn('total_amount', function ($data) {
                        return $data->total ?? '--';
                    })
                    ->addColumn('receive_amount', function ($data) {
                        $receiveAmount = MoneyReceipt::where('type', 1)
                            ->where('receipt_type', 'new')
                            ->where('month', $data->month)
                            ->where('student_id', $data->student_id)
                            ->sum('current_receive');
                        return $receiveAmount ?? '--';
                    })
                // NEW: last receive date
                    ->addColumn('last_receive_date', function ($data) {
                        $lastDate = MoneyReceipt::where('type', 1)
                            ->where('receipt_type', 'new')
                            ->where('month', $data->month)
                            ->where('student_id', $data->student_id)
                            ->where('current_receive', '>', 0) // only rows that actually received
                            ->max('date');                     // Y-m-d (assumed)
                        return $lastDate ? \Carbon\Carbon::parse($lastDate)->format('d-M-Y') : '--';
                    })
                    ->addColumn('due_amount', function ($data) {
                        $receive = MoneyReceipt::where('type', 1)
                            ->where('receipt_type', 'new')
                            ->where('month', $data->month)
                            ->where('student_id', $data->student_id)
                            ->sum('current_receive');

                        $discount = FundDetailType::where('payment_type', 1)
                            ->where('receipt_type', 'new')
                            ->where('month', $data->month)
                            ->where('student_id', $data->student_id)
                            ->sum('discount');

                        $dueAmount = ($data->total ?? 0) - $receive - $discount;
                        return $dueAmount ?? '--';
                    })
                    ->addColumn('discount_amount', function ($data) {
                        $discount = FundDetailType::where('payment_type', 1)
                            ->where('receipt_type', 'new')
                            ->where('month', $data->month)
                            ->where('student_id', $data->student_id)
                            ->sum('discount');
                        return $discount ?? '--';
                    })
                    ->addColumn('action', function ($data) {
                        $details = '<a type="button" id="print" class="btn btn-sm btn-primary print mr-1" href="' . route('payment.list.details', ['month' => $data->month, 'student_id' => $data->student_id]) . '" title="Print"><i class="fa fa-eye"></i></a>';
                        return $details;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['department', 'action', 'student'])
                    ->toJson();
            }

            $classes = StudentClass::whereNull('deleted_at')->get();

            return view('dashboard.payments.index_new', compact('classes'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function PaymentList(Request $request)
    {
        try {
            if ($request->ajax()) {

                $data = MoneyReceipt::with('students', 'stuclass')
                    ->where('type', 1)
                    ->selectRaw('month, student_id, SUM(amount) as total_amount')
                    ->groupBy('month', 'student_id')
                    ->whereNull('receipt_type')
                    ->orderBy('id', 'desc')
                    ->get();

                return DataTables::of($data)
                    ->addColumn('department', function ($data) {
                        return optional($data->students->stuclass)->name ?? '--';
                    })
                    ->addColumn('student', function ($data) {
                        $name   = optional($data->students)->name ?? '--';
                        $reg    = optional($data->students)->register_no ?? '--';
                        $mobile = optional($data->students)->gurdian_mobile ?? '--';
                        return $name . '<br>' . $reg . '<br>' . $mobile;
                    })
                    ->addColumn('date', function ($data) {

                        return \Carbon\Carbon::createFromFormat('Y-m', trim($data->month))->format('M-Y');
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->total_amount ?? '--';
                    })
                    ->addColumn('due', function ($data) {
                        $month      = $data->month;
                        $student_id = $data->student_id;

                        $student = Student::find($student_id);
                        if (! $student) {
                            return '--';
                        }

                        $class_id   = $student->class_id;
                        $session_id = $student->session_id;

                        $academicAmount = FeeAssignDetail::where('type', 1)
                            ->where('class_id', $class_id)
                            ->where('session_id', $session_id)
                            ->where('month', $month)
                            ->sum('amount');

                        $hostelAmount = 0;
                        if ($student->student_type == 2) {
                            $hostelAmount = FeeAssignDetail::where('type', 2)
                                ->where('class_id', $class_id)
                                ->where('session_id', $session_id)
                                ->where('month', $month)
                                ->sum('amount');
                        }

                        // Assigned Other Fee
                        $otherAmount = FeeAssignDetail::where('type', 3)
                            ->where('class_id', $class_id)
                            ->where('session_id', $session_id)
                            ->where('month', $month)
                            ->sum('amount');

                        $onabashikAmount = 0;
                        if ($student->student_type == 1) {
                            $onabashikAmount = FeeAssignDetail::where('type', 4)
                                ->where('class_id', $class_id)
                                ->where('session_id', $session_id)
                                ->where('month', $month)
                                ->sum('amount');
                        }

                        $specialAmount = DB::table('fee_assign_details as fd')
                            ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id') // ✅ join condition
                            ->whereNull('fd.deleted_at')                                                           // ✅ soft delete check
                            ->whereNull('fds.deleted_at')                                                          // ✅ soft delete check
                            ->where('fd.type', 5)
                            ->where('fds.student_id', $student_id)
                            ->where('fd.session_id', $session_id)
                            ->where('fd.class_id', $class_id)
                            ->where('fd.month', $month)
                            ->sum('fds.amount');

                        $monthFee = $academicAmount + $hostelAmount + $otherAmount + $onabashikAmount + $specialAmount;

                        $receivedData = FundDetail::where('student_id', $student_id)
                            ->where('month', $month)
                            ->whereIn('type', [6, 16])
                            ->where('payment_type', 1)
                            ->select(
                                DB::raw('SUM(amount) as total_received'),
                                DB::raw('SUM(discount) as total_discount')
                            )
                            ->first();

                        $received = $receivedData->total_received ?? 0;
                        $discount = $receivedData->total_discount ?? 0;

                        $due = $monthFee - $received - $discount;

                        return round(max($due, 0), 2);
                    })

                    ->addColumn('action', function ($data) {
                        $details = '<a type="button" id="print" class="btn btn-sm btn-primary print mr-1" href="' . route('payment.list.details', ['month' => $data->month, 'student_id' => $data->student_id]) . '" title="Print"><i class="fa fa-eye"></i></a>';
                        return $details;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['department', 'action', 'student', 'due'])
                    ->toJson();
            }

            return view('dashboard.payments.index');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function PaymentListDetail(Request $request, $month, $student_id)
    {
        try {
            if ($request->ajax()) {

                $data = MoneyReceipt::with('payment', 'students:id,name,status')->where('month', $month)
                    ->where('student_id', $student_id)
                    ->where('type', 1)
                    ->get();

                // dd($data);

                if ($data->isEmpty()) {
                    return response()->json([
                        'data'            => [],
                        'recordsTotal'    => 0,
                        'recordsFiltered' => 0,
                    ]);
                }

                $lastRecord = MoneyReceipt::where('month', $month)
                    ->where('student_id', $student_id)
                    ->where('type', 1)
                    ->orderBy('id', 'desc')
                    ->first();

                return Datatables::of($data)
                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d/m/Y');
                    })
                    ->addColumn('total_amount', function ($data) {
                        return $data->total ?? '';
                    })
                    ->addColumn('receive_amount', function ($data) {
                        return $data->amount ?? '';
                    })
                    ->addColumn('due_amount', function ($data) {
                        return $data->due ?? '';
                    })
                    ->addColumn('discount_amount', function ($data) {
                        return $data->discount ?? '';
                    })
                    ->addColumn('action', function ($data) use ($lastRecord) {

                        if (isset($data->students) && ($data->students->status == 1 || $data->students->status == null)) {

                            $edit = '';
                            $show = '';

                            $edit = '<a id="edit" href="' . route('payment.edit', $data->id) . '" class="btn btn-sm btn-info edit" title="Edit"><i class="fa fa-edit"></i></a> ';

                            $show = '<a id="show" href="' . route('payment.show', $data->id) . '" class="btn btn-sm btn-info show" title="Show"><i class="fa fa-eye"></i></a> ';

                            $delete = '';
                            if ($this->DeleteData()) {
                                $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('payment.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                            }

                            $print = '<a type="button" id="print" class="btn btn-sm btn-primary print mr-1" href="' . route('payment.print', $data->id) . ' " title="a6 page"><i class="fa fa-file"></i></a>';

                            return $show . $edit . $print . $delete;
                        }
                    })
                    ->addIndexColumn()
                    ->rawColumns(['action', 'date', 'amount', 'discount'])
                    ->toJson();
            }

            $student = Student::find($student_id);
            if (! $student) {
                return redirect()->back()->with('error', 'Student not found.');
            }

            $studentName  = $student->name ?? '';
            $studentClass = $student->stuclass->name ?? '';
            $session      = $student->stusession->name ?? '';

            return view('dashboard.payments.payment_details', compact('student_id', 'month', 'studentName', 'studentClass', 'session'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function PaymentListDetail_old(Request $request, $month, $student_id)
    {
        try {
            if ($request->ajax()) {

                $data = MoneyReceipt::with('payment', 'students:id,name,status')->where('month', $month)
                    ->where('student_id', $student_id)
                    ->where('type', 1)
                    ->get();

                if ($data->isEmpty()) {
                    return response()->json([
                        'data'            => [],
                        'recordsTotal'    => 0,
                        'recordsFiltered' => 0,
                    ]);
                }

                $lastRecord = MoneyReceipt::where('month', $month)
                    ->where('student_id', $student_id)
                    ->where('type', 1)
                    ->orderBy('id', 'desc')
                    ->first();

                return Datatables::of($data)
                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d/m/Y');
                    })
                    ->addColumn('amount', function ($data) {
                        return $data->amount ?? '';
                    })
                    ->addColumn('discount', function ($data) {
                        return $data->payment->discount ?? '';
                    })
                    ->addColumn('action', function ($data) use ($lastRecord) {

                        if (isset($data->students) && ($data->students->status == 1 || $data->students->status == null)) {

                            $edit = '';

                            if ($data->id == $lastRecord->id) {

                                $edit = '<a id="edit" href="' . route('payment.edit', $data->id) . '" class="btn btn-sm btn-info edit" title="Edit"><i class="fa fa-edit"></i></a> ';
                            }

                            $delete = '';
                            if ($this->DeleteData()) {
                                $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('payment.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                            }

                            $print = '<a type="button" id="print" class="btn btn-sm btn-primary print mr-1" href="' . route('payment.print', $data->id) . ' " title="a6 page"><i class="fa fa-file"></i></a>';

                            return $edit . $print . $delete;
                        }
                    })
                    ->addIndexColumn()
                    ->rawColumns(['action', 'date', 'amount', 'discount'])
                    ->toJson();
            }

            $student = Student::find($student_id);
            if (! $student) {
                return redirect()->back()->with('error', 'Student not found.');
            }

            $studentName  = $student->name ?? '';
            $studentClass = $student->stuclass->name ?? '';
            $session      = $student->stusession->name ?? '';

            return view('dashboard.payments.payment_details', compact('student_id', 'month', 'studentName', 'studentClass', 'session'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];
        $id         = MoneyReceipt::count();
        $receipt    = str_replace($en_numbers, $bn_numbers, ($id + 1));
        $class      = StudentClass::all();
        return view('dashboard.payments.create', compact('receipt', 'class'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function PaymentStore(Request $request)
    {
        // foreach ($request->rows as $key => $row) {
        //     dd($row['enabled'], $key, $request->all());
        // }
        // dd($request->all());

        $lastFundDetail = FundDetail::whereNotNull('receipt_no')
            ->orderByDesc('receipt_no')
            ->first();
        $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;

        $totalReceived = FundDetailType::where('student_id', $request->student_id)
            ->where('month', Carbon::parse($request->month)->format('Y-m'))
            ->where('receipt_type', "new")
            ->whereNull('deleted_at')
            ->sum('receive');

        $totalDiscount = FundDetailType::where('student_id', $request->student_id)
            ->where('month', Carbon::parse($request->month)->format('Y-m'))
            ->where('receipt_type', "new")
            ->whereNull('deleted_at')
            ->sum('discount');

        $totalAmount = (float) $request->total_amount;
        $dueAmount   = (float) $totalAmount - ((float) $totalReceived + (float) $totalDiscount);

        // dd($totalReceived, $totalDiscount, $totalAmount, $dueAmount);

        // if ($dueAmount == 0) {

        //     return 'Payment already paid for ' . Carbon::parse($request->month)->format('F Y');
        // }

        DB::beginTransaction();

        try {
            if ($request->due_type == "old") {
                FundDetail::where('student_id', $request->student_id)
                    ->where('month', $request->month)
                    ->whereIn('type', [6, 16])
                    ->where('payment_type', 1)
                    ->whereNull('deleted_at')
                    ->whereNull('receipt_type')
                    ->whereNull('fund_detail_type_id')
                    ->whereNull('fee_id')
                    ->delete();
                // dd($totalReceived);
            }

            $student = Student::findOrFail($request->student_id);

            $data                  = new MoneyReceipt();
            $data->student_id      = $request->student_id;
            $data->receipt_no      = $receipt_no;
            $data->register_no     = $student->register_no;
            $data->date            = now()->format('Y-m-d');
            $data->name            = $student->name;
            $data->class_id        = $student->class_id;
            $data->month           = Carbon::parse($request->month)->format('Y-m');
            $data->amount          = $request->receive_amount;
            $data->current_receive = $request->current_receive_amount;
            $data->discount        = $request->discount_amount;
            $data->due             = $request->due_amount;
            $data->total           = $request->total_amount;
            $data->type            = 1;
            $data->receipt_type    = 'new';
            $data->save();

            foreach ($request->rows[$request->month] as $key => $row) {
                $fundKeyMap = [
                    'other'     => 'other_fee_paid',
                    'academic'  => 'academic_fee_paid',
                    'hostel'    => 'hostel_fee_paid',
                    'onabashik' => 'onabashik_fee_paid',
                    'special'   => 'special_fee_paid',
                ];

                // dd($request->all(), $request->rows[$request->month], $key,$request->{$key . '_amount_total'}[$request->month], $request->{'fund_id_' . $key . 'fee_ids'}[$request->month]);
                if (isset($fundKeyMap[$key])) {
                    $feeKey = $fundKeyMap[$key];
                    // dd($feeKey, $key, $request->all());

                    $fundDetailTypes = FundDetailType::create([
                        'student_id'         => $request->student_id,
                        'class_id'           => $request->class_id,
                        'session_id'         => $request->session_id,
                        'receipt_no'         => $receipt_no,
                        'purpose'            => 'Student Monthly Payment',
                        'student_payment_id' => $data->id,
                        'fee_type'           => $key,
                        'month'              => Carbon::parse($request->month)->format('Y-m'),
                        'total'              => $request->{$key . '_amount_total'}[$request->month],
                        'prev_receive'       => $request->{$key . '_previous_receive'}[$request->month],
                        'prev_discount'      => $request->{$key . '_previous_discount'}[$request->month],
                        'receive'            => $request->{$feeKey}[$request->month],
                        'discount'           => $request->{$key . '_fee_discount'}[$request->month],
                        'due'                => $request->{$key . '_fee_due'}[$request->month],
                        'date'               => now()->format('Y-m-d'),
                        'type'               => 6,
                        'payment_type'       => 1,
                        'fund_type'          => 1,
                        'receipt_type'       => 'new',

                    ]);

                    if (isset($request->{'fund_id_' . $key . 'fee_ids'}[$request->month])) {
                        foreach ($request->{'fund_id_' . $key . 'fee_ids'}[$request->month] as $fee_id => $amount) {

                            //  dd($fee_id, $fundId, $key, $request->{'fund_id_' . $key . 'fee_amount'}, $request->all());
                            FundDetail::create([
                                'fund_detail_type_id' => $fundDetailTypes->id,
                                'fund_id'             => $request->{'fund_id_' . $key}[$request->month][$fee_id],
                                'student_payment_id'  => $data->id,
                                'student_id'          => $request->student_id,
                                'invoice_no'          => $request->invoice_no,
                                'receipt_no'          => $receipt_no,
                                'purpose'             => 'Student Monthly Payment',
                                'date'                => now()->format('Y-m-d'),
                                'type'                => 6,
                                'payment_type'        => 1,
                                'fund_type'           => 1,
                                'month'               => Carbon::parse($request->month)->format('Y-m'),
                                'total'               => $amount,
                                'amount'              => $request->{'fund_id_' . $key . 'fee_paid'}[$request->month][$fee_id],
                                'discount'            => $request->{'fund_id_' . $key . 'fee_discount'}[$request->month][$fee_id],
                                'due'                 => $request->{'fund_id_' . $key . 'fee_due'}[$request->month][$fee_id],
                                'fee_id'              => $fee_id,
                                'receipt_type'        => 'new',
                            ]);
                        }
                    }
                }
            }

            $status = optional(SmsStatus::first())->student_payment;
            $site   = DB::table('settings')->first();
            $to     = $student->gurdian_mobile;
            $text   = $student->name . ', ' . $data->month . ' মাসিক বেতন Paid : ' . $data->amount . ' TK';
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            DB::commit();

            return redirect()->route('payment.print', ['id' => $data->id])->with('success', 'Amount received successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            dd($e);
            return back()->with('error', 'Error occurred: ' . $e->getMessage());
        }
    }

    public function PaymentStoreMulti(Request $request)
    {
        DB::beginTransaction();

        try {
            $student = Student::findOrFail($request->student_id);

            $fundKeyMap = [
                'academic'  => 'academic_fee_paid',
                'hostel'    => 'hostel_fee_paid',
                'special'   => 'special_fee_paid',
                'onabashik' => 'onabashik_fee_paid',
                'other'     => 'other_fee_paid',
                // add other keys like onabashik, other, special if used
            ];

            // dd($request->all());

            $moneyReceiptMulti                  = new MoneyReceiptMultiple();
            $moneyReceiptMulti->student_id      = $request->student_id;
            $moneyReceiptMulti->class_id        = $request->class_id;
            $moneyReceiptMulti->date            = now()->format('Y-m-d');
            $moneyReceiptMulti->name            = $student->name;
            $moneyReceiptMulti->total_amount    = $request->total_amount;
            $moneyReceiptMulti->current_receive = $request->current_receive_amount;
            $moneyReceiptMulti->total_receive   = $request->receive_amount;
            $moneyReceiptMulti->total_discount  = $request->discount_amount;
            $moneyReceiptMulti->total_due       = $request->due_amount;
            $moneyReceiptMulti->save();

            foreach ($request->months as $month) {

                $totalAmount      = 0;
                $currentReceive   = 0;
                $previousReceive  = 0;
                $previousDiscount = 0;
                $currentDiscount  = 0;
                $currentDue       = 0;
                foreach ($fundKeyMap as $key => $value) {
                    if (isset($request->{$key . '_amount_total'}[$month])) {
                        $totalAmount += (float) $request->{$key . '_amount_total'}[$month];
                    }

                    if (isset($request->{$key . '_fee_paid'}[$month])) {
                        $currentReceive += (float) $request->{$key . '_fee_paid'}[$month];
                    }

                    if (isset($request->{$key . '_fee_discount'}[$month])) {
                        $currentDiscount += (float) $request->{$key . '_fee_discount'}[$month];
                    }

                    if (isset($request->{$key . '_fee_due'}[$month])) {
                        $currentDue += (float) $request->{$key . '_fee_due'}[$month];
                    }

                    if (isset($request->{$key . '_previous_receive'}[$month])) {
                        $previousReceive += (float) $request->{$key . '_previous_receive'}[$month];
                    }

                    if (isset($request->{$key . '_previous_discount'}[$month])) {
                        $previousDiscount += (float) $request->{$key . '_previous_discount'}[$month];
                    }

                }

                // Skip if already fully paid
                $totalReceived = $currentReceive + $previousReceive;
                $totalDiscount = $currentDiscount + $previousDiscount;

                $dueAmount = $totalAmount - ($totalReceived + $totalDiscount);
                // dd($request->all(), $totalAmount, $currentReceive, $currentDiscount, $currentDue, $previousReceive,$previousDiscount, $totalReceived, $totalDiscount, $dueAmount);

                //     if ($dueAmount <= 0) {
                //     dd($request->all(), $moneyReceiptMulti, $dueAmount, $totalReceived, $totalDiscount, $totalAmount, $currentReceive, $currentDiscount, $currentDue, $previousReceive,$previousDiscount);
                //     continue;
                // }

                $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                    ->orderByDesc('receipt_no')
                    ->first();
                $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;

                if ($request->due_type === "old") {
                    FundDetail::where('student_id', $request->student_id)
                        ->where('month', $month)
                        ->whereIn('type', [6, 16])
                        ->where('payment_type', 1)
                        ->whereNull('deleted_at')
                        ->whereNull('receipt_type')
                        ->whereNull('fund_detail_type_id')
                        ->whereNull('fee_id')
                        ->delete();
                }

                // Save main money receipt
                $moneyReceipt                            = new MoneyReceipt();
                $moneyReceipt->student_id                = $request->student_id;
                $moneyReceipt->receipt_no                = $receipt_no;
                $moneyReceipt->register_no               = $student->register_no;
                $moneyReceipt->date                      = now()->format('Y-m-d');
                $moneyReceipt->name                      = $student->name;
                $moneyReceipt->class_id                  = $student->class_id;
                $moneyReceipt->month                     = $month;
                $moneyReceipt->amount                    = $totalReceived;
                $moneyReceipt->current_receive           = $currentReceive;
                $moneyReceipt->discount                  = $totalDiscount;
                $moneyReceipt->due                       = $dueAmount;
                $moneyReceipt->total                     = $totalAmount;
                $moneyReceipt->type                      = 1;
                $moneyReceipt->receipt_type              = 'new';
                $moneyReceipt->money_receipt_multiple_id = $moneyReceiptMulti->id;
                $moneyReceipt->save();

                foreach (($request->rows[$month] ?? []) as $type => $row) {
                    if (! isset($row['enabled'])) {
                        continue;
                    }

                    $feeKey = $fundKeyMap[$type] ?? null;
                    if (! $feeKey) {
                        continue;
                    }

                    $fundDetailType = FundDetailType::create([
                        'student_id'         => $request->student_id,
                        'class_id'           => $request->class_id,
                        'session_id'         => $request->session_id,
                        'receipt_no'         => $receipt_no,
                        'purpose'            => 'Student Monthly Payment',
                        'student_payment_id' => $moneyReceipt->id,
                        'fee_type'           => $type,
                        'month'              => $month,
                        'total'              => $request[$type . '_amount_total'][$month] ?? 0,
                        'prev_receive'       => $request[$type . '_previous_receive'][$month] ?? 0,
                        'prev_discount'      => $request[$type . '_previous_discount'][$month] ?? 0,
                        'receive'            => $request[$type . '_fee_paid'][$month] ?? 0,
                        'discount'           => $request[$type . '_fee_discount'][$month] ?? 0,
                        'due'                => $request[$type . '_fee_due'][$month] ?? 0,
                        'date'               => now()->format('Y-m-d'),
                        'type'               => 6,
                        'payment_type'       => 1,
                        'fund_type'          => 1,
                        'receipt_type'       => 'new',
                    ]);

                    $feeIDs  = $request->{'fund_id_' . $type . 'fee_ids'}[$month] ?? [];
                    $funds   = $request->{'fund_id_' . $type}[$month] ?? [];
                    $amounts = $request->{'fund_id_' . $type . 'fee_amount'}[$month] ?? [];

                    // dd($feeIDs, $funds, $amounts, $request->all());

                    foreach ($feeIDs as $fee_id => $amount) {
                        FundDetail::create([
                            'fund_detail_type_id' => $fundDetailType->id,
                            'fund_id'             => $funds[$fee_id] ?? null,
                            'student_payment_id'  => $moneyReceipt->id,
                            'student_id'          => $request->student_id,
                            'receipt_no'          => $receipt_no,
                            'purpose'             => 'Student Monthly Payment',
                            'date'                => now()->format('Y-m-d'),
                            'type'                => 6,
                            'payment_type'        => 1,
                            'fund_type'           => 1,
                            'month'               => $month,

                            'total'               => $amount,
                            'amount'              => $request->{'fund_id_' . $type . 'fee_paid'}[$month][$fee_id],
                            'discount'            => $request->{'fund_id_' . $type . 'fee_discount'}[$month][$fee_id],
                            'due'                 => $request->{'fund_id_' . $type . 'fee_due'}[$month][$fee_id],
                            'fee_id'              => $fee_id,
                            'receipt_type'        => 'new',
                        ]);
                    }
                }

                // SMS (optional)
                $status = optional(SmsStatus::first())->student_payment;
                $site   = DB::table('settings')->first();
                $to     = $student->gurdian_mobile;
                $text   = $student->name . ', ' . $month . ' মাসিক বেতন Paid : ' . $moneyReceipt->amount . ' TK';
                if ($status == 1) {
                    $this->SendSms($to, $text);
                }
            }

            DB::commit();
            return redirect()->route('payment.print.multi', ['id' => $moneyReceiptMulti->id])->with('success', 'Amount received successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error occurred: ' . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $data       = MoneyReceipt::with('students')->findOrFail($id);
        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

        $bn_months = ["জানুয়ারী", "ফেব্রুয়ারী ", "মার্চ", "এপ্রিল", "মে", "জুন", "জুলাই", "অগাস্ট", "সেপ্টেম্বর", "অক্টোবর", "নভেম্বর", "ডিসেম্বর"];

        $en_months = ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"];

        $receiptNo = str_replace($en_numbers, $bn_numbers, $data->receipt_no);
        $date      = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->date)->format('d/m/Y'));

        $monthFormat = Carbon::parse($data->month)->format('F, Y');

        $month1 = str_replace($en_numbers, $bn_numbers, $monthFormat);

        $month = str_replace($en_months, $bn_months, $month1);

        return view('dashboard.money_receipt.show', compact('data', 'receiptNo', 'date', 'month'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {

        $data       = MoneyReceipt::with('students')->findOrFail($id);
        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

        $receiptNo = str_replace($en_numbers, $bn_numbers, $data->receipt_no);
        $amount    = str_replace($en_numbers, $bn_numbers, $data->amount);
        return view('backend.money_receipt.edit', compact('data', 'receiptNo', 'amount'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        DB::beginTransaction();

        try {

            $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                ->orderByDesc('receipt_no')
                ->first();
            $receipt_no    = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;
            $totalReceived = FundDetail::where('student_id', $request->student_id)
                ->where('type', 6)
                ->where('month', Carbon::parse($request->month)->format('Y-m'))
                ->sum('amount');

            $totalAmount   = (float) $request->total_amount;
            $totalReceived = (float) $totalReceived + ((float) $request->amount - (float) $request->discount);

            if ($totalReceived > $totalAmount) {
                return 'Payment already paid for ' . Carbon::parse($request->month)->format('F Y');
            }

            $student = Student::findOrFail($request->student_id);

            $data              = new MoneyReceipt();
            $data->student_id  = $request->student_id;
            $data->receipt_no  = $receipt_no;
            $data->register_no = $student->register_no;
            $data->date        = now()->format('Y-m-d');
            $data->name        = $student->name;
            $data->class_id    = $student->class_id;
            $data->month       = Carbon::parse($request->month)->format('Y-m');
            $data->amount      = $request->amount;
            $data->type        = 1;
            $data->save();

            $fund                     = new FundDetail();
            $fund->fund_id            = $request->fund_id;
            $fund->student_payment_id = $data->id;
            $fund->student_id         = $request->student_id;
            $fund->invoice_no         = $request->invoice_no;
            $fund->receipt_no         = $receipt_no;
            $fund->purpose            = 'Student Monthly Payment';
            $fund->date               = now()->format('Y-m-d');
            $fund->type               = 6;
            $fund->payment_type       = 1;
            $fund->fund_type          = 1;
            $fund->month              = Carbon::parse($request->month)->format('Y-m');
            $fund->amount             = $request->amount;
            $fund->discount           = $request->discount;
            $fund->note               = $request->note;
            $fund->save();

            $status = optional(SmsStatus::first())->student_payment;
            $site   = DB::table('settings')->first();
            $to     = $student->gurdian_mobile;
            $text   = $student->name . ', ' . $data->month . ' মাসিক বেতন Paid : ' . $data->amount . ' TK';
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            DB::commit();

            return redirect()->route('payment.print', ['id' => $data->id])->with('success', 'Amount received successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function PaymentUpdate(Request $request, $id)
    {

        // dd($request->all());

        DB::beginTransaction();

        $totalReceived = FundDetail::where('student_id', $request->student_id)
            ->where('type', 6)
            ->where('month', Carbon::parse($request->month)->format('Y-m'))
            ->select(DB::raw('SUM(amount + discount) as total'))
            ->value('total');

        $preReceived = $totalReceived - ((float) $request->amount + (float) $request->discount);

        $totalAmount = (float) $request->total_amount;

        $totalReceivedAmount = (float) $preReceived + ((float) $request->amount + (float) $request->discount);

        // if ($totalReceivedAmount > $totalAmount) {
        //     return 'Payment already paid for ' . Carbon::parse($request->month)->format('F Y');
        // }

        $student = Student::findOrFail($request->student_id);

        $data                  = MoneyReceipt::findOrFail($id);
        $data->amount          = $request->receive_amount;
        $data->current_receive = $request->current_receive_amount;
        $data->discount        = $request->discount_amount;
        $data->due             = $request->due_amount;
        $data->total           = $request->total_amount;
        $data->save();

        foreach ($request->rows as $key => $row) {
            $fundKeyMap = [
                'other'     => 'other_fee_paid',
                'academic'  => 'academic_fee_paid',
                'hostel'    => 'hostel_fee_paid',
                'onabashik' => 'onabashik_fee_paid',
                'special'   => 'special_fee_paid',
            ];

            if (isset($fundKeyMap[$key])) {
                $feeKey = $fundKeyMap[$key];

                $fundDetailTypes = FundDetailType::where('student_payment_id', $data->id)
                    ->where('fee_type', $key)
                    ->where('month', Carbon::parse($request->month)->format('Y-m'))
                    ->find($row['id']);

                // dd($feeKey, $key, $request->all());

                $fundDetailTypes->update([

                    'student_payment_id' => $data->id,
                    'fee_type'           => $key,

                    'total'              => $request->{$key . '_amount_total'},
                    'prev_receive'       => $request->{$key . '_previous_receive'},
                    'prev_discount'      => $request->{$key . '_previous_discount'},
                    'receive'            => $request->{$feeKey},
                    'discount'           => $request->{$key . '_fee_discount'},
                    'due'                => $request->{$key . '_fee_due'},
                    'updated_at'         => now()->format('Y-m-d'),

                ]);

                if ($request->{'detail_id_' . $key}) {
                    foreach ($request->{'detail_id_' . $key} as $detail_id) {
                        $fundDetail = FundDetail::findOrFail($detail_id);
                        // dd($request->all(), $key,$row['id'], $feeKey, $fundDetailTypes, $fundDetail, $request->{'fund_id_' . $key}[$fundDetail->fee_id]);
                        $fundDetail->update([
                            'student_payment_id'  => $data->id,
                            'fund_detail_type_id' => $fundDetailTypes->id,
                            'fund_id'             => $request->{'fund_id_' . $key}[$fundDetail->fee_id],
                            'updated_at'          => now()->format('Y-m-d'),
                        ]);

                        //  dd($fee_id, $fundId, $key, $request->{'fund_id_' . $key . 'fee_amount'}, $request->all());

                    }
                }

            }
        }

        $status = optional(SmsStatus::first())->student_payment;
        $site   = DB::table('settings')->first();
        $to     = $student->gurdian_mobile;
        $text   = $student->name . ', ' . $data->month . ' মাসিক বেতন Paid: ' . $data->amount . ' TK';
        if ($status == 1) {
            $this->SendSms($to, $text);
        }

        DB::commit();

        return redirect()->route('payment.print', ['id' => $data->id])->with('success', 'Amount received successfully');

        DB::rollBack();
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $data = MoneyReceipt::findOrFail($id);
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Money receipt deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Money receipt delete failed',
            ]);
        }
    }
    public function PaymentDestroy($id)
    {
        try {
            $data = MoneyReceipt::findOrFail($id);
            // dd($data);
            FundDetail::where('student_payment_id', $id)->where('type', 6)->delete();
            FundDetailType::where("student_payment_id", $data->id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Money receipt deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Money receipt delete failed',
            ]);
        }
    }
    public function print($id)
    {
        $data = MoneyReceipt::with('students')->findOrFail($id);

        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

        $receiptNo = str_replace($en_numbers, $bn_numbers, $data->receipt_no);
        $month     = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->month)->format('m/ Y'));
        $amount    = str_replace($en_numbers, $bn_numbers, $data->amount);

        $date = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->date)->format('d/m/Y'));

        $bn_months = ["জানুয়ারী", "ফেব্রুয়ারী ", "মার্চ", "এপ্রিল", "মে", "জুন", "জুলাই", "অগাস্ট", "সেপ্টেম্বর", "অক্টোবর", "নভেম্বর", "ডিসেম্বর"];

        $en_months = ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"];

        $receiptNo = str_replace($en_numbers, $bn_numbers, $data->receipt_no);
        $date      = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->date)->format('d/m/Y'));

        $monthFormat = Carbon::parse($data->month)->format('F, Y');

        $month1 = str_replace($en_numbers, $bn_numbers, $monthFormat);

        $month = str_replace($en_months, $bn_months, $month1);

        return view('dashboard.money_receipt.print', compact('data', 'receiptNo', 'date', 'month', 'amount'));
    }

    public function PaymentCreate(Request $request)
    {
        $class_id   = $request->class_id;
        $student_id = $request->student_id;

        if (! $student_id == null) {
            //  dd($request->all(), $student_id, $class_id);
            $student = Student::findOrFail($student_id);
            if ($student->etim == 1) {
                return redirect()->route('sponsor.amount.take');
            }
        }

        // dd($student);
        $month          = $request->month;
        $studentSession = $request->student_id ? Student::where('id', $student_id)->first()->session_id : null;
        $students       = Student::with('stuclass')->where('etim', 2)
            ->where(function ($query) {
                $query->where('status', 1)
                    ->orWhereNull('status');
            })->get();
        $funds   = Fund::all();
        $classes = StudentClass::all();
        return view('dashboard.payments.create', compact('students', 'funds', 'classes', 'request', 'studentSession'));
    }

    public function PaymentMultiMonthCreate(Request $request)
    {
        // dd($request->all());
        $class_id       = $request->class_id;
        $student_id     = $request->student_id;
        $month          = $request->month;
        $studentSession = $request->student_id ? Student::where('id', $student_id)->first()->session_id : null;
        $students       = Student::with('stuclass')->where('etim', 2)
            ->where(function ($query) {
                $query->where('status', 1)
                    ->orWhereNull('status');
            })->get();
        $funds   = Fund::all();
        $classes = StudentClass::all();
        return view('dashboard.payments.create_multi', compact('students', 'funds', 'classes', 'request', 'studentSession'));
    }

    public function PaymentPrint($id)
    {
        $data      = MoneyReceipt::with('students')->findOrFail($id);
        $donarId   = $data->donar_id;
        $donarName = null;
        if ($donarId) {
            $donarName = Donar::findOrFail($donarId)->name;
        }

        $fundDetailTypes = DB::table('fund_details_type as fdt')
            ->where('fdt.student_payment_id', $id)
            ->select('fdt.*')
            ->get();

        // dd($data, $fundDetailTypes);

        foreach ($fundDetailTypes as $fundDetail) {
            $fundDetail->fundDetails = DB::table('fund_details')
                ->join('students', 'students.id', '=', 'fund_details.student_id')
                ->join('student_fees', 'student_fees.id', '=', 'fund_details.fee_id')
                ->join("funds", "funds.id", "=", "fund_details.fund_id")
                ->where('student_payment_id', $id)
                ->where('fund_detail_type_id', $fundDetail->id)
                ->select(
                    'fund_details.*',
                    'student_fees.name as fee_name',
                    'student_fees.amount as fee_amount',
                    'students.name as student_name',
                    'funds.name as fund_name'
                )
                ->get();
        }

        $design = InvoiceDesign::first();

        // dd($data, $fundDetailTypes);
        return view('dashboard.payments.print', compact(
            'design',
            'data',
            'donarName',
            'fundDetailTypes'

        ));
    }

    public function PaymentPrintMulti($id)
    {
        $multiMonthData = DB::table('money_receipt_multiple')
            ->join('students', 'students.id', '=', 'money_receipt_multiple.student_id')
            ->join('student_classes', 'student_classes.id', '=', 'money_receipt_multiple.class_id')
            ->whereNull('money_receipt_multiple.deleted_at')
            ->whereNull('students.deleted_at')
            ->whereNull('student_classes.deleted_at')
            ->select('money_receipt_multiple.*',
                'students.name as student_name',
                'students.register_no as register_no',
                'student_classes.name as class_name')
            ->where('money_receipt_multiple.id', $id)->first();

        // dd($multiMonthData,$id);

        $moneyReceipts = MoneyReceipt::with('students')->where('money_receipt_multiple_id', $id)->get();
        // $fundDetailTypes = [];
        foreach ($moneyReceipts as $moneyReceipt) {
            $moneyReceipt->fundDetailTypes = DB::table('fund_details_type as fdt')
                ->where('fdt.student_payment_id', $moneyReceipt->id)
                ->select('fdt.*')
                ->get();

            foreach ($moneyReceipt->fundDetailTypes as $fundDetail) {
                $fundDetail->fundDetails = DB::table('fund_details')
                    ->join('students', 'students.id', '=', 'fund_details.student_id')
                    ->join('student_fees', 'student_fees.id', '=', 'fund_details.fee_id')
                    ->join("funds", "funds.id", "=", "fund_details.fund_id")
                    ->where('student_payment_id', $moneyReceipt->id)
                    ->where('fund_detail_type_id', $fundDetail->id)
                    ->select(
                        'fund_details.*',
                        'student_fees.name as fee_name',
                        'student_fees.amount as fee_amount',
                        'students.name as student_name',
                        'funds.name as fund_name'
                    )
                    ->get();
            }
        }

        $design = InvoiceDesign::first();
        // dd($id, $multiMonthData, $moneyReceipts);
        return view('dashboard.payments.print_multi', compact(
            'design',
            'multiMonthData',
            'moneyReceipts'
        ));

    }

    public function PaymentPrint_old($id)
    {
        $data = FundDetail::with('students:id,name,roll_no,register_no,class_id')
            ->where('student_payment_id', $id)
            ->firstOrFail();

        $month = $data->month;

        $donar_id = $data->donar_id;

        $fundQuery = FundDetail::where('student_id', $data->student_id)
            ->where('month', $month)
            ->whereIn('type', [6, 16])
            ->where('payment_type', 1)
            ->select(
                DB::raw('SUM(amount) as total_received'),
                DB::raw('SUM(discount) as total_discount')
            );

        if ($donar_id) {
            $fundQuery->where('donar_id', $donar_id);
        }

        $total = $fundQuery->first();

        $totalReceived = $total->total_received ?? 0;
        $totalDiscount = $total->total_discount ?? 0;

        $academicFee     = [];
        $academicAmount  = 0;
        $hostelFee       = [];
        $hostelAmount    = 0;
        $onabashikAmount = 0;
        $onabashikFee    = [];
        $otherFee        = [];
        $otherAmount     = 0;
        $specialFee      = [];
        $specialAmount   = 0;

        $type = Student::findOrFail($data->student_id);

        if ($type->student_type == 2) {
            $hostelAmount = FeeAssignDetail::with('class:id,name', 'studentsession:id,name')
                ->where('type', 2)
                ->where('class_id', $type->class_id)
                ->where('session_id', $type->session_id)
                ->where('month', $month)
                ->sum('amount');

            $hostelFee = FeeAssignDetail::with('fees') // fee() relation must return ->belongsTo(Fee::class)
                ->where('type', 2)
                ->where('class_id', $type->class_id)
                ->where('session_id', $type->session_id)
                ->where('month', $month)
                ->get()
                ->groupBy('fee_id')
                ->map(function ($items, $fee_id) {
                    $fee_name = optional($items->first()->fees)->name; // fee name from relation
                    $amount   = $items->sum('amount');

                    return [
                        'fee_id'   => $fee_id,
                        'fee_name' => $fee_name,
                        'amount'   => $amount,
                    ];
                })
                ->values();
        }

        if ($type->student_type == 1) {
            $onabashikAmount = FeeAssignDetail::with('class:id,name', 'studentsession:id,name')
                ->where('type', 4)
                ->where('class_id', $type->class_id)
                ->where('session_id', $type->session_id)
                ->where('month', $month)
                ->sum('amount');

            $onabashikFee = FeeAssignDetail::with('fees') // fee() relation must return ->belongsTo(Fee::class)
                ->where('type', 4)
                ->where('class_id', $type->class_id)
                ->where('session_id', $type->session_id)
                ->where('month', $month)
                ->get()
                ->groupBy('fee_id')
                ->map(function ($items, $fee_id) {
                    $fee_name = optional($items->first()->fees)->name; // fee name from relation
                    $amount   = $items->sum('amount');

                    return [
                        'fee_id'   => $fee_id,
                        'fee_name' => $fee_name,
                        'amount'   => $amount,
                    ];
                })
                ->values();
        }

        $academicAmount = FeeAssignDetail::with('class:id,name', 'studentsession:id,name')
            ->where('type', 1)
            ->where('class_id', $type->class_id)
            ->where('session_id', $type->session_id)
            ->where('month', $month)
            ->sum('amount');

        $academicFee = FeeAssignDetail::with('fees') // fee() relation must return ->belongsTo(Fee::class)
            ->where('type', 1)
            ->where('class_id', $type->class_id)
            ->where('session_id', $type->session_id)
            ->where('month', $month)
            ->get()
            ->groupBy('fee_id')
            ->map(function ($items, $fee_id) {
                $fee_name = optional($items->first()->fees)->name; // fee name from relation
                $amount   = $items->sum('amount');

                return [
                    'fee_id'   => $fee_id,
                    'fee_name' => $fee_name,
                    'amount'   => $amount,
                ];
            })
            ->values();

        $otherAmount = FeeAssignDetail::with('class:id,name', 'studentsession:id,name')
            ->where('type', 3)
            ->where('class_id', $type->class_id)
            ->where('session_id', $type->session_id)
            ->where('month', $month)
            ->sum('amount');

        $otherFee = FeeAssignDetail::with('fees') // fee() relation must return ->belongsTo(Fee::class)
            ->where('type', 3)
            ->where('class_id', $type->class_id)
            ->where('session_id', $type->session_id)
            ->where('month', $month)
            ->get()
            ->groupBy('fee_id')
            ->map(function ($items, $fee_id) {
                $fee_name = optional($items->first()->fees)->name; // fee name from relation
                $amount   = $items->sum('amount');

                return [
                    'fee_id'   => $fee_id,
                    'fee_name' => $fee_name,
                    'amount'   => $amount,
                ];
            })
            ->values(); // to reset keys

        $specialAmount = DB::table('fee_assign_details as fd')
            ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id') // ✅ join condition
            ->whereNull('fd.deleted_at')                                                           // ✅ soft delete check
            ->whereNull('fds.deleted_at')                                                          // ✅ soft delete check
            ->where('fd.type', 5)
            ->where('fds.student_id', $data->student_id)
            ->where('fd.session_id', $type->session_id)
            ->where('fd.class_id', $type->class_id)
            ->where('fd.month', $month)
            ->sum('fds.amount');

        $specialFee = DB::table('fee_assign_details as fad')
            ->join('feeassign_details_students as fads', 'fads.fee_assign_details_id', '=', 'fad.id')
            ->join('student_fees as fees', 'fees.id', '=', 'fad.fee_id')
            ->whereNull('fad.deleted_at')  // ✅ soft delete check
            ->whereNull('fads.deleted_at') // ✅ soft delete check
            ->whereNull('fees.deleted_at') // ✅ soft delete check
            ->where('fad.type', 5)
            ->where('fad.class_id', $type->class_id)
            ->where('fad.session_id', $type->session_id)
            ->where('fad.month', $month)
            ->where('fads.student_id', $data->student_id)
            ->select('fad.fee_id', 'fees.name as fee_name', 'fads.amount')
            ->get()
            ->groupBy('fee_id')
            ->map(function ($items, $fee_id) {
                $fee_name = $items->first()->fee_name;
                $amount   = $items->sum('amount');

                return [
                    'fee_id'   => $fee_id,
                    'fee_name' => $fee_name,
                    'amount'   => $amount,
                ];
            })
            ->values();

        // dd($otherFee);
        $totalAmount = $academicAmount + $hostelAmount + $otherAmount + $onabashikAmount + $specialAmount - $totalDiscount;
        // dd($totalAmount);

        $design = InvoiceDesign::first();
        return view('dashboard.payments.print', compact(
            'design',
            'data',
            'academicAmount',
            'hostelAmount',
            'onabashikAmount',
            'otherAmount',
            'totalAmount',
            'totalReceived',
            'totalDiscount',
            'otherFee',
            'academicFee',
            'onabashikFee',
            'hostelFee',
            'specialFee',
            'specialAmount'

        ));
    }

    public function PaymentShow($id)
    {
        $data = DB::table('money_receipts as mr')
            ->join('students as s', 's.id', '=', 'mr.student_id')
            ->join('student_classes as sc', 'sc.id', '=', 'mr.class_id')
            ->join('student_sessions as ss', 'ss.id', '=', 's.session_id')
            ->where('mr.id', $id)
            ->select(
                'mr.id as id',
                'mr.receipt_no as receipt_no',
                'mr.student_id as student_id',
                's.name as student_name',
                'mr.date as date',
                'mr.month as month',
                'sc.name as class_name',
                'ss.name as session_name',

                "mr.amount as receive_amount",
                "mr.total as total_amount",
                "mr.discount as discount_amount",
                "mr.due as due_amount",
            )->first();

        $fundDetailType = DB::table('fund_details_type as fdt')
            ->join('students as s', 's.id', '=', 'fdt.student_id')
            ->join('student_classes as sc', 'sc.id', '=', 'fdt.class_id')
            ->join('student_sessions as ss', 'ss.id', '=', 's.session_id')
            ->where('fdt.student_id', $data->student_id)
            ->where('fdt.student_payment_id', $data->id)
            ->select(
                'fdt.*',
                // 'fd.*',
                's.name as student_name',
                'sc.name as class_name',
                'ss.name as session_name',
            )
            ->get();

        foreach ($fundDetailType as $fdt) {
            $fdt->fundDetail = DB::table('fund_details as fd')
                ->join('funds as f', 'f.id', '=', 'fd.fund_id')
                ->join('student_fees as sf', 'sf.id', '=', 'fd.fee_id')
                ->where('fd.fund_detail_type_id', $fdt->id)
                ->select(
                    // 'fd.*',
                    'fd.id as fd_id',
                    'fd.amount as amount',
                    'sf.name as fee_name',
                    'f.name as fund_name',
                )
                ->get()
                ->toArray();
        }
        // dd($fundDetailType);

        return view('dashboard.payments.show', compact('data', 'fundDetailType'));
    }

    public function PaymentEdit($id)
    {
        $students = Student::with('stuclass')->where('etim', 2)
            ->where(function ($query) {
                $query->where('status', 1)
                    ->orWhereNull('status');
            })->get();
        $funds = Fund::all();
        $data  = DB::table('money_receipts as mr')
            ->join('students as s', 's.id', '=', 'mr.student_id')
            ->join('student_classes as sc', 'sc.id', '=', 'mr.class_id')
            ->join('student_sessions as ss', 'ss.id', '=', 's.session_id')
            ->where('mr.id', $id)
            ->select(
                'mr.id as id',
                'mr.receipt_no as receipt_no',
                'mr.student_id as student_id',
                's.name as student_name',
                'mr.date as date',
                'mr.month as month',
                "sc.id as class_id",
                'sc.name as class_name',
                'ss.name as session_name',

                "mr.amount as receive_amount",
                "mr.total as total_amount",
                "mr.discount as discount_amount",
                "mr.due as due_amount",
                "mr.current_receive as current_receive",
            )->first();

        $fundDetailType = DB::table('fund_details_type as fdt')
            ->join('students as s', 's.id', '=', 'fdt.student_id')
            ->join('student_classes as sc', 'sc.id', '=', 'fdt.class_id')
            ->join('student_sessions as ss', 'ss.id', '=', 's.session_id')
            ->where('fdt.student_id', $data->student_id)
            ->where('fdt.student_payment_id', $data->id)
            ->select(
                'fdt.*',
                // 'fd.*',
                's.name as student_name',
                'sc.name as class_name',
                'ss.name as session_name',
            )
            ->get();

        foreach ($fundDetailType as $fdt) {
            $fdt->details = DB::table('fund_details as fd')
                ->join('funds as f', 'f.id', '=', 'fd.fund_id')
                ->join('student_fees as sf', 'sf.id', '=', 'fd.fee_id')
                ->where('fd.fund_detail_type_id', $fdt->id)
                ->select(
                    // 'fd.*',
                    'fd.id as fd_id',
                    'fd.amount as amount',
                    'sf.id as fee_id',
                    'sf.name as fee_name',
                    'f.id as fund_id',
                    'f.name as fund_name',
                )
                ->get()
                ->toArray();
        }

        // dd($data, $fundDetailType);

        $funds = Fund::select('id', 'name')->whereNull('deleted_at')->get();

        // dd($data, $fundDetailType,$funds);
        $dataFund = FundDetail::with('students')->where('student_id', $data->student_id)->where('student_payment_id', $id)->first();
        return view('dashboard.payments.edit', compact('students', 'funds', 'data', 'dataFund', 'fundDetailType'));
    }

    public function StudentPayment(Request $request)
    {

        $result = $this->getSingleMonthBreakdown($request);

        return response()->json($result);
    }

    public function StudentPaymentMulti(Request $request)
    {

        $months       = $request->month;
        $responseList = [];

        foreach ($months as $month) {

            // dd($month, $request->all());
            $request->merge(['month' => $month]);

            $result = $this->getSingleMonthBreakdown($request);

            $responseList[$month] = $result;
        }

        // dd($request->all(), $responseList);
        return response()->json([
            'multiMonthBreakdown' => $responseList,
        ]);

    }

    protected function getSingleMonthBreakdown(Request $request)
    {
        $month     = trim($request->month); // expects 'YYYY-MM'
        $donar_id  = $request->donar_id;
        $donarName = '';

        if ($donar_id) {
            $donar     = Donar::find($donar_id);
            $donarName = $donar?->name ?? '';
        }

        $funds = Fund::all();

        // ---------- Student & join_date guard ----------
        /** @var \App\Models\Student $student */
        $student = Student::select('id', 'student_type', 'join_date', 'session_id', 'class_id')
            ->findOrFail($request->student_id);

        $targetYm = substr($month, 0, 7); // 'YYYY-MM'
        $joinYm   = $student->join_date ? Carbon::parse($student->join_date)->format('Y-m') : null;

        // যদি join_date এই মাসের পরের মাস হয় => এই মাসে কিছু assign/collect দেখাবে না
        if ($joinYm && $joinYm > $targetYm) {
            return 'not-assign';
        }

        // ---------- OLD flow totals (amount + discount) ----------
        $fundQuery = FundDetail::where('student_id', $student->id)
            ->where('month', $month)
            ->whereIn('type', [6, 16])
            ->where('payment_type', 1)
            ->whereNull('deleted_at')
            ->whereNull('receipt_type')
            ->whereNull('fund_detail_type_id')
            ->whereNull('fee_id')
            ->select(
                DB::raw('SUM(amount) as total_received'),
                DB::raw('SUM(discount) as total_discount')
            );

        if ($donar_id) {
            $fundQuery->where('donar_id', $donar_id);
        }

        $total         = $fundQuery->first();
        $totalReceived = (float) ($total->total_received ?? 0);
        $totalDiscount = (float) ($total->total_discount ?? 0);

        // ---------- Buckets ----------
        $academicAmount                 = 0;
        $academicPreviousReceiveAmount  = 0;
        $academicPreviousDiscountAmount = 0;

        $hostelAmount                 = 0;
        $hostelPreviousReceiveAmount  = 0;
        $hostelPreviousDiscountAmount = 0;
        $hostelDueAmount              = 0;

        $onabashikFee                    = 0;
        $onabashikPreviousReceiveAmount  = 0;
        $onabashikPreviousDiscountAmount = 0;
        $onabashikDueAmount              = 0;

        $otherAmount                 = 0;
        $otherPreviousReceiveAmount  = 0;
        $otherPreviousDiscountAmount = 0;

        $specialFee                    = 0;
        $specialPreviousReceiveAmount  = 0;
        $specialPreviousDiscountAmount = 0;

        $hostelDetails    = collect();
        $onabashikDetails = collect();
        $academicDetails  = collect();
        $otherDetails     = collect();
        $specialDetails   = collect();

        // ---------- HOSTEL (type=2) ----------
        if ((int) $student->student_type === 2) {
            $hostelAmount = FeeAssignDetail::where('type', 2)
                ->where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->where('month', $month)
                ->whereNull('deleted_at')
                ->sum('amount');

            $hostelPreviousReceiveAmount = DB::table('fund_details_type')
                ->join('students', 'students.id', '=', 'fund_details_type.student_id')
                ->join('student_sessions', 'student_sessions.id', '=', 'students.session_id')
                ->where('student_id', $student->id)
                ->where('fund_details_type.fee_type', 'hostel')
                ->where('fund_details_type.month', $month)
                ->where('fund_details_type.class_id', $request->class_id)
                ->where('students.session_id', $request->session_id)
                ->whereNull('fund_details_type.deleted_at')
                ->sum('fund_details_type.receive');

            $hostelPreviousDiscountAmount = DB::table('fund_details_type')
                ->join('students', 'students.id', '=', 'fund_details_type.student_id')
                ->join('student_sessions', 'student_sessions.id', '=', 'students.session_id')
                ->where('student_id', $student->id)
                ->where('fund_details_type.fee_type', 'hostel')
                ->where('fund_details_type.month', $month)
                ->where('fund_details_type.class_id', $request->class_id)
                ->where('students.session_id', $request->session_id)
                ->whereNull('fund_details_type.deleted_at')
                ->sum('fund_details_type.discount');

            $hostelDueAmount = $hostelAmount - ($hostelPreviousReceiveAmount + $hostelPreviousDiscountAmount);

            $hostelDetails = FeeAssignDetail::with('fees')
                ->where('type', 2)
                ->where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->where('month', $month)
                ->whereNull('deleted_at')
                ->get()
                ->groupBy('fee_id')
                ->map(function ($items, $fee_id) {
                    $fee_name = optional($items->first()->fees)->name;
                    $amount   = $items->sum('amount');
                    $fund_id  = optional($items->first()->fees)->fund_id;

                    return [
                        'fee_id'   => $fee_id,
                        'fee_name' => $fee_name,
                        'amount'   => $amount,
                        'fund_id'  => $fund_id,
                    ];
                })
                ->values();
        }

        // ---------- ONABASHIK (type=4) ----------
        if ((int) $student->student_type === 1) {
            $onabashikFee = FeeAssignDetail::where('type', 4)
                ->where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->where('month', $month)
                ->whereNull('deleted_at')
                ->sum('amount');

            $onabashikPreviousReceiveAmount = DB::table('fund_details_type')
                ->join('students', 'students.id', '=', 'fund_details_type.student_id')
                ->join('student_sessions', 'student_sessions.id', '=', 'students.session_id')
                ->where('student_id', $student->id)
                ->where('fund_details_type.fee_type', 'onabashik')
                ->where('fund_details_type.month', $month)
                ->where('fund_details_type.class_id', $request->class_id)
                ->where('students.session_id', $request->session_id)
                ->whereNull('fund_details_type.deleted_at')
                ->sum('fund_details_type.receive');

            $onabashikPreviousDiscountAmount = DB::table('fund_details_type')
                ->join('students', 'students.id', '=', 'fund_details_type.student_id')
                ->join('student_sessions', 'student_sessions.id', '=', 'students.session_id')
                ->where('student_id', $student->id)
                ->where('fund_details_type.fee_type', 'onabashik')
                ->where('fund_details_type.month', $month)
                ->where('fund_details_type.class_id', $request->class_id)
                ->where('students.session_id', $request->session_id)
                ->whereNull('fund_details_type.deleted_at')
                ->sum('fund_details_type.discount');

            $onabashikDueAmount = $onabashikFee - ($onabashikPreviousReceiveAmount + $onabashikPreviousDiscountAmount);

            $onabashikDetails = FeeAssignDetail::with('fees')
                ->where('type', 4)
                ->where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->where('month', $month)
                ->whereNull('deleted_at')
                ->get()
                ->groupBy('fee_id')
                ->map(function ($items, $fee_id) {
                    $fee_name = optional($items->first()->fees)->name;
                    $amount   = $items->sum('amount');
                    $fund_id  = optional($items->first()->fees)->fund_id;

                    return [
                        'fee_id'   => $fee_id,
                        'fee_name' => $fee_name,
                        'amount'   => $amount,
                        'fund_id'  => $fund_id,
                    ];
                })
                ->values();
        }

        // ---------- ACADEMIC (type=1) ----------
        $academicAmount = FeeAssignDetail::where('type', 1)
            ->where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->where('month', $month)
            ->whereNull('deleted_at')
            ->sum('amount');

        $academicPreviousReceiveAmount = DB::table('fund_details_type')
            ->join('students', 'students.id', '=', 'fund_details_type.student_id')
            ->join('student_sessions', 'student_sessions.id', '=', 'students.session_id')
            ->where('student_id', $student->id)
            ->where('fund_details_type.fee_type', 'academic')
            ->where('fund_details_type.month', $month)
            ->where('fund_details_type.class_id', $request->class_id)
            ->where('students.session_id', $request->session_id)
            ->whereNull('fund_details_type.deleted_at')
            ->sum('fund_details_type.receive');

        $academicPreviousDiscountAmount = DB::table('fund_details_type')
            ->join('students', 'students.id', '=', 'fund_details_type.student_id')
            ->join('student_sessions', 'student_sessions.id', '=', 'students.session_id')
            ->where('student_id', $student->id)
            ->where('fund_details_type.fee_type', 'academic')
            ->where('fund_details_type.month', $month)
            ->where('fund_details_type.class_id', $request->class_id)
            ->where('students.session_id', $request->session_id)
            ->whereNull('fund_details_type.deleted_at')
            ->sum('fund_details_type.discount');

        $academicDueAmount = $academicAmount - ($academicPreviousReceiveAmount + $academicPreviousDiscountAmount);

        $academicDetails = FeeAssignDetail::with('fees')
            ->where('type', 1)
            ->where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->where('month', $month)
            ->whereNull('deleted_at')
            ->get()
            ->groupBy('fee_id')
            ->map(function ($items, $fee_id) {
                $fee_name = optional($items->first()->fees)->name;
                $amount   = $items->sum('amount');
                $fund_id  = optional($items->first()->fees)->fund_id;

                return [
                    'fee_id'   => $fee_id,
                    'fee_name' => $fee_name,
                    'amount'   => $amount,
                    'fund_id'  => $fund_id,
                ];
            })
            ->values();

        // ---------- OTHER (type=3) ----------
        $otherAmount = FeeAssignDetail::where('type', 3)
            ->where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->where('month', $month)
            ->whereNull('deleted_at')
            ->sum('amount');

        $otherPreviousReceiveAmount = DB::table('fund_details_type')
            ->join('students', 'students.id', '=', 'fund_details_type.student_id')
            ->join('student_sessions', 'student_sessions.id', '=', 'students.session_id')
            ->where('student_id', $student->id)
            ->where('fund_details_type.fee_type', 'other')
            ->where('fund_details_type.month', $month)
            ->where('fund_details_type.class_id', $request->class_id)
            ->where('students.session_id', $request->session_id)
            ->whereNull('fund_details_type.deleted_at')
            ->sum('fund_details_type.receive');

        $otherPreviousDiscountAmount = DB::table('fund_details_type')
            ->join('students', 'students.id', '=', 'fund_details_type.student_id')
            ->join('student_sessions', 'student_sessions.id', '=', 'students.session_id')
            ->where('student_id', $student->id)
            ->where('fund_details_type.fee_type', 'other')
            ->where('fund_details_type.month', $month)
            ->where('fund_details_type.class_id', $request->class_id)
            ->where('students.session_id', $request->session_id)
            ->whereNull('fund_details_type.deleted_at')
            ->sum('fund_details_type.discount');

        $receivedAmount = ($otherPreviousReceiveAmount ?? 0) + ($otherPreviousDiscountAmount ?? 0);
        $otherDueAmount = $otherAmount - $receivedAmount;

        $otherDetails = FeeAssignDetail::with('fees')
            ->where('type', 3)
            ->where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->where('month', $month)
            ->whereNull('deleted_at')
            ->get()
            ->groupBy('fee_id')
            ->map(function ($items, $fee_id) {
                $fee_name = optional($items->first()->fees)->name;
                $amount   = $items->sum('amount');
                $fund_id  = optional($items->first()->fees)->fund_id;

                return [
                    'fee_id'   => $fee_id,
                    'fee_name' => $fee_name,
                    'amount'   => $amount,
                    'fund_id'  => $fund_id,
                ];
            })
            ->values();

        // ---------- SPECIAL (type=5) ----------
        $specialFee = DB::table('fee_assign_details as fd')
            ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id')
            ->whereNull('fd.deleted_at')
            ->whereNull('fds.deleted_at')
            ->where('fd.type', 5)
            ->where('fds.student_id', $student->id)
            ->where('fd.session_id', $request->session_id)
            ->where('fd.class_id', $request->class_id)
            ->where('fd.month', $month) // keep fd.month for consistency
            ->sum('fds.amount');

        $specialPreviousReceiveAmount = DB::table('fund_details_type')
            ->join('students', 'students.id', '=', 'fund_details_type.student_id')
            ->join('student_sessions', 'student_sessions.id', '=', 'students.session_id')
            ->where('student_id', $student->id)
            ->where('fund_details_type.fee_type', 'special')
            ->where('fund_details_type.month', $month)
            ->where('fund_details_type.class_id', $request->class_id)
            ->where('students.session_id', $request->session_id)
            ->whereNull('fund_details_type.deleted_at')
            ->sum('fund_details_type.receive');

        $specialPreviousDiscountAmount = DB::table('fund_details_type')
            ->join('students', 'students.id', '=', 'fund_details_type.student_id')
            ->join('student_sessions', 'student_sessions.id', '=', 'students.session_id')
            ->where('student_id', $student->id)
            ->where('fund_details_type.fee_type', 'special')
            ->where('fund_details_type.month', $month)
            ->where('fund_details_type.class_id', $request->class_id)
            ->where('students.session_id', $request->session_id)
            ->whereNull('fund_details_type.deleted_at')
            ->sum('fund_details_type.discount');

        $specialReceivedAmount = ($specialPreviousReceiveAmount ?? 0) + ($specialPreviousDiscountAmount ?? 0);
        $specialDueAmount      = $specialFee - $specialReceivedAmount;

        if ($specialFee > 0) {
            $specialDetails = DB::table('fee_assign_details as fd')
                ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id')
                ->join("student_fees as sf", 'sf.id', '=', 'fd.fee_id')
                ->where('fd.type', 5)
                ->where('fds.student_id', $student->id)
                ->where('fd.session_id', $request->session_id)
                ->where('fd.class_id', $request->class_id)
                ->where('fd.month', $month) // use fd.month for consistency
                ->whereNull('fds.deleted_at')
                ->whereNull('fd.deleted_at')
                ->select('fd.fee_id', 'fds.amount', 'sf.name as fee_name', 'sf.fund_id')
                ->get();
        }

        // ---------- Totals ----------
        $totalAmount                 = $academicAmount + $hostelAmount + $otherAmount + $onabashikFee + $specialFee;
        $totalPreviousReceiveAmount  = $academicPreviousReceiveAmount + $hostelPreviousReceiveAmount + $otherPreviousReceiveAmount + $onabashikPreviousReceiveAmount + $specialPreviousReceiveAmount;
        $totalPreviousDiscountAmount = $academicPreviousDiscountAmount + $hostelPreviousDiscountAmount + $otherPreviousDiscountAmount + $onabashikPreviousDiscountAmount + $specialPreviousDiscountAmount;
        $totalReceivedAmount         = $totalPreviousReceiveAmount + $totalPreviousDiscountAmount;
        $totalDueAmount              = $totalAmount - $totalReceivedAmount;

        // কোনো টাইপই assign না থাকলে
        if (
            $specialDetails->isEmpty()
            && $academicDetails->isEmpty()
            && $hostelDetails->isEmpty()
            && $otherDetails->isEmpty()
            && $onabashikDetails->isEmpty()
        ) {
            return 'not-assign';
        }

        if ($totalDueAmount == 0) {
            return 'paid';
        }

        // ---------- helpers ----------
        $get = function ($x, $key, $default = null) {
            return is_array($x) ? ($x[$key] ?? $default) : ($x->$key ?? $default);
        };

        $renderFeeList = function ($details, $funds, $namePrefix, $isDisabled, $studentId, $month, $totalAmount, $previousReceiveAmount, $previousDiscountAmount, $dueAmount, $typeName) use ($get) {

            $html = '<table class="table" style="margin-left:20px; width:max-content;">';
            $html .= '  <thead>
                        <tr>
                          <th style="width:5%;">Sl.</th>
                          <th style="width:20%; text-align:left;">Fee Name</th>
                          <th style="width:10%; text-align:center;">Fund</th>
                          <th style="width:10%; text-align:center;">Amount</th>
                          <th style="width:10%; text-align:center;">P. Receive</th>
                          <th style="width:10%; text-align:center;">P. Discount</th>
                          <th style="width:10%; text-align:center;">Paid</th>
                          <th style="width:10%; text-align:center;">Discount</th>
                          <th style="width:10%; text-align:center;">Due</th>
                        </tr>
                      </thead>';
            $html .= '  <tbody>';

            foreach ($details as $i => $item) {
                $feeName = $get($item, 'fee_name', '');
                $amount  = (float) $get($item, 'amount', 0);
                $fundId  = $get($item, 'fund_id');
                $feeId   = $get($item, 'fee_id', $i);

                // prev (per fee) from fund_details (new-format detail rows)
                $previousReceive = DB::table('fund_details')
                    ->where('student_id', $studentId)
                    ->where('fee_id', $feeId)
                    ->where('month', $month)
                    ->whereNull('deleted_at')
                    ->whereNotNull('fund_detail_type_id')
                    ->sum('amount');

                $previousDiscount = DB::table('fund_details')
                    ->where('student_id', $studentId)
                    ->where('fee_id', $feeId)
                    ->where('month', $month)
                    ->whereNull('deleted_at')
                    ->whereNotNull('fund_detail_type_id')
                    ->sum('discount');

                $totalReceived = (float) ($previousReceive ?? 0) + (float) ($previousDiscount ?? 0);
                $due           = max(0, $amount - $totalReceived);

                $checkboxId = $namePrefix . 'fee_' . $feeId;

                $html .= '    <tr>';
                $html .= '      <td>
                              <input type="hidden" name="' . $namePrefix . 'fee_amount[' . $month . '][' . $feeId . ']" value="' . htmlspecialchars((string) $amount) . '">
                              <input type="checkbox"
                                     id="' . $checkboxId . '"
                                     name="' . $namePrefix . 'fee_ids[' . $month . '][' . $feeId . ']"
                                     data-month="' . $month . '"
                                     value="' . htmlspecialchars((string) $amount) . '"
                                     class="fee_checked"
                                     ' . ($isDisabled ? 'disabled' : 'checked') . '
                                     ' . ($due > 0 ? '' : 'disabled') . '>
                            </td>';

                $html .= '      <td>' . htmlspecialchars((string) $feeName) . '</td>';

                $isSelectRequired = ! ($isDisabled || $due <= 0);
                $html .= '      <td style="text-align:center;">
                              <select id="' . $namePrefix . '" name="' . $namePrefix . '[' . $month . '][' . $feeId . ']" class="form-control" style="width:100%;" ' . ($isSelectRequired ? 'required' : '') . '>
                                <option value="">Select fund</option>';
                foreach ($funds as $fund) {
                    $selected = ((string) $fund->id === (string) $fundId) ? ' selected' : '';
                    $html .= '        <option value="' . htmlspecialchars((string) $fund->id) . '"' . $selected . '>' . htmlspecialchars((string) $fund->name) . '</option>';
                }
                $html .= '      </select>
                           </td>';

                $html .= '      <td style="text-align:center;" class="feeAmount" data-amount="' . $amount . '">' . number_format($amount, 2) . '</td>';
                $html .= '      <td style="text-align:center;" class="feePreviousPaid" data-amount="' . $previousReceive . '">' . number_format((float) $previousReceive, 2) . '</td>';
                $html .= '      <td style="text-align:center;" class="feePreviousDiscount" data-amount="' . $previousDiscount . '">' . number_format((float) $previousDiscount, 2) . '</td>';

                $html .= '      <td style="text-align:center;">
                              <input type="text" class="form-control feePaid" id="' . $namePrefix . 'fee_paid[' . $month . '][' . $feeId . ']"
                                     placeholder="Paid" name="' . $namePrefix . 'fee_paid[' . $month . '][' . $feeId . ']" value="' . $due . '">
                            </td>';

                $html .= '      <td style="text-align:center;">
                              <input type="text" class="form-control feeDiscount" id="' . $namePrefix . 'fee_discount[' . $month . '][' . $feeId . ']"
                                     placeholder="Discount" name="' . $namePrefix . 'fee_discount[' . $month . '][' . $feeId . ']" value="0">
                            </td>';

                $html .= '      <td style="text-align:center;">
                              <input type="text" class="form-control feeDue" id="' . $namePrefix . 'fee_due[' . $month . '][' . $feeId . ']"
                                     placeholder="Due" name="' . $namePrefix . 'fee_due[' . $month . '][' . $feeId . ']" value="0" readonly>
                            </td>';

                $html .= '    </tr>';
            }

            $html .= '  </tbody>';

            // footer totals (row-level)
            $html .= '  <tfoot>
                      <tr>
                        <td></td>
                        <td colspan="2" style="text-align:right;">Total:</td>
                        <td id="' . $typeName . '_amount_total" style="text-align:center;">' . number_format((float) $totalAmount, 2) . '</td>
                        <input type="hidden" name="' . $typeName . '_amount_total[' . $month . ']" id="' . $typeName . '_amount_total" value="' . htmlspecialchars((string) $totalAmount) . '">

                        <td style="text-align:center;" id="' . $typeName . '_receive_total">' . htmlspecialchars((string) $previousReceiveAmount) . '</td>
                        <input type="hidden" name="' . $typeName . '_previous_receive[' . $month . ']" id="' . $typeName . '_previous_receive" value="' . htmlspecialchars((string) $previousReceiveAmount) . '">

                        <td style="text-align:center;" id="' . $typeName . '_discount_total">' . htmlspecialchars((string) $previousDiscountAmount) . '</td>
                        <input type="hidden" name="' . $typeName . '_previous_discount[' . $month . ']" id="' . $typeName . '_previous_discount" value="' . htmlspecialchars((string) $previousDiscountAmount) . '">

                        <td><input type="text" name="' . $typeName . '_fee_paid[' . $month . ']" class="rowFeesPaid form-control" id="' . $typeName . '_fee_paid" placeholder="' . ucfirst($typeName) . ' Fee Paid" value="' . htmlspecialchars((string) $dueAmount) . '" data-amount="' . htmlspecialchars((string) $dueAmount) . '" readonly></td>
                        <td><input type="text" name="' . $typeName . '_fee_discount[' . $month . ']" class="rowFeesDiscount form-control" id="' . $typeName . '_fee_discount" placeholder="' . ucfirst($typeName) . ' Fee Discount" value="0" readonly></td>
                        <td><input type="text" name="' . $typeName . '_fee_due[' . $month . ']" class="rowFeesDue form-control" id="' . $typeName . '_fee_due" placeholder="' . ucfirst($typeName) . ' Fee Due" value="0" readonly></td>
                      </tr>
                    </tfoot>';
            $html .= '</table>';

            return $html;
        };

        // -------- Build table rows --------
        $typeList = '';

        $feeTypes = [
            ['academic', $academicDetails, $academicAmount, $academicPreviousReceiveAmount, $academicPreviousDiscountAmount, $academicDueAmount],
            ['hostel', $hostelDetails, $hostelAmount, $hostelPreviousReceiveAmount, $hostelPreviousDiscountAmount, $hostelDueAmount],
            ['onabashik', $onabashikDetails, $onabashikFee, $onabashikPreviousReceiveAmount, $onabashikPreviousDiscountAmount, $onabashikDueAmount],
            ['other', $otherDetails, $otherAmount, $otherPreviousReceiveAmount, $otherPreviousDiscountAmount, $otherDueAmount],
            ['special', $specialDetails, $specialFee, $specialPreviousReceiveAmount, $specialPreviousDiscountAmount, $specialDueAmount],
        ];

        foreach ($feeTypes as $feeType) {
            [$typeName, $details, $amount, $previousReceiveAmount, $previousDiscountAmount, $dueAmount] = $feeType;

            if ((float) $amount === 0.0) {
                continue;
            }

            $banglaTypeName = match ($typeName) {
                'academic'  => 'একাডেমিক',
                'hostel'    => 'আবাসিক',
                'onabashik' => 'অনাবাসিক',
                'other'     => 'অন্যান্য',
                'special'   => 'স্পেশাল',
                default     => ucfirst($typeName),
            };

            $isDisabled = ((float) $amount === (float) ($previousReceiveAmount + $previousDiscountAmount));

            $typeList .= '<tr class="type-row">';
            $typeList .= '<input type="hidden" class="prev-receive" value="' . ($totalPreviousReceiveAmount) . '">';
            $typeList .= '<input type="hidden" class="prev-discount" value="' . ($totalPreviousDiscountAmount) . '">';
            $typeList .= '<td><input type="checkbox" name="rows[' . $month . '][' . $typeName . '][enabled]" class="row-enable" data-month="' . $month . '" value="1" ' . (! $isDisabled ? 'checked' : 'disabled') . '></td>';
            $typeList .= '<td>' . $banglaTypeName . '</td>';
            $typeList .= '<td class="fee-table">' .
                $renderFeeList($details, $funds, 'fund_id_' . $typeName, $isDisabled, $student->id, $month, $amount, $previousReceiveAmount, $previousDiscountAmount, $dueAmount, $typeName)
                . '</td>';
            $typeList .= '</tr>';
        }

        // যদি কোনো টাইপ নাই, treat as not-assign
        if (
            $specialDetails->isEmpty()
            && $academicDetails->isEmpty()
            && $hostelDetails->isEmpty()
            && $otherDetails->isEmpty()
            && $onabashikDetails->isEmpty()
        ) {
            return 'not-assign';
        }

        if (($totalAmount - ($totalPreviousReceiveAmount + $totalPreviousDiscountAmount)) <= 0) {
            return 'paid';
        }

        return [
            'typeList'         => $typeList,
            'academicAmount'   => (int) $academicAmount,
            'hostelAmount'     => (int) $hostelAmount,
            'onabashikFee'     => (int) $onabashikFee,
            'specialFee'       => (int) $specialFee,
            'otherAmount'      => (int) $otherAmount,
            'totalAmount'      => (int) ($academicAmount + $hostelAmount + $otherAmount + $onabashikFee + $specialFee),
            'totalReceived'    => (int) $totalReceived,
            'previousReceive'  => (int) ($academicPreviousReceiveAmount + $hostelPreviousReceiveAmount + $otherPreviousReceiveAmount + $onabashikPreviousReceiveAmount + $specialPreviousReceiveAmount),
            'previousDiscount' => (int) ($academicPreviousDiscountAmount + $hostelPreviousDiscountAmount + $otherPreviousDiscountAmount + $onabashikPreviousDiscountAmount + $specialPreviousDiscountAmount),
            'totalDiscount'    => (int) $totalDiscount,
            'donar_id'         => $donar_id,
            'donarName'        => $donarName,
        ];
    }

    public function PaymentCheckAvailableBalance(Request $request)
    {
        // dd($request->all());
        $currentAmount   = $request->currentAmount ?? 0;
        $currentDiscount = $request->currentDiscount ?? 0;
        $currentDue      = $request->currentDue ?? 0;
        $feeType         = $request->feeType;
        $month           = $request->month;
        $student_id      = $request->studentId;
        $totalAmount     = $request->totalAmount ?? 0;

        $fundDetailType = DB::table('fund_details_type as fdt')
            ->where('fee_type', $feeType)
            ->where('month', $month)
            ->where('student_id', $student_id)
            ->get();

        $received = $fundDetailType->sum('receive');
        $discount = $fundDetailType->sum('discount');

        $totalReceived   = $received + $discount;
        $currentReceived = $currentAmount + $currentDiscount;
        $overallReceived = $totalReceived + $currentReceived;

        $afterCalculateDue = $totalAmount - ($currentReceived + $currentDue);

        $amountWillReceive = $totalAmount - $overallReceived + $afterCalculateDue;

        return response()->json([
            'amountWillReceive' => $amountWillReceive,
            'overallReceived'   => $overallReceived,
            'totalRecived'      => $totalReceived,
        ]);
    }

    public function StudentSponsorAmount(Request $request)
    {
        $student    = Student::findOrFail($request->student_id);
        $class_id   = $student->class_id;
        $session_id = $student->session_id;

        $feesAmount    = FeeAssignDetail::where('class_id', $class_id)->where('session_id', $session_id)->get();
        $totalAmount   = $feesAmount->sum('amount');
        $totalMonths   = $feesAmount->groupBy('month')->count();
        $averageAmount = $totalMonths ? $totalAmount / $totalMonths : 0;

        return response()->json(['averageAmount' => round($averageAmount)]);
    }

    public function SponsorStudent(Request $request)
    {
        $students = StudentSponsor::with(['student:id,name,register_no,roll_no,class_id,session_id,status,monthly_dis'])
            ->where('donar_id', $request->donar_id)
            ->whereHas('student', function ($query) {
                $query->where('status', 1)->orWhereNull('status');
            })
            ->get();

        return response()->json($students);
    }

    public function GeneralFund(Request $request)
    {
        try {
            if ($request->ajax()) {
                // Only General funds (fund_type = 1) and SKIP Opening (payment_type=3)
                $query = FundDetail::with(['funds', 'donar'])
                    ->whereHas('funds', function ($q) {
                        $q->where('fund_type', 1);
                    })
                    ->whereIn('payment_type', [1, 2]); // 1=Income, 2=Expense (skip 3)

                // Fund filter (optional)
                if ($request->filled('fund_id')) {
                    $query->where('fund_id', $request->fund_id);
                }

                // Type filter (optional): only 1 or 2 allowed now
                if ($request->filled('payment_type') && in_array((int) $request->payment_type, [1, 2], true)) {
                    $query->where('payment_type', (int) $request->payment_type);
                }

                // Totals on the same filtered dataset (Opening excluded)
                $totalsQ      = (clone $query);
                $incomeTotal  = (clone $totalsQ)->where('payment_type', 1)->sum('amount');
                $expenseTotal = (clone $totalsQ)->where('payment_type', 2)->sum('amount');
                $balance      = $incomeTotal - $expenseTotal;

                // Table data
                $data = $query->orderByDesc('id')->get();

                return Datatables::of($data)
                    ->addColumn('date', function ($row) {
                        return $row->date ? \Carbon\Carbon::parse($row->date)->format('d-M-Y') : '';
                    })
                    ->addColumn('fund', function ($row) {
                        return optional($row->funds)->name ?? '--';
                    })
                    ->addColumn('donar', function ($row) {
                        $name  = $row->donar_id ? optional($row->donar)->name : ($row->name ?? '');
                        $phone = $row->donar_id ? optional($row->donar)->phone : ($row->phone ?? '');
                        return trim($name . '<br>' . $phone);
                    })
                    ->addColumn('type', function ($row) {
                        if ($row->payment_type == 1) {
                            return '<b>Income</b>';
                        }

                        if ($row->payment_type == 2) {
                            return '<b>Expense</b>';
                        }

                        return '';
                    })
                    ->addColumn('purpose', fn($row) => $row->purpose)
                    ->addColumn('amount', fn($row) => $row->amount ?? 0)
                    ->addColumn('action', function ($row) {
                        $details = '<a href="' . route('fund.collect.show', $row->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i> Details</a> ';
                        $edit    = '<a href="' . route('fund.collect.edit', $row->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i> Edit</a> ';
                        $print   = '<a href="' . route('fund.collect.print', $row->id) . '" class="btn btn-sm btn-primary" title="Print"><i class="fa fa-file"></i> Print</a> ';
                        $delete  = '';

                        if (method_exists($this, 'DeleteData') && $this->DeleteData()) {
                            $delete = '<a href="#" data-remote="' . route('fund.collect.destroy', $row->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i> Delete</a>';
                        }
                        // আপনার আগের কন্ডিশন রাখলে:
                        return ($row->type == 1) ? ($details . $edit . $print . $delete) : $details;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['date', 'fund', 'donar', 'type', 'purpose', 'amount', 'action'])
                    ->with([
                        'summary' => [
                            'income'  => (float) $incomeTotal,
                            'expense' => (float) $expenseTotal,
                            'balance' => (float) $balance,
                        ],
                    ])
                    ->toJson();
            }

            // Non-AJAX view data
            $invoice = InvoiceDesign::first();
            $funds   = Fund::where('fund_type', 1)->orderBy('name')->get(['id', 'name']);

            return view('dashboard.funds.general', compact('invoice', 'funds'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function GurabaFund(Request $request)
    {
        try {
            if ($request->ajax()) {
                // Guraba funds only (fund_type = 2) + SKIP Opening (payment_type=3)
                $query = \App\Models\FundDetail::with(['funds', 'donar'])
                    ->whereHas('funds', function ($q) {
                        $q->where('fund_type', 2);
                    })
                    ->whereIn('payment_type', [1, 2]); // 1=Income, 2=Expense

                // Optional: filter by specific fund_id (still fund_type=2)
                if ($request->filled('fund_id')) {
                    $query->where('fund_id', $request->fund_id);
                }

                // Optional: filter by type (1 or 2)
                if ($request->filled('payment_type') && in_array((int) $request->payment_type, [1, 2], true)) {
                    $query->where('payment_type', (int) $request->payment_type);
                }

                // Totals based on the SAME filtered dataset
                $totalsQ      = (clone $query);
                $incomeTotal  = (clone $totalsQ)->where('payment_type', 1)->sum('amount');
                $expenseTotal = (clone $totalsQ)->where('payment_type', 2)->sum('amount');
                $balance      = $incomeTotal - $expenseTotal;

                // Table data
                $data = $query->orderByDesc('id')->get();

                return \DataTables::of($data)
                    ->addColumn('date', function ($row) {
                        return $row->date ? \Carbon\Carbon::parse($row->date)->format('d-M-Y') : '';
                    })
                    ->addColumn('fund', function ($row) {
                        return optional($row->funds)->name ?? '--';
                    })
                    ->addColumn('donar', function ($row) {
                        $name  = $row->donar_id ? optional($row->donar)->name : ($row->name ?? '');
                        $phone = $row->donar_id ? optional($row->donar)->phone : ($row->phone ?? '');
                        return trim($name . '<br>' . $phone);
                    })
                    ->addColumn('type', function ($row) {
                        if ($row->payment_type == 1) {
                            return '<b>Income</b>';
                        }

                        if ($row->payment_type == 2) {
                            return '<b>Expense</b>';
                        }

                        return '';
                    })
                    ->addColumn('purpose', fn($row) => $row->purpose)
                    ->addColumn('amount', fn($row) => $row->amount ?? 0)
                    ->addColumn('action', function ($row) {
                        $details = '<a href="' . route('fund.collect.show', $row->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i> Details</a> ';
                        $edit    = '<a href="' . route('guraba.collect.edit', $row->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i> Edit</a> ';
                        $print   = '<a href="' . route('fund.collect.print', $row->id) . '" class="btn btn-sm btn-primary" title="Print"><i class="fa fa-file"></i> Print</a> ';
                        $delete  = '';

                        if (method_exists($this, 'DeleteData') && $this->DeleteData()) {
                            $delete = '<a href="#" data-remote="' . route('fund.collect.destroy', $row->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i> Delete</a>';
                        }

                        // আপনার আগের কন্ডিশন রাখা হলো (প্রয়োজনে বদলাতে পারেন)
                        return ($row->type == 1) ? ($details . $edit . $print . $delete) : $details;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['date', 'fund', 'donar', 'type', 'purpose', 'amount', 'action'])
                    ->with([
                        'summary' => [
                            'income'  => (float) $incomeTotal,
                            'expense' => (float) $expenseTotal,
                            'balance' => (float) $balance,
                        ],
                    ])
                    ->toJson();
            }

            // Non-AJAX view: invoice + funds for filter (guraba only)
            $invoice = \App\Models\InvoiceDesign::first();
            $funds   = \App\Models\Fund::where('fund_type', 2)->orderBy('name')->get(['id', 'name']);

            return view('dashboard.funds.guraba', compact('invoice', 'funds'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function Boarding(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = Boarding::with('funds:id,name', 'createdBy:id,name')->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y');
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount;
                    })

                    ->addColumn('bazarBy', function ($data) {
                        return $data->bazar_by ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        $details = '<a href="' . route('boarding.show', $data->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i> Details</a> ';
                        $edit    = '<a href="' . route('boarding.edit', $data->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i> Edit</a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<a href="#" data-remote="' . route('boarding.destroy', $data->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i> Delete</a>';
                        }

                        return $details . $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['amount', 'date', 'action', 'bazarBy'])
                    ->toJson();
            }
            return view('dashboard.boarding.index');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function BoardingCreate()
    {
        $fund           = Fund::all();
        $lastFundDetail = FundDetail::whereNotNull('receipt_no')
            ->orderByDesc('receipt_no')
            ->first();
        $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;
        return view('dashboard.boarding.create', compact('fund', 'receipt_no'));
    }
    public function BoardingCostCreate()
    {
        $fund           = Fund::all();
        $lastFundDetail = FundDetail::whereNotNull('receipt_no')
            ->orderByDesc('receipt_no')
            ->first();
        $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;
        return view('dashboard.boarding.cost_create', compact('fund', 'receipt_no'));
    }
    public function BoardingStore(Request $request)
    {
        $messages = [
            'date.required'   => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date'   => 'required',
            'amount' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {
            $data              = new Boarding();
            $data->date        = Carbon::createFromFormat('d/m/Y', $request->date)->format('Y-m-d');
            $data->fund_id     = $request->fund_id;
            $data->amount      = $request->amount;
            $data->bazar_list  = $request->bazar_list;
            $data->bazar_items = json_encode($request->bazar_items ?? []);
            $data->bazar_by    = $request->bazar_by;
            $data->note        = $request->note;
            $data->created_by  = Auth::user()->id;
            $data->save();

            $fund               = new FundDetail();
            $fund->date         = Carbon::createFromFormat('d/m/Y', $request->date)->format('Y-m-d');
            $fund->boarding_id  = $data->id;
            $fund->fund_id      = $request->fund_id;
            $fund->amount       = $request->amount;
            $fund->receipt_no   = $request->receipt_no;
            $fund->purpose      = 'Boarding Bazar';
            $fund->type         = 7;
            $fund->payment_type = 2;
            $fund->created_by   = Auth::user()->id;
            $fund->save();

            DB::commit();

            return redirect()->back()->with('success', 'Bazar created successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function BoardingShow($id)
    {
        try {
            $data    = Boarding::with('funds:id,name', 'createdBy:id,name')->findOrFail($id);
            $invoice = InvoiceDesign::first();
            return view('dashboard.boarding.show', compact('data', 'invoice'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function BoardingEdit($id)
    {
        try {
            $data       = Boarding::with('funds:id,name', 'createdBy:id,name')->findOrFail($id);
            $funds      = Fund::all();
            $receipt_no = FundDetail::where('boarding_id', $id)->where('type', 7)->first();
            return view('dashboard.boarding.edit', compact('data', 'funds', 'receipt_no'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function BoardingUpdate(Request $request, $id)
    {
        // dd($request->all());
        $messages = [
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'amount' => 'required',
        ], $messages);

        DB::beginTransaction();
        if ($request->date) {
            try {
                // Try d/m/Y format
                $date = Carbon::createFromFormat('d/m/Y', $request->date);
            } catch (\Exception $e) {
                try {
                    // Try Y-m-d format
                    $date = Carbon::createFromFormat('Y-m-d', $request->date);
                } catch (\Exception $e) {
                    // Optional: fallback or throw
                    $date = now();
                }
            }
        } else {
            $date = now();
        }
        try {
            $data       = Boarding::findOrFail($id);
            $data->date = $date->format('Y-m-d');

            $data->fund_id     = $request->fund_id;
            $data->amount      = $request->amount;
            $data->bazar_list  = $request->bazar_list;
            $data->bazar_items = json_encode($request->bazar_items ?? []);
            $data->bazar_by    = $request->bazar_by;
            $data->note        = $request->note;
            $data->updated_by  = Auth::user()->id;
            $data->save();

            $fund               = FundDetail::where('boarding_id', $id)->where('type', 7)->first();
            $fund->date         = $data->date;
            $fund->boarding_id  = $data->id;
            $fund->fund_id      = $request->fund_id;
            $fund->amount       = $request->amount;
            $fund->receipt_no   = $request->receipt_no;
            $fund->purpose      = 'Boarding Bazar';
            $fund->type         = 7;
            $fund->payment_type = 2;
            $fund->updated_by   = Auth::user()->id;
            $fund->save();

            DB::commit();

            return redirect()->route('boarding')->with('success', 'Bazar updated successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            // dd($exception, $request->all());
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function BoardingDestroy($id)
    {
        try {
            $data = Boarding::findOrFail($id);
            FundDetail::where('boarding_id', $id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }

    public function Mahfil(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = Mahfil::with('funds:id,name', 'createdBy:id,name')->where('type', 2)->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        $name = carbon::parse($data->date)->format('d M, Y');
                        return $name;
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount;
                    })

                    ->addColumn('name', function ($data) {
                        return $data->name;
                    })

                    ->addColumn('address', function ($data) {
                        return $data->address;
                    })

                    ->addColumn('action', function ($data) {
                        $details = '<a href="' . route('mahfil.show', $data->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i> Details</a> ';
                        $edit    = '<a href="' . route('mahfil.edit', $data->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i> Edit</a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<a href="#" data-remote="' . route('mahfil.destroy', $data->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i> Delete</a>';
                        }

                        return $details . $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['amount', 'date', 'action', 'name', 'address'])
                    ->toJson();
            }
            return view('dashboard.mahfil.index');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function MahfilCreate()
    {
        $funds          = Fund::where('fund_type', 3)->get();
        $lastFundDetail = FundDetail::whereNotNull('receipt_no')
            ->orderByDesc('receipt_no')
            ->first();
        $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;
        return view('dashboard.mahfil.create', compact('funds', 'receipt_no'));
    }
    public function MahfilStore(Request $request)
    {
        $messages = [
            'date.required'   => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date'   => 'required',
            'amount' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {
            $data             = new Mahfil();
            $data->date       = Carbon::parse($request->date)->format('Y-m-d');
            $data->fund_id    = $request->fund_id;
            $data->amount     = $request->amount;
            $data->name       = $request->name;
            $data->type       = 2;
            $data->address    = $request->address;
            $data->note       = $request->note;
            $data->created_by = Auth::user()->id;
            $data->save();

            $fund               = new FundDetail();
            $fund->date         = $data->date;
            $fund->mahfil_id    = $data->id;
            $fund->fund_id      = $request->fund_id;
            $fund->amount       = $request->amount;
            $fund->receipt_no   = $request->receipt_no;
            $fund->type         = 8;
            $fund->purpose      = 'Paid For Mahfil';
            $fund->payment_type = 2;
            $fund->fund_type    = 3;
            $fund->created_by   = Auth::user()->id;
            $fund->save();

            DB::commit();

            return redirect()->route('mahfil')->with('success', 'Mahfil created successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function MahfilShow($id)
    {
        try {
            $data = Mahfil::with('funds:id,name', 'createdBy:id,name', 'fundDetails:id,receipt_no,mahfil_id')
                ->where('type', 2)
                ->findOrFail($id);
            $invoice = InvoiceDesign::first();
            return view('dashboard.mahfil.show', compact('data', 'invoice'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function MahfilEdit($id)
    {
        try {
            $data = Mahfil::with('funds:id,name', 'createdBy:id,name', 'fundDetails:id,receipt_no,mahfil_id')
                ->where('type', 2)
                ->findOrFail($id);
            $funds = Fund::where('fund_type', 3)->get();
            return view('dashboard.mahfil.edit', compact('data', 'funds'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function MahfilUpdate(Request $request, $id)
    {
        $messages = [
            'date.required'   => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date'   => 'required',
            'amount' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {

            $data             = Mahfil::findOrFail($id);
            $data->date       = Carbon::parse($request->date)->format('Y-m-d');
            $data->fund_id    = $request->fund_id;
            $data->amount     = $request->amount;
            $data->name       = $request->name;
            $data->type       = 2;
            $data->address    = $request->address;
            $data->note       = $request->note;
            $data->created_by = Auth::user()->id;
            $data->save();

            $fund               = FundDetail::where('mahfil_id', $id)->first();
            $fund->date         = $data->date;
            $fund->mahfil_id    = $data->id;
            $fund->fund_id      = $request->fund_id;
            $fund->amount       = $request->amount;
            $fund->receipt_no   = $request->receipt_no;
            $fund->type         = 8;
            $fund->purpose      = 'Paid For Mahfil';
            $fund->payment_type = 2;
            $fund->fund_type    = 3;
            $fund->created_by   = Auth::user()->id;
            $fund->save();

            DB::commit();

            return redirect()->route('mahfil')->with('success', 'Mahfil updated successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function MahfilDestroy($id)
    {
        try {
            $data = Mahfil::findOrFail($id);
            FundDetail::where('mahfil_id', $id)->where('type', 8)->delete();
            $data->delete();

            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }
    public function MahfilTake(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = Mahfil::with('funds:id,name', 'createdBy:id,name')->where('type', 1)->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        $name = carbon::parse($data->date)->format('d M, Y');
                        return $name;
                    })

                    ->addColumn('amount', function ($data) {
                        return $data->amount;
                    })

                    ->addColumn('name', function ($data) {
                        return $data->name;
                    })

                    ->addColumn('address', function ($data) {
                        return $data->address;
                    })

                    ->addColumn('action', function ($data) {
                        $details = '<a href="' . route('mahfil.take.show', $data->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i> Details</a> ';
                        $edit    = '<a href="' . route('mahfil.take.edit', $data->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i> Edit</a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<a href="#" data-remote="' . route('mahfil.take.destroy', $data->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i> Delete</a>';
                        }

                        return $details . $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['amount', 'date', 'action', 'name', 'address'])
                    ->toJson();
            }
            return view('dashboard.mahfil.mahfil_given');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function MahfilTakeCreate()
    {
        $funds          = Fund::where('fund_type', 3)->get();
        $lastFundDetail = FundDetail::whereNotNull('receipt_no')
            ->orderByDesc('receipt_no')
            ->first();
        $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;
        return view('dashboard.mahfil.mahfil_given_create', compact('funds', 'receipt_no'));
    }
    public function MahfilTakeStore(Request $request)
    {
        $messages = [
            'date.required'   => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date'   => 'required',
            'amount' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {
            $data             = new Mahfil();
            $data->date       = $this->formatAnyDateToYmd($request->date);
            $data->fund_id    = $request->fund_id;
            $data->amount     = $request->amount;
            $data->name       = $request->name;
            $data->type       = 1;
            $data->address    = $request->address;
            $data->note       = $request->note;
            $data->created_by = Auth::user()->id;
            $data->save();

            $fund               = new FundDetail();
            $fund->date         = $data->date;
            $fund->mahfil_id    = $data->id;
            $fund->fund_id      = $request->fund_id;
            $fund->amount       = $request->amount;
            $fund->receipt_no   = $request->receipt_no;
            $fund->type         = 15;
            $fund->purpose      = 'Taken For Mahfil';
            $fund->payment_type = 1;
            $fund->fund_type    = 3;
            $fund->created_by   = Auth::user()->id;
            $fund->save();

            DB::commit();

            return redirect()->route('mahfil.take')->with('success', 'Data created successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function MahfilTakeShow($id)
    {
        try {
            $data = Mahfil::with('funds:id,name', 'createdBy:id,name', 'fundDetails:id,receipt_no,mahfil_id')
                ->where('type', 1)
                ->findOrFail($id);
            $invoice = InvoiceDesign::first();
            return view('dashboard.mahfil.mahfil_given_show', compact('data', 'invoice'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function MahfilTakeEdit($id)
    {
        try {
            $data = Mahfil::with('funds:id,name', 'createdBy:id,name', 'fundDetails:id,receipt_no,mahfil_id')
                ->where('type', 1)
                ->findOrFail($id);
            $funds = Fund::where('fund_type', 3)->get();
            return view('dashboard.mahfil.mahfil_given_edit', compact('data', 'funds'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function MahfilTakeUpdate(Request $request, $id)
    {
        $messages = [
            'date.required'   => 'Enter date',
            'amount.required' => 'Enter amount',
        ];

        $request->validate([
            'date'   => 'required',
            'amount' => 'required',
        ], $messages);

        DB::beginTransaction();

        try {

            $data             = Mahfil::findOrFail($id);
            $data->date       = $this->formatAnyDateToYmd($request->date);
            $data->fund_id    = $request->fund_id;
            $data->amount     = $request->amount;
            $data->name       = $request->name;
            $data->address    = $request->address;
            $data->type       = 1;
            $data->note       = $request->note;
            $data->created_by = Auth::user()->id;
            $data->save();

            $fund               = FundDetail::where('mahfil_id', $id)->first();
            $fund->date         = $data->date;
            $fund->mahfil_id    = $data->id;
            $fund->fund_id      = $request->fund_id;
            $fund->amount       = $request->amount;
            $fund->receipt_no   = $request->receipt_no;
            $fund->type         = 15;
            $fund->purpose      = 'Taken For Mahfil';
            $fund->payment_type = 1;
            $fund->fund_type    = 3;
            $fund->created_by   = Auth::user()->id;
            $fund->save();

            DB::commit();

            return redirect()->route('mahfil.take')->with('success', 'Mahfil updated successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function MahfilTakeDestroy($id)
    {
        try {
            $data = Mahfil::findOrFail($id);
            FundDetail::where('mahfil_id', $id)->where('type', 15)->delete();
            $data->delete();

            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }

    public function SponsorAmountTake()
    {
        $donars = Donar::all();
        $funds  = Fund::all();

        return view('dashboard.payments.sponsor_payment', compact('donars', 'funds'));
    }

    public function SponsorAmountTakeList(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = MoneyReceipt::with('students', 'stuclass')
                    ->where('type', 16)
                    ->selectRaw('month, student_id, SUM(amount) as total_amount, id')
                    ->groupBy('month', 'student_id')
                    ->get();

                return DataTables::of($data)
                    ->addColumn('department', function ($data) {
                        return optional($data->students->stuclass)->name ?? '--';
                    })

                    ->addColumn('donar', function ($data) {
                        $donar = StudentSponsor::with('donar')->where('student_id', $data->student_id)->first();
                        $value = isset($donar->donar->name) ? $donar->donar->name : '--';
                        return $value;
                    })
                    ->addColumn('month', function ($data) {
                        $month = Carbon::parse($data->month)->format('F Y');
                        return $month;
                    })

                    ->addColumn('student', function ($data) {
                        return optional($data->students)->name ?? '--';
                    })
                    ->addColumn('amount', function ($data) {
                        $fundDetailsAmount = FundDetail::where('month', $data->month)->where('type', 16)->whereNull('deleted_at')->sum('amount');
                        // dd($dat);
                        return $fundDetailsAmount ?? '--';
                    })
                    ->addColumn('action', function ($data) {
                        $details = '<a type="button" id="print" class="btn btn-sm btn-primary print mr-1" href="' . route('sponsor.amount.take.details', ['month' => $data->month, 'student_id' => $data->student_id]) . '" title="Print"><i class="fa fa-eye"></i></a>';
                        return $details;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['department', 'action', 'student', 'donar', 'amount'])
                    ->toJson();
            }

            return view('dashboard.payments.sponsor_payment_list');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function SponsorAmountTakeListDetail(Request $request, $month, $student_id)
    {
        try {
            if ($request->ajax()) {
                $data = MoneyReceipt::with('payment')->where('month', $month)
                    ->where('student_id', $student_id)
                    ->where('type', 16)
                    ->get();

                if ($data->isEmpty()) {
                    return response()->json([
                        'error' => 'No payments found for this student and month.',
                    ]);
                }

                $lastRecord = MoneyReceipt::where('month', $month)
                    ->where('student_id', $student_id)
                    ->where('type', 16)
                    ->orderBy('id', 'desc')
                    ->first();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d/m/Y');
                    })
                    ->addColumn('total_amount', function ($data) {
                        return $data->total ?? '';
                    })
                    ->addColumn('receive_amount', function ($data) {
                        // dd($data);
                        return $data->current_receive ?? '';
                    })
                    ->addColumn('due_amount', function ($data) {
                        return $data->due ?? '';
                    })
                    ->addColumn('discount_amount', function ($data) {
                        return $data->discount ?? '';
                    })

                    ->addColumn('action', function ($data) use ($lastRecord) {

                        $edit = '';

                        if ($data->id == $lastRecord->id) {

                            $edit = '<a id="edit" href="' . route('sponsor.amount.edit', $data->id) . '" class="btn btn-sm btn-primary edit" title="Edit"><i class="fa fa-edit"></i></a> ';
                        }

                        $print = '<a id="print" class="btn btn-sm btn-info print" href="' . route('payment.print', $data->id) . '" title="Print"
                                    style="margin-right: 5px !important;">
                                    <i class="fa fa-file"></i>
                                  </a>';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote="' . route('sponsor.amount.take.destroy', $data->id) . '" title="Delete">
                                <i class="fa fa-trash-alt"></i>
                        </button>';
                        }

                        return $edit . $print . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['action', 'date', 'amount', 'discount'])
                    ->toJson();
            }

            // Check if the student exists
            $student = Student::find($student_id);

            if (! $student) {
                return redirect()->back()->with('error', 'Student not found.');
            }

            $donar = StudentSponsor::with('donar')->where('student_id', $student_id)->first();

            if (! $donar) {
                return redirect()->back()->with('error', 'Donar not found.');
            }

            $studentName  = $student->name;
            $studentClass = $student->stuclass->name;
            $session      = $student->stusession->name;

            return view('dashboard.payments.sponsor_payment_details', compact('student_id', 'month', 'studentName', 'studentClass', 'session', 'donar'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function SponsorAmountEdit($id)
    {
        $students = Student::with('stuclass')->where(function ($query) {
            $query->where('status', 1)
                ->orWhereNull('status');
        })->get();
        $funds  = Fund::all();
        $donars = Donar::all();
        $data   = DB::table('money_receipts as mr')
            ->join('students as s', 's.id', '=', 'mr.student_id')
            ->join('student_classes as sc', 'sc.id', '=', 'mr.class_id')
            ->join('student_sessions as ss', 'ss.id', '=', 's.session_id')
            ->join('donars as d', 'd.id', '=', 'mr.donar_id')
            ->where('mr.id', $id)
            ->select(
                'mr.id as id',
                'mr.receipt_no as receipt_no',
                'mr.student_id as student_id',
                's.name as student_name',
                'mr.date as date',
                'mr.month as month',
                "sc.id as class_id",
                'sc.name as class_name',
                'ss.name as session_name',
                'd.name as donar_name',
                'd.id as donar_id',

                "mr.amount as receive_amount",
                "mr.total as total_amount",
                "mr.discount as discount_amount",
                "mr.due as due_amount",
                "mr.current_receive as current_receive",
            )->first();

        $fundDetailType = DB::table('fund_details_type as fdt')
            ->join('students as s', 's.id', '=', 'fdt.student_id')
            ->join('student_classes as sc', 'sc.id', '=', 'fdt.class_id')
            ->join('student_sessions as ss', 'ss.id', '=', 's.session_id')
            ->where('fdt.student_id', $data->student_id)
            ->where('fdt.student_payment_id', $data->id)
            ->select(
                'fdt.*',
                // 'fd.*',
                's.name as student_name',
                'sc.name as class_name',
                'ss.name as session_name',
            )
            ->get();

        foreach ($fundDetailType as $fdt) {
            $fdt->details = DB::table('fund_details as fd')
                ->join('funds as f', 'f.id', '=', 'fd.fund_id')
                ->join('student_fees as sf', 'sf.id', '=', 'fd.fee_id')
                ->where('fd.fund_detail_type_id', $fdt->id)
                ->select(
                    // 'fd.*',
                    'fd.id as fd_id',
                    'fd.amount as amount',
                    'sf.id as fee_id',
                    'sf.name as fee_name',
                    'f.id as fund_id',
                    'f.name as fund_name',
                )
                ->get()
                ->toArray();
        }
        $dataFund = FundDetail::with('students')->where('student_id', $data->student_id)
            ->where('student_payment_id', $id)->first();
        return view('dashboard.payments.sponsor_amount_edit', compact('students', 'funds', 'data', 'dataFund', 'donars', 'fundDetailType'));
    }

    public function SponsorAmountUpdate(Request $request, $id)
    {

        // dd($request->all());

        DB::beginTransaction();

        $totalReceived = FundDetail::where('student_id', $request->student_id)
            ->where('type', 6)
            ->where('month', Carbon::parse($request->month)->format('Y-m'))
            ->select(DB::raw('SUM(amount + discount) as total'))
            ->value('total');

        $preReceived = $totalReceived - ((float) $request->amount + (float) $request->discount);

        $totalAmount = (float) $request->total_amount;

        $totalReceivedAmount = (float) $preReceived + ((float) $request->amount + (float) $request->discount);

        // if ($totalReceivedAmount > $totalAmount) {
        //     return 'Payment already paid for ' . Carbon::parse($request->month)->format('F Y');
        // }

        $student = Student::findOrFail($request->student_id);

        $data                  = MoneyReceipt::findOrFail($id);
        $data->amount          = $request->receive_amount;
        $data->current_receive = $request->current_receive_amount;
        $data->discount        = $request->discount_amount;
        $data->due             = $request->due_amount;
        $data->total           = $request->total_amount;
        $data->donar_id        = $request->donar_id;
        $data->save();

        foreach ($request->rows as $key => $row) {
            $fundKeyMap = [
                'other'     => 'other_fee_paid',
                'academic'  => 'academic_fee_paid',
                'hostel'    => 'hostel_fee_paid',
                'onabashik' => 'onabashik_fee_paid',
                'special'   => 'special_fee_paid',
            ];

            if (isset($fundKeyMap[$key])) {
                $feeKey = $fundKeyMap[$key];

                $fundDetailTypes = FundDetailType::where('student_payment_id', $data->id)
                    ->where('fee_type', $key)
                    ->where('month', Carbon::parse($request->month)->format('Y-m'))
                    ->find($row['id']);

                // dd($feeKey, $key, $request->all());

                $fundDetailTypes->update([

                    'student_payment_id' => $data->id,
                    'fee_type'           => $key,

                    'total'              => $request->{$key . '_amount_total'},
                    'prev_receive'       => $request->{$key . '_previous_receive'},
                    'prev_discount'      => $request->{$key . '_previous_discount'},
                    'receive'            => $request->{$feeKey},
                    'discount'           => $request->{$key . '_fee_discount'},
                    'due'                => $request->{$key . '_fee_due'},
                    'updated_at'         => now()->format('Y-m-d'),

                ]);

                if ($request->{'detail_id_' . $key}) {
                    foreach ($request->{'detail_id_' . $key} as $detail_id) {
                        $fundDetail = FundDetail::findOrFail($detail_id);
                        // dd($request->all(), $key,$row['id'], $feeKey, $fundDetailTypes, $fundDetail, $request->{'fund_id_' . $key}[$fundDetail->fee_id]);
                        $fundDetail->update([
                            'student_payment_id'  => $data->id,
                            'fund_detail_type_id' => $fundDetailTypes->id,
                            'fund_id'             => $request->{'fund_id_' . $key}[$fundDetail->fee_id],
                            'updated_at'          => now()->format('Y-m-d'),
                        ]);

                        //  dd($fee_id, $fundId, $key, $request->{'fund_id_' . $key . 'fee_amount'}, $request->all());

                    }
                }

            }
        }

        $status = optional(SmsStatus::first())->student_payment;
        $site   = DB::table('settings')->first();
        $to     = $student->gurdian_mobile;
        $text   = $student->name . ', ' . $data->month . ' মাসিক বেতন Paid: ' . $data->amount . ' TK';
        if ($status == 1) {
            $this->SendSms($to, $text);
        }

        DB::commit();

        return redirect()->route('payment.print', ['id' => $data->id])->with('success', 'Amount received successfully');

        DB::rollBack();
    }

    public function SponsorAmountPrint($id)
    {
        $data = FundDetail::with('students:id,name,roll_no,register_no,class_id', 'donar')
            ->where('student_payment_id', $id)
            ->firstOrFail();

        $month = $data->month;

        $total = FundDetail::where('student_id', $data->student_id)
            ->where('month', $month)
            ->where('type', 16)
            ->where('payment_type', 1)
            ->where('donar_id', $data->donar_id)
            ->select(
                DB::raw('SUM(amount) as total_received'),
                DB::raw('SUM(discount) as total_discount')
            )
            ->first();

        $totalReceived = $total->total_received ?? 0;
        $totalDiscount = $total->total_discount ?? 0;

        $academicAmount = 0;
        $hostelAmount   = 0;
        $otherAmount    = 0;

        $type = Student::findOrFail($data->student_id);

        if ($type->student_type == 2) {
            $hostelAmount = FeeAssignDetail::where('type', 2)
                ->where('class_id', $type->class_id)
                ->where('session_id', $type->session_id)
                ->where('month', $month)
                ->sum('amount');
        }

        $academicAmount = FeeAssignDetail::where('type', 1)
            ->where('class_id', $type->class_id)
            ->where('session_id', $type->session_id)
            ->where('month', $month)
            ->sum('amount');

        $otherAmount = FeeAssignDetail::where('type', 3)
            ->where('class_id', $type->class_id)
            ->where('session_id', $type->session_id)
            ->where('month', $month)
            ->sum('amount');

        $totalAmount = $academicAmount + $hostelAmount + $otherAmount - $totalDiscount;

        $design = InvoiceDesign::first();

        return view('dashboard.payments.sponsor_amount_print', compact(
            'design',
            'data',
            'academicAmount',
            'hostelAmount',
            'otherAmount',
            'totalAmount',
            'totalReceived',
            'totalDiscount'
        ));
    }

    public function SponsorAmountTakeStore(Request $request)
    {
        // foreach ($request->rows as $key => $row) {
        //     dd($row['enabled'], $key, $request->all());
        // }
        //    dd($request->all());

        $lastFundDetail = FundDetail::whereNotNull('receipt_no')
            ->orderByDesc('receipt_no')
            ->first();
        $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;

        $totalReceived = FundDetailType::where('student_id', $request->student_id)
            ->where('month', Carbon::parse($request->month)->format('Y-m'))
            ->where('receipt_type', "new")
            ->whereNull('deleted_at')
            ->sum('receive');

        $totalDiscount = FundDetailType::where('student_id', $request->student_id)
            ->where('month', Carbon::parse($request->month)->format('Y-m'))
            ->where('receipt_type', "new")
            ->whereNull('deleted_at')
            ->sum('discount');

        $totalAmount = (float) $request->total_amount;
        $dueAmount   = (float) $totalAmount - ((float) $totalReceived + (float) $totalDiscount);

        // dd($totalReceived, $totalDiscount, $totalAmount, $dueAmount);

        // if ($dueAmount == 0) {

        //     return 'Payment already paid for ' . Carbon::parse($request->month)->format('F Y');
        // }

        DB::beginTransaction();

        try {
            if ($request->due_type == "old") {
                FundDetail::where('student_id', $request->student_id)
                    ->where('month', $request->month)
                    ->whereIn('type', [6, 16])
                    ->where('payment_type', 1)
                    ->whereNull('deleted_at')
                    ->whereNull('receipt_type')
                    ->whereNull('fund_detail_type_id')
                    ->whereNull('fee_id')
                    ->delete();
                // dd($totalReceived);
            }

            $student = Student::findOrFail($request->student_id);

            $data                  = new MoneyReceipt();
            $data->student_id      = $request->student_id;
            $data->receipt_no      = $receipt_no;
            $data->register_no     = $student->register_no;
            $data->date            = now()->format('Y-m-d');
            $data->name            = $student->name;
            $data->class_id        = $student->class_id;
            $data->month           = Carbon::parse($request->month)->format('Y-m');
            $data->amount          = $request->receive_amount;
            $data->current_receive = $request->current_receive_amount;
            $data->discount        = $request->discount_amount;
            $data->due             = $request->due_amount;
            $data->total           = $request->total_amount;
            $data->donar_id        = $request->donar_id;
            $data->type            = 16;
            $data->receipt_type    = 'new';
            $data->save();

            foreach ($request->rows[$request->month] as $key => $row) {
                $fundKeyMap = [
                    'other'     => 'other_fee_paid',
                    'academic'  => 'academic_fee_paid',
                    'hostel'    => 'hostel_fee_paid',
                    'onabashik' => 'onabashik_fee_paid',
                    'special'   => 'special_fee_paid',
                ];

                // dd($request->all(), $request->rows[$request->month], $key,$request->{$key . '_amount_total'}[$request->month], $request->{'fund_id_' . $key . 'fee_ids'}[$request->month]);
                if (isset($fundKeyMap[$key])) {
                    $feeKey = $fundKeyMap[$key];
                    // dd($feeKey, $key, $request->all());

                    $fundDetailTypes = FundDetailType::create([
                        'student_id'         => $request->student_id,
                        'class_id'           => $student->class_id,
                        'session_id'         => $student->session_id,
                        'receipt_no'         => $receipt_no,
                        'purpose'            => 'Student Monthly Payment Collect From Sponsor',
                        'student_payment_id' => $data->id,
                        'fee_type'           => $key,
                        'month'              => Carbon::parse($request->month)->format('Y-m'),
                        'total'              => $request->{$key . '_amount_total'}[$request->month],
                        'prev_receive'       => $request->{$key . '_previous_receive'}[$request->month],
                        'prev_discount'      => $request->{$key . '_previous_discount'}[$request->month],
                        'receive'            => $request->{$feeKey}[$request->month],
                        'discount'           => $request->{$key . '_fee_discount'}[$request->month],
                        'due'                => $request->{$key . '_fee_due'}[$request->month],
                        'date'               => now()->format('Y-m-d'),
                        'type'               => 16,
                        'payment_type'       => 1,
                        'fund_type'          => 1,
                        'receipt_type'       => 'new',

                    ]);

                    if (isset($request->{'fund_id_' . $key . 'fee_ids'}[$request->month])) {
                        foreach ($request->{'fund_id_' . $key . 'fee_ids'}[$request->month] as $fee_id => $amount) {

                            //  dd($fee_id, $fundId, $key, $request->{'fund_id_' . $key . 'fee_amount'}, $request->all());
                            FundDetail::create([
                                'fund_detail_type_id' => $fundDetailTypes->id,
                                'fund_id'             => $request->{'fund_id_' . $key}[$request->month][$fee_id],
                                'student_payment_id'  => $data->id,
                                'student_id'          => $request->student_id,
                                'invoice_no'          => $request->invoice_no,
                                'receipt_no'          => $receipt_no,
                                'purpose'             => 'Student Monthly Payment Collect From Sponsor',
                                'date'                => now()->format('Y-m-d'),
                                'type'                => 16,
                                'payment_type'        => 1,
                                'fund_type'           => 1,
                                'month'               => Carbon::parse($request->month)->format('Y-m'),
                                'total'               => $amount,
                                'amount'              => $request->{'fund_id_' . $key . 'fee_paid'}[$request->month][$fee_id],
                                'discount'            => $request->{'fund_id_' . $key . 'fee_discount'}[$request->month][$fee_id],
                                'due'                 => $request->{'fund_id_' . $key . 'fee_due'}[$request->month][$fee_id],
                                'fee_id'              => $fee_id,
                                'receipt_type'        => 'new',
                                'donar_id'            => $request->donar_id ?? '',
                            ]);
                        }
                    }
                }
            }

            $status = optional(SmsStatus::first())->student_payment;
            $donar  = Donar::findOrFail($request->donar_id);
            $site   = DB::table('settings')->first();
            $to     = $student->gurdian_mobile;
            $text   = $site->name . ', ' . $donar->name . ', আপনার পক্ষ থেকে এতিম ' . $student->name . ' এর ' . $request->month . ' মাসিক বেতন Paid ' . $request->current_receive_amount . ' Tk ';

            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            DB::commit();

            return redirect()->route('payment.print', ['id' => $data->id])->with('success', 'Amount received successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            dd($e);
            return back()->with('error', 'Error occurred: ' . $e->getMessage());
        }
    }

    public function SponsorAmountTakeStore_old(Request $request)
    {
        DB::beginTransaction();

        try {

            $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                ->orderByDesc('receipt_no')
                ->first();
            $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;

            $totalReceived = FundDetail::where('student_id', $request->student_id)
                ->where('donar_id', $request->donar_id)
                ->where('type', 16)
                ->where('month', Carbon::parse($request->month)->format('Y-m'))
                ->sum('amount');

            $totalAmount   = (float) $request->total_amount;
            $totalReceived = (float) $totalReceived + ((float) $request->amount - (float) $request->discount);

            if ($totalReceived > $totalAmount) {
                return 'Payment already paid for ' . Carbon::parse($request->month)->format('F Y');
            }

            $student = Student::findOrFail($request->student_id);

            $data              = new MoneyReceipt();
            $data->student_id  = $request->student_id;
            $data->receipt_no  = $receipt_no;
            $data->register_no = $student->register_no;
            $data->date        = now()->format('Y-m-d');
            $data->name        = $student->name;
            $data->class_id    = $student->class_id;
            $data->month       = Carbon::parse($request->month)->format('Y-m');
            $data->amount      = $request->amount;
            $data->type        = 16;
            $data->save();

            $fund                     = new FundDetail();
            $fund->fund_id            = $request->fund_id;
            $fund->student_payment_id = $data->id;
            $fund->student_id         = $request->student_id;
            $fund->donar_id           = $request->donar_id;
            $fund->invoice_no         = $request->invoice_no;
            $fund->receipt_no         = $receipt_no;
            $fund->purpose            = 'Student Monthly Payment Collect From Sponsor';
            $fund->date               = now()->format('Y-m-d');
            $fund->type               = 16;
            $fund->payment_type       = 1;
            $fund->fund_type          = 1;
            $fund->month              = Carbon::parse($request->month)->format('Y-m');
            $fund->amount             = $request->amount;
            $fund->discount           = $request->discount;
            $fund->note               = $request->note;
            $fund->save();

            $status = optional(SmsStatus::first())->donar_payment;

            $donar = Donar::findOrFail($request->donar_id);
            $site  = DB::table('settings')->first();
            $to    = $donar->phone;

            $text = $site->name . ', ' . $donar->name . ', আপনার পক্ষ থেকে এতিম ' . $student->name . ' এর ' . $fund->month . ' মাসিক বেতন Paid ' . $fund->amount . ' Tk ';
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            DB::commit();

            return redirect()->route('payment.print', ['id' => $data->id])->with('success', 'Amount received successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function SponsorAmountTakeDestroy($id)
    {
        try {
            $data = MoneyReceipt::findOrFail($id);
            FundDetail::where('student_payment_id', $id)->where('type', 16)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Money receipt deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Money receipt delete failed',
            ]);
        }
    }

    public function StudentMonthlyPayment(Request $request)
    {
        $months     = $request->months;
        $student_id = $request->student_id;
        $class_id   = $request->class_id;
        $session_id = $request->session_id;
        $donar_id   = $request->donar_id;

        $monthlyData    = [];
        $totalAcademic  = 0;
        $totalHostel    = 0;
        $totalOnabashik = 0;
        $totalOther     = 0;
        $totalSpecial   = 0;
        $totalReceived  = 0;
        $totalDiscount  = 0;
        $totalDue       = 0;

        $student = Student::findOrFail($student_id);

        foreach ($months as $monthString) {
            $carbonDate = Carbon::createFromFormat('F-Y', $monthString);
            $month      = $carbonDate->format('Y-m');

            $academicAmount = FeeAssignDetail::where('type', 1)
                ->where('class_id', $class_id)
                ->where('session_id', $session_id)
                ->where('month', $month)
                ->sum('amount');

            $hostelAmount = 0;

            if ($student->student_type == 2) {
                $hostelAmount = FeeAssignDetail::where('type', 2)
                    ->where('class_id', $class_id)
                    ->where('session_id', $session_id)
                    ->where('month', $month)
                    ->sum('amount');
            }

            $onabashikAmount = 0;
            if ($student->student_type == 1) {
                $onabashikAmount = FeeAssignDetail::where('type', 4)
                    ->where('class_id', $class_id)
                    ->where('session_id', $session_id)
                    ->where('month', $month)
                    ->sum('amount');
            }

            $otherAmount = FeeAssignDetail::where('type', 3)
                ->where('class_id', $class_id)
                ->where('session_id', $session_id)
                ->where('month', $month)
                ->sum('amount');

            $specialAmount = DB::table('fee_assign_details as fad')
                ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', 'fad.id')
                ->where('fds.student_id', $student_id)
                ->where('fad.type', 5)
                ->where('fad.class_id', $class_id)
                ->where('fad.session_id', $session_id)
                ->where('fad.month', $month)
                ->whereNull('fad.deleted_at')
                ->whereNull('fds.deleted_at')
                ->sum('fds.amount');

            $monthFee = $academicAmount + $hostelAmount + $onabashikAmount + $otherAmount + $specialAmount;

            $fundQuery = FundDetail::where('student_id', $student_id)
                ->where('month', $month)
                ->whereIn('type', [6, 16])
                ->where('payment_type', 1);

            if ($donar_id) {
                $fundQuery->where('donar_id', $donar_id);
            }

            $receivedData = $fundQuery->select(
                DB::raw('SUM(amount) as total_received'),
                DB::raw('SUM(discount) as total_discount')
            )->first();

            $received = $receivedData->total_received ?? 0;
            $discount = $receivedData->total_discount ?? 0;
            $due      = $monthFee - $received - $discount;

            $totalAcademic += $academicAmount;
            $totalHostel += $hostelAmount;
            $totalOnabashik += $onabashikAmount;
            $totalOther += $otherAmount;
            $totalSpecial += $specialAmount;
            $totalReceived += $received;
            $totalDiscount += $discount;
            $totalDue += $due;

            $monthlyData[] = [
                'month'         => $monthString,
                'academicFee'   => $academicAmount,
                'hostelFee'     => $hostelAmount,
                'onabashikFee'  => $onabashikAmount,
                'otherFee'      => $otherAmount,
                'specialFee'    => $specialAmount,
                'totalFee'      => $monthFee,
                'totalReceived' => $received,
                'totalDiscount' => $discount,
                'due'           => $due,
            ];
        }

        return response()->json([
            'monthlyData'    => $monthlyData,
            'totalAcademic'  => $totalAcademic,
            'totalHostel'    => $totalHostel,
            'totalOnabashik' => $totalOnabashik,
            'totalOther'     => $totalOther,
            'totalSpecial'   => $totalSpecial,
            'totalReceived'  => $totalReceived,
            'totalDiscount'  => $totalDiscount,
            'totalDue'       => $totalDue,
        ]);
    }

    public function DueAdvance()
    {
        $students = Student::with('stuclass')->where('etim', 2)
            ->where(function ($query) {
                $query->where('status', 1)
                    ->orWhereNull('status');
            })->get();
        $funds = Fund::all();
        return view('dashboard.payments.due_create', compact('students', 'funds'));
    }
    public function SponsorDueAdvance(Request $request)
    {
        $class_id       = $request->class_id;
        $student_id     = $request->student_id;
        $month          = $request->month;
        $studentSession = $request->student_id ? Student::where('id', $student_id)->first()->session_id : null;

        $students = Student::with('stuclass')->where('etim', 1)
            ->where(function ($query) {
                $query->where('status', 1)
                    ->orWhereNull('status');
            })->get();
        $donars = Donar::all();
        $funds  = Fund::all();
        return view('dashboard.payments.sponsor_due_create', compact('students', 'donars', 'funds', 'request', 'studentSession'));
    }

    public function SponsorDueAdvance_old()
    {
        $donars = Donar::all();
        $funds  = Fund::all();
        return view('dashboard.payments.sponsor_due_create', compact('donars', 'funds'));
    }

    public function StudentPaymentStore(Request $request)
    {
        // dd($request->all());
        DB::beginTransaction();

        try {

            $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                ->orderByDesc('receipt_no')
                ->first();

            $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;

            $student        = Student::findOrFail($request->student_id);
            $months         = $request->months;
            $totalAmount    = (float) $request->total_amount;
            $amountReceived = (float) $request->amount;
            $discount       = (float) $request->discount;
            $monthsCount    = count($months);

            $alreadyPaidMonths = [];

            foreach ($months as $month) {
                $monthFormatted = Carbon::parse($month)->format('Y-m');

                $monthlyPaid = FundDetail::where('student_id', $request->student_id)
                    ->where('type', 6)
                    ->where('month', $monthFormatted)
                    ->sum(DB::raw('amount + COALESCE(discount, 0)'));

                $academicAmount = FeeAssignDetail::where('type', 1)
                    ->where('class_id', $student->class_id)
                    ->where('session_id', $student->session_id)
                    ->where('month', $monthFormatted)
                    ->sum('amount');

                $hostelAmount = 0;
                if ($student->student_type == 2) {
                    $hostelAmount = FeeAssignDetail::where('type', 2)
                        ->where('class_id', $student->class_id)
                        ->where('session_id', $student->session_id)
                        ->where('month', $monthFormatted)
                        ->sum('amount');
                }

                $onabashikAmount = 0;
                if ($student->student_type == 1) {
                    $onabashikAmount = FeeAssignDetail::where('type', 4)
                        ->where('class_id', $student->class_id)
                        ->where('session_id', $student->session_id)
                        ->where('month', $monthFormatted)
                        ->sum('amount');

                }

                $specialAmount = DB::table('fee_assign_details as fd')
                    ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id') // ✅ join condition
                    ->whereNull('fd.deleted_at')                                                           // ✅ soft delete check
                    ->whereNull('fds.deleted_at')                                                          // ✅ soft delete check
                    ->where('fd.type', 5)
                    ->where('fds.student_id', $request->student_id)
                    ->where('fd.session_id', $student->session_id)
                    ->where('fd.class_id', $student->class_id)
                    ->where('fd.month', $monthFormatted)
                    ->sum('fds.amount');

                $otherAmount = FeeAssignDetail::where('type', 3)
                    ->where('class_id', $student->class_id)
                    ->where('session_id', $student->session_id)
                    ->where('month', $monthFormatted)
                    ->sum('amount');

                $monthlyAmount = $academicAmount + $hostelAmount + $otherAmount + $onabashikAmount + $specialAmount;
                $monthlyDue    = max(0, $monthlyAmount - $monthlyPaid);

                if ($monthlyDue <= 0) {
                    $alreadyPaidMonths[] = Carbon::parse($month)->format('F Y');
                }
            }

            if (! empty($alreadyPaidMonths)) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Payment already completed for month(s): ' . implode(', ', $alreadyPaidMonths),
                ], 422);
            }

            $baseDiscount      = floor($discount / $monthsCount);
            $remaining         = $discount - ($baseDiscount * $monthsCount);
            $discountsPerMonth = array_fill(0, $monthsCount, $baseDiscount);

            for ($i = 0; $i < $remaining; $i++) {
                $discountsPerMonth[$i]++;
            }

            $processedMonths   = [];
            $remainingAmount   = $amountReceived;
            $remainingDiscount = $discount;

            foreach ($months as $index => $month) {
                if ($remainingAmount <= 0 && $remainingDiscount <= 0) {
                    break;
                }

                $monthFormatted = Carbon::parse($month)->format('Y-m');

                $monthlyPaid = FundDetail::where('student_id', $request->student_id)
                    ->where('type', 6)
                    ->where('month', $monthFormatted)
                    ->sum(DB::raw('amount + COALESCE(discount, 0)'));

                $academicAmount = FeeAssignDetail::where('type', 1)
                    ->where('class_id', $student->class_id)
                    ->where('session_id', $student->session_id)
                    ->where('month', $monthFormatted)
                    ->sum('amount');

                $hostelAmount = 0;
                if ($student->student_type == 2) {
                    $hostelAmount = FeeAssignDetail::where('type', 2)
                        ->where('class_id', $student->class_id)
                        ->where('session_id', $student->session_id)
                        ->where('month', $monthFormatted)
                        ->sum('amount');
                }

                $onabashikAmount = 0;
                if ($student->student_type == 1) {
                    $onabashikAmount = FeeAssignDetail::where('type', 4)
                        ->where('class_id', $student->class_id)
                        ->where('session_id', $student->session_id)
                        ->where('month', $monthFormatted)
                        ->sum('amount');

                }

                $specialAmount = DB::table('fee_assign_details as fd')
                    ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id') // ✅ join condition
                    ->whereNull('fd.deleted_at')                                                           // ✅ soft delete check
                    ->whereNull('fds.deleted_at')                                                          // ✅ soft delete check
                    ->where('fd.type', 5)
                    ->where('fds.student_id', $request->student_id)
                    ->where('fd.session_id', $student->session_id)
                    ->where('fd.class_id', $student->class_id)
                    ->where('fd.month', $monthFormatted)
                    ->sum('fds.amount');

                $otherAmount = FeeAssignDetail::where('type', 3)
                    ->where('class_id', $student->class_id)
                    ->where('session_id', $student->session_id)
                    ->where('month', $monthFormatted)
                    ->sum('amount');

                $monthlyAmount = $academicAmount + $hostelAmount + $otherAmount + $onabashikAmount + $specialAmount;
                $monthlyDue    = max(0, $monthlyAmount - $monthlyPaid);

                if ($monthlyDue <= 0) {
                    continue;
                }

                $discountForThisMonth = min($discountsPerMonth[$index], $remainingDiscount);
                $amountForThisMonth   = min($monthlyDue - $discountForThisMonth, $remainingAmount);

                if ($amountForThisMonth <= 0 && $discountForThisMonth <= 0) {
                    continue;
                }

                // dd($amountForThisMonth, $monthlyAmount,$academicAmount,$hostelAmount,$otherAmount,$onabashikAmount,$specialAmount);

                $data              = new MoneyReceipt();
                $data->student_id  = $request->student_id;
                $data->receipt_no  = $receipt_no;
                $data->register_no = $student->register_no;
                $data->date        = now()->format('Y-m-d');
                $data->name        = $student->name;
                $data->class_id    = $student->class_id;
                $data->month       = $monthFormatted;
                $data->amount      = $amountForThisMonth;
                $data->type        = 1;
                $data->save();

                $fund                     = new FundDetail();
                $fund->fund_id            = $request->fund_id;
                $fund->student_payment_id = $data->id;
                $fund->student_id         = $request->student_id;
                $fund->invoice_no         = $request->invoice_no;
                $fund->receipt_no         = $receipt_no;
                $fund->purpose            = 'Student Monthly Payment for ' . $monthFormatted;
                $fund->date               = now()->format('Y-m-d');
                $fund->type               = 6;
                $fund->payment_type       = 1;
                $fund->fund_type          = 1;
                $fund->month              = $monthFormatted;
                $fund->amount             = $amountForThisMonth;
                $fund->discount           = $discountForThisMonth;
                $fund->note               = $request->note;
                $fund->save();

                $remainingAmount -= $amountForThisMonth;
                $remainingDiscount -= $discountForThisMonth;
                $processedMonths[] = $monthFormatted;
            }

            if (! empty($processedMonths)) {
                $status = optional(SmsStatus::first())->student_payment;
                if ($status == 1) {
                    $site       = DB::table('settings')->first();
                    $to         = $student->gurdian_mobile;
                    $monthsList = implode(', ', array_map(function ($m) {
                        return Carbon::parse($m)->format('F Y');
                    }, $processedMonths));

                    $text = $site->name . ', ' . $student->name . ', Payment received for: ' . $monthsList .
                        ', Amount: ' . ($amountReceived - $remainingAmount) . ' TK, Discount: ' . ($discount - $remainingDiscount) . ' TK';
                    $this->SendSms($to, $text);
                }

                DB::commit();
                return redirect()->route('student.payment.print', ['id' => $data->student_id, 'receipt' => $data->receipt_no])->with('success', 'Payment processed successfully');
            } else {
                DB::rollBack();
                return back()->with('error', 'No payment was processed - all selected months are already paid');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error processing payment: ' . $e->getMessage());
        }
    }

    public function StudentPaymentPrint($id, $receipt)
    {
        $fundDetails = FundDetail::with('students:id,name,roll_no,register_no,class_id')
            ->where('student_id', $id)
            ->where('type', 6)
            ->where('payment_type', 1)
            ->whereIn('receipt_no', [$receipt])
            ->get();

        if ($fundDetails->isEmpty()) {
            abort(404, 'No payment data found.');
        }

        $student = $fundDetails->first()->students;

        $months = $fundDetails->pluck('month')->unique()->sort();

        $totalReceived = FundDetail::where('student_id', $id)
            ->where('type', 6)
            ->where('payment_type', 1)
            ->whereIn('month', $months)
            ->sum(DB::raw('amount + COALESCE(discount, 0)'));

        $totalDiscount = $fundDetails->sum('discount');

        $invoice = FundDetail::where('student_id', $id)->latest()->first();

        $currentEntries = $fundDetails->filter(function ($item) use ($invoice) {
            return $item->date == $invoice->date && $item->receipt_no == $invoice->receipt_no;
        });

        $previousEntries = $fundDetails->reject(function ($item) use ($invoice) {
            return $item->date == $invoice->date && $item->receipt_no == $invoice->receipt_no;
        });

        $currentReceived = $currentEntries->sum(function ($entry) {
            return $entry->amount + ($entry->discount ?? 0);
        });

        $previousReceived = $totalReceived - $currentReceived;

        $studentData = Student::findOrFail($id);

        $academicAmount  = 0;
        $academicFee     = [];
        $hostelAmount    = 0;
        $hostelFee       = [];
        $onabashikAmount = 0;
        $onabashikFee    = [];
        $otherAmount     = 0;
        $otherFee        = [];
        $specialAmount   = 0;
        $specialFee      = [];

        foreach ($months as $month) {
            $academicAmount += FeeAssignDetail::where('type', 1)
                ->where('class_id', $studentData->class_id)
                ->where('session_id', $studentData->session_id)
                ->where('month', $month)
                ->sum('amount');

            $monthlyAcademicFee = FeeAssignDetail::with('fees')
                ->where('type', 1)
                ->where('class_id', $studentData->class_id)
                ->where('session_id', $studentData->session_id)
                ->where('month', $month)
                ->get()
                ->groupBy('fee_id')
                ->map(function ($items, $fee_id) {
                    $fee_name = optional($items->first()->fees)->name;
                    $amount   = $items->sum('amount');

                    return [
                        'fee_id'   => $fee_id,
                        'fee_name' => $fee_name,
                        'amount'   => $amount,
                    ];
                })
                ->values();

            $academicFee = array_merge($academicFee, $monthlyAcademicFee->toArray());

            if ($studentData->student_type == 2) {
                $hostelAmount += FeeAssignDetail::where('type', 2)
                    ->where('class_id', $studentData->class_id)
                    ->where('session_id', $studentData->session_id)
                    ->where('month', $month)
                    ->sum('amount');

                $monthlyHostelFee = FeeAssignDetail::with('fees')
                    ->where('type', 2)
                    ->where('class_id', $studentData->class_id)
                    ->where('session_id', $studentData->session_id)
                    ->where('month', $month)
                    ->get()
                    ->groupBy('fee_id')
                    ->map(function ($items, $fee_id) {
                        $fee_name = optional($items->first()->fees)->name;
                        $amount   = $items->sum('amount');

                        return [
                            'fee_id'   => $fee_id,
                            'fee_name' => $fee_name,
                            'amount'   => $amount,
                        ];
                    })
                    ->values();

                $hostelFee = array_merge($hostelFee, $monthlyHostelFee->toArray());
            }

            if ($studentData->student_type == 1) {
                $onabashikAmount += FeeAssignDetail::where('type', 4)
                    ->where('class_id', $studentData->class_id)
                    ->where('session_id', $studentData->session_id)
                    ->where('month', $month)
                    ->sum('amount');

                $monthlyOnabashikFee = FeeAssignDetail::with('fees')
                    ->where('type', 4)
                    ->where('class_id', $studentData->class_id)
                    ->where('session_id', $studentData->session_id)
                    ->where('month', $month)
                    ->get()
                    ->groupBy('fee_id')
                    ->map(function ($items, $fee_id) {
                        $fee_name = optional($items->first()->fees)->name;
                        $amount   = $items->sum('amount');

                        return [
                            'fee_id'   => $fee_id,
                            'fee_name' => $fee_name,
                            'amount'   => $amount,
                        ];
                    })
                    ->values();

                $onabashikFee = array_merge($onabashikFee, $monthlyOnabashikFee->toArray());
            }

            $otherAmount += FeeAssignDetail::where('type', 3)
                ->where('class_id', $studentData->class_id)
                ->where('session_id', $studentData->session_id)
                ->where('month', $month)
                ->sum('amount');

            $monthlyOtherFee = FeeAssignDetail::with('fees')
                ->where('type', 3)
                ->where('class_id', $studentData->class_id)
                ->where('session_id', $studentData->session_id)
                ->where('month', $month)
                ->get()
                ->groupBy('fee_id')
                ->map(function ($items, $fee_id) {
                    $fee_name = optional($items->first()->fees)->name;
                    $amount   = $items->sum('amount');

                    return [
                        'fee_id'   => $fee_id,
                        'fee_name' => $fee_name,
                        'amount'   => $amount,
                    ];
                })
                ->values();

            $otherFee = array_merge($otherFee, $monthlyOtherFee->toArray());

            $specialAmount += DB::table('fee_assign_details as fad')
                ->join('feeassign_details_students as fads', 'fads.fee_assign_details_id', '=', 'fad.id')
                ->whereNull('fad.deleted_at')  // ✅ soft delete check
                ->whereNull('fads.deleted_at') // ✅ soft delete check
                ->where('fad.type', 5)
                ->where('fad.class_id', $studentData->class_id)
                ->where('fad.session_id', $studentData->session_id)
                ->where('fad.month', $month)
                ->where('fads.student_id', $id)
                ->sum('fads.amount');

            $monthlySpecialFee = DB::table('fee_assign_details as fad')
                ->join('feeassign_details_students as fads', 'fads.fee_assign_details_id', '=', 'fad.id')
                ->join('student_fees as fees', 'fees.id', '=', 'fad.fee_id')
                ->whereNull('fad.deleted_at')  // ✅ soft delete check
                ->whereNull('fads.deleted_at') // ✅ soft delete check
                ->whereNull('fees.deleted_at') // ✅ soft delete check
                ->where('fad.type', 5)
                ->where('fad.class_id', $studentData->class_id)
                ->where('fad.session_id', $studentData->session_id)
                ->where('fad.month', $month)
                ->where('fads.student_id', $id)
                ->select('fad.fee_id', 'fees.name as fee_name', 'fads.amount')
                ->get()
                ->groupBy('fee_id')
                ->map(function ($items, $fee_id) {
                    $fee_name = $items->first()->fee_name;
                    $amount   = $items->sum('amount');

                    return [
                        'fee_id'   => $fee_id,
                        'fee_name' => $fee_name,
                        'amount'   => $amount,
                    ];
                })
                ->values();

            $specialFee = array_merge($specialFee, $monthlySpecialFee->toArray());

            // dd($specialFee);

        }

        $totalAmount = $academicAmount + $hostelAmount + $otherAmount + $onabashikAmount + $specialAmount - $totalDiscount;

        $design = InvoiceDesign::first();
        $data   = Student::findOrFail($id);

        return view('dashboard.payments.student_payment_print', compact(
            'design',
            'data',
            'fundDetails',
            'months',
            'academicAmount',
            'academicFee',
            'hostelAmount',
            'hostelFee',
            'onabashikAmount',
            'onabashikFee',
            'otherAmount',
            'otherFee',
            'specialAmount',
            'specialFee',
            'totalAmount',
            'totalReceived',
            'totalDiscount',
            'invoice',
            'currentReceived',
            'previousReceived'
        ));

    }

    public function SponsorStudentPaymentStore(Request $request)
    {
        DB::beginTransaction();

        try {
            $student = Student::findOrFail($request->student_id);

            $fundKeyMap = [
                'academic'  => 'academic_fee_paid',
                'hostel'    => 'hostel_fee_paid',
                'special'   => 'special_fee_paid',
                'onabashik' => 'onabashik_fee_paid',
                'other'     => 'other_fee_paid',
                // add other keys like onabashik, other, special if used
            ];

            // dd($request->all());

            $moneyReceiptMulti                  = new MoneyReceiptMultiple();
            $moneyReceiptMulti->student_id      = $request->student_id;
            $moneyReceiptMulti->class_id        = $student->class_id;
            $moneyReceiptMulti->date            = now()->format('Y-m-d');
            $moneyReceiptMulti->name            = $student->name;
            $moneyReceiptMulti->total_amount    = $request->total_amount;
            $moneyReceiptMulti->current_receive = $request->current_receive_amount;
            $moneyReceiptMulti->total_receive   = $request->receive_amount;
            $moneyReceiptMulti->total_discount  = $request->discount_amount;
            $moneyReceiptMulti->total_due       = $request->due_amount;
            $moneyReceiptMulti->save();

            foreach ($request->months as $month) {

                $totalAmount      = 0;
                $currentReceive   = 0;
                $previousReceive  = 0;
                $previousDiscount = 0;
                $currentDiscount  = 0;
                $currentDue       = 0;
                foreach ($fundKeyMap as $key => $value) {
                    if (isset($request->{$key . '_amount_total'}[$month])) {
                        $totalAmount += (float) $request->{$key . '_amount_total'}[$month];
                    }

                    if (isset($request->{$key . '_fee_paid'}[$month])) {
                        $currentReceive += (float) $request->{$key . '_fee_paid'}[$month];
                    }

                    if (isset($request->{$key . '_fee_discount'}[$month])) {
                        $currentDiscount += (float) $request->{$key . '_fee_discount'}[$month];
                    }

                    if (isset($request->{$key . '_fee_due'}[$month])) {
                        $currentDue += (float) $request->{$key . '_fee_due'}[$month];
                    }

                    if (isset($request->{$key . '_previous_receive'}[$month])) {
                        $previousReceive += (float) $request->{$key . '_previous_receive'}[$month];
                    }

                    if (isset($request->{$key . '_previous_discount'}[$month])) {
                        $previousDiscount += (float) $request->{$key . '_previous_discount'}[$month];
                    }

                }

                // Skip if already fully paid
                $totalReceived = $currentReceive + $previousReceive;
                $totalDiscount = $currentDiscount + $previousDiscount;

                $dueAmount = $totalAmount - ($totalReceived + $totalDiscount);
                // dd($request->all(), $totalAmount, $currentReceive, $currentDiscount, $currentDue, $previousReceive,$previousDiscount, $totalReceived, $totalDiscount, $dueAmount);

                //     if ($dueAmount <= 0) {
                //     dd($request->all(), $moneyReceiptMulti, $dueAmount, $totalReceived, $totalDiscount, $totalAmount, $currentReceive, $currentDiscount, $currentDue, $previousReceive,$previousDiscount);
                //     continue;
                // }

                $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                    ->orderByDesc('receipt_no')
                    ->first();
                $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;

                if ($request->due_type === "old") {
                    FundDetail::where('student_id', $request->student_id)
                        ->where('month', $month)
                        ->whereIn('type', [6, 16])
                        ->where('payment_type', 1)
                        ->whereNull('deleted_at')
                        ->whereNull('receipt_type')
                        ->whereNull('fund_detail_type_id')
                        ->whereNull('fee_id')
                        ->delete();
                }

                // Save main money receipt
                $moneyReceipt                            = new MoneyReceipt();
                $moneyReceipt->student_id                = $request->student_id;
                $moneyReceipt->receipt_no                = $receipt_no;
                $moneyReceipt->register_no               = $student->register_no;
                $moneyReceipt->date                      = now()->format('Y-m-d');
                $moneyReceipt->name                      = $student->name;
                $moneyReceipt->class_id                  = $student->class_id;
                $moneyReceipt->month                     = $month;
                $moneyReceipt->amount                    = $totalReceived;
                $moneyReceipt->current_receive           = $currentReceive;
                $moneyReceipt->discount                  = $totalDiscount;
                $moneyReceipt->due                       = $dueAmount;
                $moneyReceipt->total                     = $totalAmount;
                $moneyReceipt->type                      = 16;
                $moneyReceipt->receipt_type              = 'new';
                $moneyReceipt->money_receipt_multiple_id = $moneyReceiptMulti->id;
                $moneyReceipt->donar_id                  = $request->donar_id;
                $moneyReceipt->save();

                foreach (($request->rows[$month] ?? []) as $type => $row) {
                    if (! isset($row['enabled'])) {
                        continue;
                    }

                    $feeKey = $fundKeyMap[$type] ?? null;
                    if (! $feeKey) {
                        continue;
                    }

                    $fundDetailType = FundDetailType::create([
                        'student_id'         => $request->student_id,
                        'class_id'           => $student->class_id,
                        'session_id'         => $student->session_id,
                        'receipt_no'         => $receipt_no,
                        'purpose'            => 'Student Monthly Payment Collect From Sponsor',
                        'student_payment_id' => $moneyReceipt->id,
                        'fee_type'           => $type,
                        'month'              => $month,
                        'total'              => $request[$type . '_amount_total'][$month] ?? 0,
                        'prev_receive'       => $request[$type . '_previous_receive'][$month] ?? 0,
                        'prev_discount'      => $request[$type . '_previous_discount'][$month] ?? 0,
                        'receive'            => $request[$type . '_fee_paid'][$month] ?? 0,
                        'discount'           => $request[$type . '_fee_discount'][$month] ?? 0,
                        'due'                => $request[$type . '_fee_due'][$month] ?? 0,
                        'date'               => now()->format('Y-m-d'),
                        'type'               => 16,
                        'payment_type'       => 1,
                        'fund_type'          => 1,
                        'receipt_type'       => 'new',
                    ]);

                    $feeIDs  = $request->{'fund_id_' . $type . 'fee_ids'}[$month] ?? [];
                    $funds   = $request->{'fund_id_' . $type}[$month] ?? [];
                    $amounts = $request->{'fund_id_' . $type . 'fee_amount'}[$month] ?? [];

                    // dd($feeIDs, $funds, $amounts, $request->all());

                    foreach ($feeIDs as $fee_id => $amount) {
                        FundDetail::create([
                            'fund_detail_type_id' => $fundDetailType->id,
                            'fund_id'             => $funds[$fee_id] ?? null,
                            'student_payment_id'  => $moneyReceipt->id,
                            'student_id'          => $request->student_id,
                            'receipt_no'          => $receipt_no,
                            'purpose'             => 'Student Monthly Payment Collect From Sponsor',
                            'date'                => now()->format('Y-m-d'),
                            'type'                => 16,
                            'payment_type'        => 1,
                            'fund_type'           => 1,
                            'month'               => $month,

                            'total'               => $amount,
                            'amount'              => $request->{'fund_id_' . $type . 'fee_paid'}[$month][$fee_id],
                            'discount'            => $request->{'fund_id_' . $type . 'fee_discount'}[$month][$fee_id],
                            'due'                 => $request->{'fund_id_' . $type . 'fee_due'}[$month][$fee_id],
                            'fee_id'              => $fee_id,
                            'receipt_type'        => 'new',
                            'donar_id'            => $request->donar_id,
                        ]);
                    }
                }

                // SMS (optional)
                $status = optional(SmsStatus::first())->student_payment;
                $site   = DB::table('settings')->first();
                $to     = $student->gurdian_mobile;
                $text   = $student->name . ', ' . $month . ' মাসিক বেতন Paid : ' . $moneyReceipt->amount . ' TK';
                if ($status == 1) {
                    $this->SendSms($to, $text);
                }
            }

            DB::commit();
            return redirect()->route('payment.print.multi', ['id' => $moneyReceiptMulti->id])->with('success', 'Amount received successfully');
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error occurred: ' . $e->getMessage());
        }
    }

    public function SponsorStudentPaymentStore_old(Request $request)
    {
        DB::beginTransaction();

        try {
            $lastFundDetail = FundDetail::whereNotNull('receipt_no')
                ->orderByDesc('receipt_no')
                ->first();
            $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;

            $student = Student::findOrFail($request->student_id);
            $months  = $request->months;

            $amountReceived = (float) $request->amount;
            $discount       = (float) $request->discount;
            $monthsCount    = count($months);

            $alreadyPaidMonths = [];

            foreach ($months as $month) {
                $monthFormatted = Carbon::parse($month)->format('Y-m');

                $monthlyPaid = FundDetail::where('student_id', $request->student_id)
                    ->where('donar_id', $request->donar_id)
                    ->where('type', 16)
                    ->where('month', $monthFormatted)
                    ->sum(DB::raw('amount + COALESCE(discount, 0)'));

                $academicAmount = FeeAssignDetail::where('type', 1)
                    ->where('class_id', $student->class_id)
                    ->where('session_id', $student->session_id)
                    ->where('month', $monthFormatted)
                    ->sum('amount');

                $hostelAmount = 0;
                if ($student->student_type == 2) {
                    $hostelAmount = FeeAssignDetail::where('type', 2)
                        ->where('class_id', $student->class_id)
                        ->where('session_id', $student->session_id)
                        ->where('month', $monthFormatted)
                        ->sum('amount');
                }

                $otherAmount = FeeAssignDetail::where('type', 3)
                    ->where('class_id', $student->class_id)
                    ->where('session_id', $student->session_id)
                    ->where('month', $monthFormatted)
                    ->sum('amount');

                $monthlyAmount = $academicAmount + $hostelAmount + $otherAmount;
                $monthlyDue    = max(0, $monthlyAmount - $monthlyPaid);

                if ($monthlyDue <= 0) {
                    $alreadyPaidMonths[] = Carbon::parse($month)->format('F Y');
                }
            }

            if (! empty($alreadyPaidMonths)) {
                return response()->json([
                    'status'  => 'error',
                    'message' => 'Payment already completed for month(s): ' . implode(', ', $alreadyPaidMonths),
                ]);
            }

            $baseDiscount      = floor($discount / $monthsCount);
            $remaining         = $discount - ($baseDiscount * $monthsCount);
            $discountsPerMonth = array_fill(0, $monthsCount, $baseDiscount);

            for ($i = 0; $i < $remaining; $i++) {
                $discountsPerMonth[$i]++;
            }

            $processedMonths   = [];
            $remainingAmount   = $amountReceived;
            $remainingDiscount = $discount;

            foreach ($months as $index => $month) {
                if ($remainingAmount <= 0 && $remainingDiscount <= 0) {
                    break;
                }

                $monthFormatted = Carbon::parse($month)->format('Y-m');

                $monthlyPaid = FundDetail::where('student_id', $request->student_id)
                    ->where('donar_id', $request->donar_id)
                    ->where('type', 16)
                    ->where('month', $monthFormatted)
                    ->sum(DB::raw('amount + COALESCE(discount, 0)'));

                $academicAmount = FeeAssignDetail::where('type', 1)
                    ->where('class_id', $student->class_id)
                    ->where('session_id', $student->session_id)
                    ->where('month', $monthFormatted)
                    ->sum('amount');

                $hostelAmount = 0;
                if ($student->student_type == 2) {
                    $hostelAmount = FeeAssignDetail::where('type', 2)
                        ->where('class_id', $student->class_id)
                        ->where('session_id', $student->session_id)
                        ->where('month', $monthFormatted)
                        ->sum('amount');
                }

                $otherAmount = FeeAssignDetail::where('type', 3)
                    ->where('class_id', $student->class_id)
                    ->where('session_id', $student->session_id)
                    ->where('month', $monthFormatted)
                    ->sum('amount');

                $monthlyAmount = $academicAmount + $hostelAmount + $otherAmount;
                $monthlyDue    = max(0, $monthlyAmount - $monthlyPaid);

                if ($monthlyDue <= 0) {
                    continue;
                }

                $discountForThisMonth = min($discountsPerMonth[$index], $remainingDiscount);
                $amountForThisMonth   = min($monthlyDue - $discountForThisMonth, $remainingAmount);

                if ($amountForThisMonth <= 0 && $discountForThisMonth <= 0) {
                    continue;
                }

                $data              = new MoneyReceipt();
                $data->student_id  = $request->student_id;
                $data->receipt_no  = $receipt_no;
                $data->register_no = $student->register_no;
                $data->date        = now()->format('Y-m-d');
                $data->name        = $student->name;
                $data->class_id    = $student->class_id;
                $data->month       = $monthFormatted;
                $data->amount      = $amountForThisMonth;
                $data->type        = 16;
                $data->save();

                $fund                     = new FundDetail();
                $fund->fund_id            = $request->fund_id;
                $fund->student_payment_id = $data->id;
                $fund->donar_id           = $request->donar_id;
                $fund->student_id         = $request->student_id;
                $fund->receipt_no         = $receipt_no;
                $fund->invoice_no         = $request->invoice_no;
                $fund->purpose            = 'Student Monthly Payment Collect From Sponsor';
                $fund->date               = now()->format('Y-m-d');
                $fund->type               = 16;
                $fund->payment_type       = 1;
                $fund->fund_type          = 1;
                $fund->month              = $monthFormatted;
                $fund->amount             = $amountForThisMonth;
                $fund->discount           = $discountForThisMonth;
                $fund->note               = $request->note;
                $fund->save();

                $remainingAmount -= $amountForThisMonth;
                $remainingDiscount -= $discountForThisMonth;
                $processedMonths[] = $monthFormatted;
            }

            if (! empty($processedMonths)) {
                $status = optional(SmsStatus::first())->student_payment;
                if ($status == 1) {
                    $site       = DB::table('settings')->first();
                    $to         = $student->gurdian_mobile;
                    $monthsList = implode(', ', array_map(function ($m) {
                        return Carbon::parse($m)->format('F Y');
                    }, $processedMonths));

                    $text = $site->name . ', ' . $student->name . ', Payment received for: ' . $monthsList .
                        ', Amount: ' . ($amountReceived - $remainingAmount) . ' TK, Discount: ' . ($discount - $remainingDiscount) . ' TK';
                    $this->SendSms($to, $text);
                }

                DB::commit();
                return redirect()->route('sponsor.student.payment.print', [
                    'id'       => $data->student_id,
                    'receipt'  => $data->receipt_no,
                    'donar_id' => $fund->donar_id,
                ])->with('success', 'Payment processed successfully');
            } else {
                DB::rollBack();
                return back()->with('error', 'No payment was processed - all selected months are already paid');
            }
        } catch (\Exception $e) {
            DB::rollBack();
            return back()->with('error', 'Error processing payment: ' . $e->getMessage());
        }
    }

    public function SponsorStudentPaymentPrint($id, $receipt, $donar_id)
    {
        $fundDetails = FundDetail::where('student_id', $id)
            ->where('type', 16)
            ->where('donar_id', $donar_id)
            ->where('payment_type', 1)
            ->whereIn('receipt_no', [$receipt])
            ->get();

        if ($fundDetails->isEmpty()) {
            abort(404, 'No payment data found.');
        }

        $months = $fundDetails->pluck('month')->unique()->sort();

        $totalReceived = FundDetail::where('student_id', $id)
            ->where('type', 16)
            ->where('donar_id', $donar_id)
            ->where('payment_type', 1)
            ->whereIn('month', $months)
            ->sum(DB::raw('amount + COALESCE(discount, 0)'));

        $totalDiscount = $fundDetails->sum('discount');

        $invoice = FundDetail::where('student_id', $id)
            ->where('type', 16)
            ->where('donar_id', $donar_id)
            ->latest()
            ->first();

        $currentEntries = $fundDetails->filter(function ($item) use ($invoice) {
            return $item->date == $invoice->date && $item->receipt_no == $invoice->receipt_no;
        });

        $currentReceived = $currentEntries->sum(function ($entry) {
            return $entry->amount + ($entry->discount ?? 0);
        });

        $previousReceived = $totalReceived - $currentReceived;

        $studentData = Student::findOrFail($id);
        $donar       = Donar::findOrFail($donar_id);

        $academicAmount = 0;
        $hostelAmount   = 0;
        $otherAmount    = 0;

        foreach ($months as $month) {
            $academicAmount += FeeAssignDetail::where('type', 1)
                ->where('class_id', $studentData->class_id)
                ->where('session_id', $studentData->session_id)
                ->where('month', $month)
                ->sum('amount');

            if ($studentData->student_type == 2) {
                $hostelAmount += FeeAssignDetail::where('type', 2)
                    ->where('class_id', $studentData->class_id)
                    ->where('session_id', $studentData->session_id)
                    ->where('month', $month)
                    ->sum('amount');
            }

            $otherAmount += FeeAssignDetail::where('type', 3)
                ->where('class_id', $studentData->class_id)
                ->where('session_id', $studentData->session_id)
                ->where('month', $month)
                ->sum('amount');
        }

        $totalAmount = $academicAmount + $hostelAmount + $otherAmount - $totalDiscount;

        $design = InvoiceDesign::first();

        return view('dashboard.payments.sponsor_student_payment_print', compact(
            'donar',
            'design',
            'fundDetails',
            'months',
            'academicAmount',
            'hostelAmount',
            'otherAmount',
            'totalAmount',
            'totalReceived',
            'totalDiscount',
            'invoice',
            'currentReceived',
            'previousReceived',
            'studentData',
        ));
    }

    //payment due
    public function PaymentDue(Request $request)
    {
        try {
            if ($request->ajax()) {

                // ---------------- Base (grouped by class + month) ----------------
                $base = DB::table('fee_assign_details')
                    ->join('students', 'fee_assign_details.class_id', '=', 'students.class_id')
                    ->join('student_classes', 'fee_assign_details.class_id', '=', 'student_classes.id')
                    ->whereNull('fee_assign_details.deleted_at');

                if ($request->filled('class_id')) {
                    $base->where('fee_assign_details.class_id', $request->class_id);
                }
                if ($request->filled('month')) { // expects 'YYYY-MM'
                    $base->where('fee_assign_details.month', $request->month);
                }

                $grouped = (clone $base)
                    ->groupBy('fee_assign_details.class_id', 'fee_assign_details.month')
                    ->select(
                        'fee_assign_details.class_id as class_id',
                        'fee_assign_details.month as month',
                        'student_classes.name as class_name',
                        DB::raw('MIN(fee_assign_details.session_id) as session_id')
                    );

                $rows = $grouped->get();
                if ($rows->isEmpty()) {
                    return DataTables::of($grouped)
                        ->with(['summary' => [
                            'students' => 0, 'assigned' => 0.0, 'received' => 0.0, 'discount' => 0.0, 'due' => 0.0,
                        ]])->toJson();
                }

                // ------------ helpers ------------
                $K        = fn($cid, $m) => $cid . '|' . $m;
                $classIds = $rows->pluck('class_id')->unique()->values();
                $months   = $rows->pluck('month')->unique()->values();

                // ============ Eligible students per (class, month) by join_date ============
                // 1) Load all active students of the classes (once)
                $allStudents = DB::table('students')
                    ->whereIn('class_id', $classIds)
                    ->where(function ($q) {$q->where('status', 1)->orWhereNull('status');})
                    ->select('id', 'class_id', 'student_type', 'join_date')
                    ->get()
                    ->groupBy('class_id'); // class_id => collection

                                      // 2) Build per (class,month) counts considering join_date <= month (or NULL)
                $eligibleCounts = []; // key => ['total'=>, 'hostel'=>, 'onab'=>]
                foreach ($rows as $r) {
                    $key             = $K($r->class_id, $r->month);
                    $ym              = substr((string) $r->month, 0, 7); // 'YYYY-MM'
                    $studentsOfClass = $allStudents->get($r->class_id, collect());

                    $total  = 0;
                    $hostel = 0;
                    $onab   = 0;

                    foreach ($studentsOfClass as $st) {
                        // join_date NULL => eligible
                        if (empty($st->join_date)) {
                            $total++;
                            if ((int) $st->student_type === 2) {
                                $hostel++;
                            }

                            if ((int) $st->student_type === 1) {
                                $onab++;
                            }

                            continue;
                        }
                        // join_date month <= target month?
                        $joinYm = \Carbon\Carbon::parse($st->join_date)->format('Y-m');
                        if ($joinYm <= $ym) {
                            $total++;
                            if ((int) $st->student_type === 2) {
                                $hostel++;
                            }

                            if ((int) $st->student_type === 1) {
                                $onab++;
                            }

                        }
                    }

                    $eligibleCounts[$key] = ['total' => $total, 'hostel' => $hostel, 'onab' => $onab];
                }

                // ============ Fee assigns & payments  ============
                // Type 1..4: sum per (class,month,type)
                $feeTypeSums = DB::table('fee_assign_details')
                    ->whereIn('class_id', $classIds)
                    ->whereIn('month', $months)
                    ->whereIn('type', [1, 2, 3, 4])
                    ->whereNull('deleted_at')
                    ->select('class_id', 'month', 'type', DB::raw('SUM(amount) as total'))
                    ->groupBy('class_id', 'month', 'type')
                    ->get()
                    ->groupBy(fn($r) => $K($r->class_id, $r->month)); // key => collection

                // Type=5: special total per (class,month)
                $specialByClassMonth = DB::table('fee_assign_details as fd')
                    ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id')
                    ->whereNull('fd.deleted_at')
                    ->whereNull('fds.deleted_at')
                    ->where('fd.type', 5)
                    ->whereIn('fd.class_id', $classIds)
                    ->whereIn('fd.month', $months)
                    ->select('fd.class_id', 'fd.month', DB::raw('SUM(fds.amount) as total'))
                    ->groupBy('fd.class_id', 'fd.month')
                    ->get()
                    ->keyBy(fn($r) => $K($r->class_id, $r->month));

                // OLD payments per (class,month)
                $oldByClassMonth = DB::table('fund_details as f')
                    ->join('students as s', 's.id', '=', 'f.student_id')
                    ->whereIn('s.class_id', $classIds)
                    ->whereIn('f.month', $months)
                    ->whereIn('f.type', [6, 16])
                    ->where('f.payment_type', 1)
                    ->whereNull('f.deleted_at')
                    ->whereNull('f.fund_detail_type_id')
                    ->whereNull('f.fee_id')
                    ->whereNull('f.receipt_type')
                    ->groupBy('s.class_id', 'f.month')
                    ->select(
                        's.class_id', 'f.month',
                        DB::raw('COALESCE(SUM(f.amount),0) as total_received'),
                        DB::raw('COALESCE(SUM(f.discount),0) as total_discount')
                    )
                    ->get()
                    ->keyBy(fn($r) => $K($r->class_id, $r->month));

                // NEW receive per (class,month)  (MoneyReceipt.current_receive)
                $newReceiveByClassMonth = DB::table('money_receipts as mr')
                    ->join('students as s', 's.id', '=', 'mr.student_id')
                    ->whereIn('s.class_id', $classIds)
                    ->whereIn('mr.month', $months)
                    ->where('mr.receipt_type', 'new')
                    ->whereNull('mr.deleted_at')
                    ->groupBy('s.class_id', 'mr.month')
                    ->select('s.class_id', 'mr.month', DB::raw('SUM(mr.current_receive) as total'))
                    ->get()
                    ->keyBy(fn($r) => $K($r->class_id, $r->month));

                // NEW discount per (class,month) from fund_details_type
                $newDiscountByClassMonth = DB::table('fund_details_type as fdt')
                    ->join('students as s', 's.id', '=', 'fdt.student_id')
                    ->whereIn('s.class_id', $classIds)
                    ->whereIn('fdt.month', $months)
                    ->where('fdt.receipt_type', 'new')
                    ->whereNull('fdt.deleted_at')
                    ->groupBy('s.class_id', 'fdt.month')
                    ->select('s.class_id', 'fdt.month', DB::raw('SUM(fdt.discount) as total'))
                    ->get()
                    ->keyBy(fn($r) => $K($r->class_id, $r->month));

                // -------- compute per-row + global summary --------
                $computed = [];
                $summary  = ['students' => 0, 'assigned' => 0.0, 'received' => 0.0, 'discount' => 0.0, 'due' => 0.0];

                foreach ($rows as $r) {
                    $key = $K($r->class_id, $r->month);

                    // ✅ মাসভিত্তিক eligible counts (join_date guard)
                    $eligible      = $eligibleCounts[$key] ?? ['total' => 0, 'hostel' => 0, 'onab' => 0];
                    $totalStudents = (int) $eligible['total'];
                    $hostel        = (int) $eligible['hostel'];
                    $onab          = (int) $eligible['onab'];

                    // type sums
                    $types = ($feeTypeSums[$key] ?? collect())->keyBy('type');
                    $t1    = (float) ($types->get(1)->total ?? 0); // academic
                    $t2    = (float) ($types->get(2)->total ?? 0); // hostel
                    $t3    = (float) ($types->get(3)->total ?? 0); // other
                    $t4    = (float) ($types->get(4)->total ?? 0); // onabashik

                    // ✅ Assigned = type(1,3) * eligible total + type(2)*eligible hostel + type(4)*eligible onab + special
                    $assigned = ($t1 + $t3) * $totalStudents
                     + ($t2 * $hostel)
                     + ($t4 * $onab)
                     + (float) ($specialByClassMonth[$key]->total ?? 0.0);

                    // Prefer OLD; else NEW
                    $old = $oldByClassMonth[$key] ?? null;
                    $nR  = (float) ($newReceiveByClassMonth[$key]->total ?? 0);
                    $nD  = (float) ($newDiscountByClassMonth[$key]->total ?? 0);

                    if ($old && (float) $old->total_received > 0) {
                        $received = (float) $old->total_received;
                        $discount = (float) $old->total_discount;
                    } elseif ($nR > 0) {
                        $received = $nR;
                        $discount = $nD;
                    } else {
                        $received = 0.0;
                        $discount = 0.0;
                    }

                    $due = max(0, $assigned - ($received + $discount));

                    $computed[$key] = (object) [
                        'students' => $totalStudents, 'assigned' => $assigned,
                        'received' => $received, 'discount'      => $discount, 'due' => $due,
                    ];

                    $summary['students'] += $totalStudents;
                    $summary['assigned'] += $assigned;
                    $summary['received'] += $received;
                    $summary['discount'] += $discount;
                    $summary['due'] += $due;
                }

                // -------------- DataTables response --------------
                return DataTables::of($grouped)
                    ->addColumn('month', function ($row) {
                        try {
                            return \Carbon\Carbon::createFromFormat('Y-m', trim($row->month))->format('M-Y');
                        } catch (\Throwable $e) {
                            return $row->month ?? '--';
                        }
                    })
                    ->addColumn('class', function ($row) {
                        return $row->class_name ?? '--';
                    })
                    ->addColumn('details', function ($row) use ($computed, $K) {
                        $key = $K($row->class_id, $row->month);
                        $c   = $computed[$key] ?? (object) ['students' => 0, 'assigned' => 0, 'received' => 0, 'discount' => 0, 'due' => 0];

                        return '<div class="text-start small">'
                        . 'Students: <b>' . number_format($c->students) . '</b><br>'
                        . 'Assigned: <b>' . number_format($c->assigned, 2) . '</b><br>'
                        . 'Received: <b>' . number_format($c->received, 2) . '</b><br>'
                        . 'Discount: <b>' . number_format($c->discount, 2) . '</b><br>'
                        . 'Due: <b>' . number_format($c->due, 2) . '</b>'
                            . '</div>';
                    })
                    ->addColumn('action', function ($row) {
                        $url = route('payment.due.details.month.class', [
                            'month'    => $row->month,
                            'class_id' => $row->class_id,
                        ]);
                        return '<a class="btn btn-sm btn-primary" href="' . $url . '" title="Details"><i class="fa fa-eye"></i></a>';
                    })
                    ->addIndexColumn()
                    ->rawColumns(['month', 'class', 'details', 'action'])
                    ->with([
                        'summary' => [
                            'students' => (int) $summary['students'],
                            'assigned' => round($summary['assigned'], 2),
                            'received' => round($summary['received'], 2),
                            'discount' => round($summary['discount'], 2),
                            'due'      => round($summary['due'], 2),
                        ],
                    ])
                    ->toJson();
            }

            // Non-AJAX
            $classes = \App\Models\StudentClass::whereNull('deleted_at')
                ->select('id', 'name')->orderBy('name')->get();

            return view('dashboard.payments.payment_due', compact('classes'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    /**
     * Compute summary numbers for a single grouped row (class_id + month)
     * Returns: [studentCount, assigned, received, discount, due]
     */
    private function calculateSummaryRow($row): array
    {
        $classId   = (int) $row->class_id;
        $sessionId = (int) $row->session_id;
        $month     = $row->month;

        // Students in class (include NULL status or status != 2)
        $studentIds = Student::whereNull('deleted_at')
            ->where('class_id', $classId)
            ->where(function ($q) {
                $q->whereNull('status')->orWhere('status', '!=', 2);
            })
            ->pluck('id');

        $studentCount = $studentIds->count();
        if ($studentCount === 0) {
            return [0, 0.0, 0.0, 0.0, 0.0];
        }

        // Base fees (types 1–4) per student (same for all students in class)
        $basePerStudent = (float) FeeAssignDetail::whereNull('deleted_at')
            ->where('class_id', $classId)
            ->where('session_id', $sessionId)
            ->where('month', $month)
            ->whereIn('type', [1, 2, 3, 4])
            ->sum('amount');

        // Special fees (type 5) only for students who have it
        $totalSpecialAll = (float) DB::table('fee_assign_details as fd')
            ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id')
            ->whereNull('fd.deleted_at')
            ->whereNull('fds.deleted_at')
            ->where('fd.type', 5)
            ->where('fd.session_id', $sessionId)
            ->where('fd.class_id', $classId)
            ->where('fd.month', $month)
            ->whereIn('fds.student_id', $studentIds)
            ->sum('fds.amount');

        // Assigned total for the class-row
        $assigned = ($basePerStudent * $studentCount) + $totalSpecialAll;

        // Payments & Discounts (old + new)
        $old = FundDetail::whereIn('student_id', $studentIds)
            ->where('month', $month)
            ->whereIn('type', [6, 16])
            ->where('payment_type', 1)
            ->whereNull('deleted_at')
            ->whereNull('fund_detail_type_id')
            ->whereNull('fee_id')
            ->whereNull('receipt_type')
            ->selectRaw('COALESCE(SUM(amount),0) as received, COALESCE(SUM(discount),0) as discount')
            ->first();

        $newReceive = (float) MoneyReceipt::whereIn('student_id', $studentIds)
            ->where('month', $month)
            ->where('receipt_type', 'new')
            ->whereNull('deleted_at')
            ->sum('current_receive');

        $newDiscount = (float) FundDetailType::whereIn('student_id', $studentIds)
            ->where('month', $month)
            ->where('receipt_type', 'new')
            ->whereNull('deleted_at')
            ->sum('discount');

        $received = (float) ($old->received ?? 0) + $newReceive;
        $discount = (float) ($old->discount ?? 0) + $newDiscount;
        $due      = max(0.0, $assigned - ($received + $discount));

        return [$studentCount, $assigned, $received, $discount, $due];
    }

    /**
     * Render the HTML for the Details column for a single grouped row
     */
    private function renderDetailsHtml($row): string
    {
        $classId   = (int) $row->class_id;
        $sessionId = (int) $row->session_id;
        $month     = $row->month;

        // Students
        $studentIds = Student::whereNull('deleted_at')
            ->where('class_id', $classId)
            ->where(function ($q) {
                $q->whereNull('status')->orWhere('status', '!=', 2);
            })
            ->pluck('id');

        $studentCount = $studentIds->count();
        if ($studentCount === 0) {
            return '<div class="text-start">
                <div><strong>Students:</strong> 0</div>
                <div><strong>Total Assigned (class):</strong> 0.00</div>
                <div><strong>Received:</strong> 0.00</div>
                <div><strong>Discount:</strong> 0.00</div>
                <div><strong>Due:</strong> <span class="badge bg-danger">0.00</span></div>
            </div>';
        }

        // Base per student (types 1–4)
        $assigned = FeeAssignDetail::selectRaw("
                COALESCE(SUM(CASE WHEN type = 1 THEN amount ELSE 0 END),0) AS academic,
                COALESCE(SUM(CASE WHEN type = 2 THEN amount ELSE 0 END),0) AS hostel,
                COALESCE(SUM(CASE WHEN type = 3 THEN amount ELSE 0 END),0) AS other_fee,
                COALESCE(SUM(CASE WHEN type = 4 THEN amount ELSE 0 END),0) AS onabashik
            ")
            ->whereNull('deleted_at')
            ->where('class_id', $classId)
            ->where('session_id', $sessionId)
            ->where('month', $month)
            ->first();

        $basePerStudent =
        (float) ($assigned->academic ?? 0) +
        (float) ($assigned->hostel ?? 0) +
        (float) ($assigned->other_fee ?? 0) +
        (float) ($assigned->onabashik ?? 0);

        // Special by student (type 5)
        $specialByStudent = DB::table('fee_assign_details as fd')
            ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id')
            ->whereNull('fd.deleted_at')
            ->whereNull('fds.deleted_at')
            ->where('fd.type', 5)
            ->where('fd.session_id', $sessionId)
            ->where('fd.class_id', $classId)
            ->where('fd.month', $month)
            ->whereIn('fds.student_id', $studentIds)
            ->select('fds.student_id', DB::raw('SUM(fds.amount) AS amount'))
            ->groupBy('fds.student_id')
            ->get()
            ->pluck('amount', 'student_id'); // [student_id => amount]

        $totalSpecialAll = (float) $specialByStudent->sum();

        // Class-row totals
        $totalAssignedAll = ($basePerStudent * $studentCount) + $totalSpecialAll;

        // Payments & discounts
        $old = FundDetail::whereIn('student_id', $studentIds)
            ->where('month', $month)
            ->whereIn('type', [6, 16])
            ->where('payment_type', 1)
            ->whereNull('deleted_at')
            ->whereNull('fund_detail_type_id')
            ->whereNull('fee_id')
            ->whereNull('receipt_type')
            ->selectRaw('COALESCE(SUM(amount),0) as received, COALESCE(SUM(discount),0) as discount')
            ->first();

        $newReceive = (float) MoneyReceipt::whereIn('student_id', $studentIds)
            ->where('month', $month)
            ->where('receipt_type', 'new')
            ->whereNull('deleted_at')
            ->sum('current_receive');

        $newDiscount = (float) FundDetailType::whereIn('student_id', $studentIds)
            ->where('month', $month)
            ->where('receipt_type', 'new')
            ->whereNull('deleted_at')
            ->sum('discount');

        $totalReceivedAll = (float) ($old->received ?? 0) + $newReceive;
        $totalDiscountAll = (float) ($old->discount ?? 0) + $newDiscount;
        $dueAll           = max(0.0, $totalAssignedAll - ($totalReceivedAll + $totalDiscountAll));

        // Preview line for first few students with special
        $firstThree = $specialByStudent->take(3)->map(function ($amt, $sid) use ($basePerStudent) {
            return "SID {$sid}: Base " . number_format($basePerStudent, 2) . ", Special " . number_format((float) $amt, 2);
        })->implode(' | ');

        return '<div class="text-start">
            <div><strong>Students:</strong> ' . $studentCount . '</div>
            <div><strong>Base (per student):</strong> ' . number_format((float) $basePerStudent, 2) . '</div>
            <div><strong>Special (sum of assigned):</strong> ' . number_format($totalSpecialAll, 2) . '</div>
            <div><strong>Total Assigned (class):</strong> ' . number_format($totalAssignedAll, 2) . '</div>
            <div><strong>Received:</strong> ' . number_format($totalReceivedAll, 2) . '</div>
            <div><strong>Discount:</strong> ' . number_format($totalDiscountAll, 2) . '</div>
            <div><strong>Due:</strong> <span class="badge bg-danger">' . number_format($dueAll, 2) . '</span></div>
            <div><small>' . e($firstThree) . ($specialByStudent->count() > 3 ? ' ...' : '') . '</small></div>
        </div>';
    }
    public function PaymentDueDetailsMonthClass(Request $request, $month, $class_id)
    {
        try {
            $funds = \App\Models\Fund::whereNull('deleted_at')->get();

            // (A) Serve fee list for the dropdown (ALL fees assigned to this class+month)
            if ($request->ajax() && $request->boolean('list_fees')) {
                $fees = DB::table('fee_assign_details as fad')
                    ->join('student_fees as sf', 'sf.id', '=', 'fad.fee_id')
                    ->whereNull('fad.deleted_at')
                    ->where('fad.class_id', $class_id)
                    ->where('fad.month', $month)
                    ->groupBy('fad.fee_id', 'sf.name')
                    ->orderBy('sf.name')
                    ->select('fad.fee_id as id', 'sf.name as text')
                    ->get();

                return response()->json($fees);
            }

            if ($request->ajax()) {
                $filterFeeId = $request->filled('filter_fee_id') ? (int) $request->get('filter_fee_id') : null;

                                                      // =============== JOIN MONTH GUARD ===============
                                                      // এই মাসের রিপোর্টে শুধু তাদেরই ধরো যারা এর আগেই বা এই মাসেই ভর্তি (join) হয়েছে।
                                                      // join_date NULL হলে allow করা হবে।
                $ymForCompare = substr($month, 0, 7); // ensure 'YYYY-MM'

                // ---------------- Base dataset ----------------
                $baseQuery = DB::table('fee_assign_details')
                    ->join('students', 'fee_assign_details.class_id', '=', 'students.class_id')
                    ->join('student_classes', 'fee_assign_details.class_id', '=', 'student_classes.id')
                    ->join('student_sessions', 'fee_assign_details.session_id', '=', 'student_sessions.id')
                    ->whereNull('fee_assign_details.deleted_at')
                    ->where('fee_assign_details.month', $month)
                    ->where(function ($q) {
                        $q->where('students.status', 1)->orWhereNull('students.status');
                    })
                    ->where('fee_assign_details.class_id', $class_id)
                // ✅ নতুন শর্ত: students.join_date <= $month (বা join_date NULL)
                    ->where(function ($q) use ($ymForCompare) {
                        $q->whereNull('students.join_date')
                            ->orWhere(DB::raw("DATE_FORMAT(students.join_date, '%Y-%m')"), '<=', $ymForCompare);
                    })
                    ->groupBy('students.id')
                    ->select(
                        'fee_assign_details.class_id',
                        'fee_assign_details.session_id',
                        'fee_assign_details.month',
                        'students.id as student_id',
                        'students.name as student_name',
                        'student_classes.name as class_name',
                        'student_sessions.name as session_name'
                    );

                if ($filterFeeId) {
                    $fid = $filterFeeId;

                    // ensure assigned
                    $baseQuery->whereExists(function ($q) use ($month, $fid) {
                        $q->select(DB::raw(1))
                            ->from('fee_assign_details as fad')
                            ->whereNull('fad.deleted_at')
                            ->whereColumn('fad.class_id', 'students.class_id')
                            ->where('fad.month', $month)
                            ->where('fad.fee_id', $fid);
                    });

                    // ensure NOT collected (new)
                    $baseQuery->whereExists(function ($q) use ($month, $fid) {
                        $q->select(DB::raw(1))
                            ->from('fee_assign_details as fad2')
                            ->leftJoin('fund_details as fd2', function ($join) use ($month) {
                                $join->on('fd2.fee_id', '=', 'fad2.fee_id')
                                    ->whereColumn('fd2.student_id', 'students.id')
                                    ->where('fd2.month', $month)
                                    ->whereNull('fd2.deleted_at')
                                    ->where('fd2.receipt_type', 'new');
                            })
                            ->whereNull('fad2.deleted_at')
                            ->whereColumn('fad2.class_id', 'students.class_id')
                            ->where('fad2.month', $month)
                            ->where('fad2.fee_id', $fid)
                            ->whereNull('fd2.id');
                    });
                }

                // Fetch base rows and IDs for preloading maps
                $rows = $baseQuery->get();
                if ($rows->isEmpty()) {
                    return DataTables::of($rows)->make(true);
                }
                $studentIds = $rows->pluck('student_id')->unique()->values();

                // ---------------- Preload: student info map ----------------
                $studentInfo = \App\Models\Student::whereIn('id', $studentIds)
                    ->get(['id', 'father_name', 'gurdian_mobile', 'gurdian_address', 'student_type', 'class_id', 'session_id'])
                    ->keyBy('id');

                // ---------------- per-session (types 1..4) sums ----------------
                $feeTypeSums = \App\Models\FeeAssignDetail::where('class_id', $class_id)
                    ->where('month', $month)
                    ->whereIn('type', [1, 2, 3, 4])
                    ->select('session_id', 'type', DB::raw('SUM(amount) as total'))
                    ->groupBy('session_id', 'type')
                    ->get()
                    ->groupBy('session_id');

                // ---------------- special (type=5) per student sum ----------------
                $specialSumByStudent = DB::table('fee_assign_details as fd')
                    ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id')
                    ->whereNull('fd.deleted_at')
                    ->whereNull('fds.deleted_at')
                    ->where('fd.type', 5)
                    ->where('fd.class_id', $class_id)
                    ->where('fd.month', $month)
                    ->whereIn('fds.student_id', $studentIds)
                    ->select('fds.student_id', DB::raw('SUM(fds.amount) as total'))
                    ->groupBy('fds.student_id')
                    ->pluck('total', 'student_id');

                // ---------------- fee list per student with status ----------------
                $subA = DB::table('fee_assign_details as fad')
                    ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fad.id')
                    ->join('student_fees as sf', 'sf.id', '=', 'fad.fee_id')
                    ->whereNull('fad.deleted_at')
                    ->whereNull('fds.deleted_at')
                    ->where('fad.type', 5)
                    ->where('fad.class_id', $class_id)
                    ->where('fad.month', $month)
                    ->whereIn('fds.student_id', $studentIds)
                    ->select(
                        'fds.student_id',
                        'fad.fee_id',
                        'sf.name as fee_name',
                        DB::raw('fds.amount as amount')
                    );

                $subB = DB::table('fee_assign_details as fad')
                    ->join('student_fees as sf', 'sf.id', '=', 'fad.fee_id')
                    ->join('students as s', 's.class_id', '=', 'fad.class_id')
                    ->whereNull('fad.deleted_at')
                    ->whereIn('fad.type', [1, 2, 3, 4])
                    ->where('fad.class_id', $class_id)
                    ->where('fad.month', $month)
                    ->whereIn('s.id', $studentIds) // only eligible students
                    ->select(
                        's.id as student_id',
                        'fad.fee_id',
                        'sf.name as fee_name',
                        DB::raw('fad.amount as amount')
                    );

                $feesSub = $subA->unionAll($subB);

                $feesPerStudent = DB::query()->fromSub($feesSub, 't')
                    ->leftJoin('fund_details as fd', function ($join) use ($month) {
                        $join->on('fd.fee_id', '=', 't.fee_id')
                            ->on('fd.student_id', '=', 't.student_id')
                            ->where('fd.month', $month)
                            ->whereNull('fd.deleted_at')
                            ->where('fd.receipt_type', 'new');
                    });

                if ($filterFeeId) {
                    $feesPerStudent->where('t.fee_id', $filterFeeId);
                }

                $feesPerStudent = $feesPerStudent
                    ->select(
                        't.student_id',
                        't.fee_id',
                        't.fee_name',
                        't.amount',
                        DB::raw("CASE WHEN fd.id IS NULL THEN 'Not Collect' ELSE 'Collected' END AS status")
                    )
                    ->get()
                    ->groupBy('student_id');

                // ---------------- payments (old/new) grouped ----------------
                $oldSums = \App\Models\FundDetail::whereIn('student_id', $studentIds)
                    ->where('month', $month)
                    ->whereIn('type', [6, 16])
                    ->where('payment_type', 1)
                    ->whereNull('deleted_at')
                    ->whereNull('fund_detail_type_id')
                    ->whereNull('fee_id')
                    ->whereNull('receipt_type')
                    ->select(
                        'student_id',
                        DB::raw('COALESCE(SUM(amount),0) as total_received'),
                        DB::raw('COALESCE(SUM(discount),0) as total_discount')
                    )
                    ->groupBy('student_id')
                    ->get()
                    ->keyBy('student_id');

                // NEW format sums (⚠️ তোমার স্কিমা অনুযায়ী ঠিক রেখেছি: FundDetail + receipt_type=new)
                $newReceive = \App\Models\FundDetail::whereIn('student_id', $studentIds)
                    ->where('month', $month)
                    ->where('receipt_type', 'new')
                    ->whereNull('deleted_at')
                    ->select('student_id', DB::raw('SUM(COALESCE(amount,0)) as total'))
                    ->groupBy('student_id')
                    ->pluck('total', 'student_id');

                $newDiscount = \App\Models\FundDetailType::whereIn('student_id', $studentIds)
                    ->where('month', $month)
                    ->where('receipt_type', 'new')
                    ->whereNull('deleted_at')
                    ->select('student_id', DB::raw('SUM(COALESCE(discount,0)) as total'))
                    ->groupBy('student_id')
                    ->pluck('total', 'student_id');

                // ---------------- feeIds for action dropdown ----------------
                $feeIdsForAction = DB::table('fee_assign_details as fad')
                    ->join('student_fees as sf', 'fad.fee_id', '=', 'sf.id')
                    ->whereNull('fad.deleted_at')
                    ->where('fad.month', $month)
                    ->where('fad.class_id', $class_id)
                    ->pluck('fad.fee_id', 'sf.name')
                    ->toArray();

                // ---------------- Build DataTable ----------------
                return DataTables::of($rows)

                    ->addColumn('name', function ($row) {
                        return $row->student_name ?? '--';
                    })

                    ->addColumn('details', function ($row) use ($studentInfo) {
                        $st = $studentInfo->get($row->student_id);
                        if (! $st) {
                            return '--';
                        }

                        $father  = $st->father_name ?? '--';
                        $mobile  = $st->gurdian_mobile ?? '--';
                        $address = $st->gurdian_address ?? '--';
                        return "Father: {$father}<br>Mobile: {$mobile}<br>Address: {$address}";
                    })

                    ->addColumn('fee_details', function ($row) use ($feesPerStudent, $filterFeeId) {
                        $list = $feesPerStudent->get($row->student_id, collect());
                        if ($list->isEmpty()) {
                            return '<span class="badge bg-secondary">No fees assigned</span>';
                        }

                        if ($filterFeeId) {
                            $list         = $list->where('fee_id', $filterFeeId)->values();
                            $notCollected = $list->where('status', 'Not Collect')->values();
                            if ($notCollected->isEmpty()) {
                                return '<span class="badge bg-success">Selected fee already collected</span>';
                            }
                            $list = $notCollected;
                        }

                        // groupBy fee_id to avoid duplicates per student
                        $preview = $list->groupBy('fee_id')->map(function ($f) {
                            $item = $f->first();
                            $amt  = number_format((float) $item->amount, 2);
                            $st   = $item->status === 'Collected'
                                ? '<span class="badge bg-success">Collected</span>'
                                : '<span class="badge bg-warning text-dark">Not Collect</span>';
                            return e($item->fee_name) . " — {$amt} {$st}";
                        })->implode('<br>');

                        return '<div class="text-start"><div class="mt-1 small">' . $preview . '</div></div>';
                    })

                    ->addColumn('payment_details', function ($row) use ($studentInfo, $feeTypeSums, $specialSumByStudent, $oldSums, $newReceive, $newDiscount) {
                        $st = $studentInfo->get($row->student_id);
                        if (! $st) {
                            return '--';
                        }

                        $session_id   = (int) $st->session_id;
                        $student_type = (int) $st->student_type;

                        $group           = $feeTypeSums->get($session_id, collect())->keyBy('type');
                        $academicAmount  = (float) ($group->get(1)->total ?? 0);
                        $hostelAmount    = $student_type === 2 ? (float) ($group->get(2)->total ?? 0) : 0.0;
                        $otherAmount     = (float) ($group->get(3)->total ?? 0);
                        $onabashikAmount = $student_type === 1 ? (float) ($group->get(4)->total ?? 0) : 0.0;
                        $specialAmount   = (float) ($specialSumByStudent[$row->student_id] ?? 0);

                        $monthFee = $academicAmount + $hostelAmount + $otherAmount + $onabashikAmount + $specialAmount;

                        $old         = $oldSums->get($row->student_id);
                        $receivedOld = (float) ($old->total_received ?? 0);
                        $discountOld = (float) ($old->total_discount ?? 0);

                        $newR = (float) ($newReceive[$row->student_id] ?? 0);
                        $newD = (float) ($newDiscount[$row->student_id] ?? 0);

                        if ($receivedOld > 0) {
                            $paid = $receivedOld; $discount = $discountOld;
                        } elseif ($newR > 0) {
                            $paid = $newR; $discount = $newD;
                        } else {
                            $paid = 0.0; $discount = 0.0;
                        }

                        $due = max(0, $monthFee - ($paid + $discount));

                        return "Total Fee: " . number_format($monthFee, 2) . "<br>"
                        . "Paid: " . number_format($paid, 2) . "<br>"
                        . "Discount: " . number_format($discount, 2) . "<br>"
                        . "Due: " . number_format($due, 2) . "<br>"
                            . "<input type='hidden' class='studentIds' data-due='{$due}' value='{$row->student_id}'>";
                    })

                    ->addColumn('action', function ($row) use ($feeIdsForAction, $oldSums) {
                        $feeJson = htmlspecialchars(json_encode($feeIdsForAction, JSON_UNESCAPED_UNICODE), ENT_QUOTES, 'UTF-8');

                        $viewUrl = route('payment.due.view.month.class.student', [
                            'month'      => $row->month,
                            'class_id'   => $row->class_id,
                            'student_id' => $row->student_id,
                        ]);

                        $collectUrl = route('payment.create', [
                            'month'      => $row->month,
                            'class_id'   => $row->class_id,
                            'student_id' => $row->student_id,
                        ]);

                        $hasOld = ((float) ($oldSums->get($row->student_id)->total_received ?? 0)) > 0;

                        $html = '<div class="d-flex flex-wrap gap-2">';
                        $html .= '<a class="btn btn-outline-primary btn-sm" href="' . $viewUrl . '" data-bs-toggle="tooltip" title="Details">' .
                            '<i class="fa fa-eye"></i> <span class="d-none d-md-inline">View</span>' .
                            '</a>';

                        $html .= '<button type="button" class="btn btn-outline-success btn-sm send-sms" ' .
                        'data-student="' . $row->student_id . '" ' .
                        'data-month="' . $row->month . '" ' .
                        'data-class="' . $row->class_id . '" ' .
                            'data-bs-toggle="tooltip" title="Send SMS">' .
                            '<i class="fa fa-envelope"></i> <span class="d-none d-md-inline">Message</span>' .
                            '</button>';

                        if ($hasOld) {
                            $html .= '<button type="button" class="btn btn-outline-warning btn-sm transfer-to-new" ' .
                            'data-student="' . $row->student_id . '" ' .
                            'data-month="' . $row->month . '" ' .
                            'data-class="' . $row->class_id . '" ' .
                            'data-name="' . htmlspecialchars($row->student_name ?? '--', ENT_QUOTES, 'UTF-8') . '" ' .
                                'data-fee=\'' . $feeJson . '\' ' .
                                'data-bs-toggle="tooltip" title="Transfer old payments to new">' .
                                '<i class="fa fa-exchange"></i> <span class="d-none d-md-inline">Transfer</span>' .
                                '</button>';
                        } else {
                            $html .= '<a class="btn btn-outline-secondary btn-sm" href="' . $collectUrl . '" data-bs-toggle="tooltip" title="Collect">' .
                                '<i class="fa fa-money"></i> <span class="d-none d-md-inline">Collect</span>' .
                                '</a>';
                        }

                        $html .= '</div>';
                        return $html;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['name', 'details', 'fee_details', 'payment_details', 'action'])
                    ->toJson();
            }

            return view('dashboard.payments.payment_due_details', [
                'month'    => $month,
                'class_id' => $class_id,
                'funds'    => $funds,
            ]);
        } catch (\Exception $exception) {
            dd($exception);
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function PaymentDueViewMonthClassStudent(Request $request, $month, $class_id, $student_id)
    {
        try {
            if ($request->ajax()) {

                // Base query with LEFT JOIN to fund_details to compute status efficiently
                $query = DB::table('fee_assign_details as fad')
                    ->join('students as s', 'fad.class_id', '=', 's.class_id')
                    ->join('student_classes as sc', 'fad.class_id', '=', 'sc.id')
                    ->join('student_sessions as ss', 'fad.session_id', '=', 'ss.id')
                    ->join('student_fees as sf', 'fad.fee_id', '=', 'sf.id')
                    ->leftJoin('fund_details as fd', function ($join) {
                        $join->on('fd.student_id', '=', 's.id')
                            ->on('fd.month', '=', 'fad.month')
                            ->on('fd.fee_id', '=', 'fad.fee_id')
                            ->whereNull('fd.deleted_at')
                            ->where('fd.receipt_type', 'new');
                    })
                    ->whereNull('fad.deleted_at')
                    ->where('fad.month', $month)
                    ->where('fad.class_id', $class_id)
                    ->where('s.id', $student_id)
                    ->groupBy('fad.fee_id')
                    ->select(
                        'fad.*',
                        's.id as student_id',
                        's.name as student_name',
                        'sc.name as class_name',
                        'ss.name as session_name',
                        'sf.name as fee_name',
                        DB::raw("CASE WHEN fd.id IS NULL THEN 'Not Collect' ELSE 'Collected' END AS status")
                    );

                // Custom filters coming from DataTables ajax.data
                if ($request->filled('status')) {
                    if ($request->status === 'Not Collect') {
                        $query->whereNull('fd.id');
                    } elseif ($request->status === 'Collected') {
                        $query->whereNotNull('fd.id');
                    }
                }

                if ($request->filled('fee_name')) {
                    $query->where('sf.name', 'like', '%' . $request->fee_name . '%');
                }

                return DataTables::of($query)
                // allow global search on computed "status"
                    ->filterColumn('status', function ($q, $keyword) {
                        $kw = strtolower($keyword);
                        if (str_contains('collected', $kw)) {
                            $q->whereNotNull('fd.id');
                        } elseif (str_contains('not collect', $kw) || str_contains('not', $kw)) {
                            $q->whereNull('fd.id');
                        }
                    })
                    ->addColumn('amount', function ($row) {
                        return $row->amount ?? '--';
                    })

                    ->addIndexColumn()
                    ->toJson();
            }

            return view('dashboard.payments.payment_due_details_view', [
                'month'      => $month,
                'class_id'   => $class_id,
                'student_id' => $student_id,
            ]);
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function PaymentDueTransfer(Request $request)
    {
        $studentId = (int) $request->student_id;
        $classId   = (int) $request->class_id;
        $monthY    = Carbon::parse($request->month)->format('Y-m');

        $student   = Student::findOrFail($studentId);
        $sessionId = (int) $student->session_id;

        // ---------- 1) Totals ----------
        $studentFees = FeeAssignDetail::where('month', $monthY)
            ->where('class_id', $classId)
            ->where('session_id', $sessionId)
            ->whereNull('deleted_at')
            ->where('type', '!=', 5)
            ->sum('amount');

        $specialAmount = DB::table('fee_assign_details as fd')
            ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id')
            ->whereNull('fd.deleted_at')
            ->whereNull('fds.deleted_at')
            ->where('fd.type', 5)
            ->where('fds.student_id', $studentId)
            ->where('fd.session_id', $sessionId)
            ->where('fd.class_id', $classId)
            ->where('fd.month', $monthY)
            ->sum('fds.amount');

        $totalAmount = (float) ($studentFees + $specialAmount);

        // ---------- 2) Build type=>[{fee_id, amount}] for non-5 ----------
        $typeFees = FeeAssignDetail::where('month', $monthY)
            ->where('class_id', $classId)
            ->where('session_id', $sessionId)
            ->whereNull('deleted_at')
            ->where('type', '!=', 5)
            ->select('type', 'fee_id', DB::raw('SUM(amount) as amount'))
            ->groupBy('type', 'fee_id')
            ->orderBy('type')->orderBy('fee_id')
            ->get()
            ->groupBy('type')
            ->map(fn($rows) => $rows->map(fn($r) => [
                'fee_id' => (int) $r->fee_id,
                'amount' => (float) $r->amount,
            ])->values()->all())
            ->toArray();

        // ---------- 3) Add type 5 (special/student-wise) ----------
        $specialRows = DB::table('fee_assign_details as fd')
            ->join('feeassign_details_students as fds', 'fds.fee_assign_details_id', '=', 'fd.id')
            ->whereNull('fd.deleted_at')
            ->whereNull('fds.deleted_at')
            ->where('fd.type', 5)
            ->where('fds.student_id', $studentId)
            ->where('fd.session_id', $sessionId)
            ->where('fd.class_id', $classId)
            ->where('fd.month', $monthY)
            ->select('fd.fee_id', DB::raw('SUM(fds.amount) as amount'))
            ->groupBy('fd.fee_id')
            ->orderBy('fd.fee_id')
            ->get();

        if ($specialRows->isNotEmpty()) {
            $typeFees[5] = $specialRows->map(fn($r) => [
                'fee_id' => (int) $r->fee_id,
                'amount' => (float) $r->amount,
            ])->values()->all();
        }

        // ---------- 4) What’s available to allocate (transfer bucket) ----------
        $receivedData = FundDetail::where('student_id', $studentId)
            ->where('month', $monthY)
            ->whereIn('type', [6, 16])
            ->where('payment_type', 1)
            ->whereNull('deleted_at')
            ->whereNull('fund_detail_type_id')
            ->whereNull('fee_id')
            ->whereNull('receipt_type')
            ->select(
                DB::raw('SUM(amount) as total_received'),
                DB::raw('SUM(discount) as total_discount')
            )
            ->first();

        $totalReceive  = (float) ($receivedData->total_received ?? 0); // raw receive bucket
        $totalDiscount = (float) ($receivedData->total_discount ?? 0); // raw discount bucket
        $totalReceived = $totalReceive + $totalDiscount;               // net available against dues
        $overallDue    = max(0.0, $totalAmount - $totalReceived);

        // ---------- 5) Allocation (receive first, then discount) ----------
        // Order types ascending for deterministic allocation
        ksort($typeFees);

        $receiveLeft  = $totalReceive;
        $discountLeft = $totalDiscount;

        $allocation = []; // per-type array of rows [{type, fee_id, amount, receive, discount, due}]
        $typeResult = []; // per-type totals
        $typeTotals = []; // per-type total amount (for reference)

        foreach ($typeFees as $type => $rows) {
            $rowsAlloc = [];
            // 1) First pass: allocate receive
            foreach ($rows as $row) {
                $need        = (float) $row['amount'];
                $takeReceive = min($receiveLeft, $need);

                $rowsAlloc[] = [
                    'type'     => (int) $type,
                    'fee_id'   => (int) $row['fee_id'],
                    'amount'   => $need,
                    'receive'  => $takeReceive,
                    'discount' => 0.0,                  // to be filled in second pass
                    'due'      => $need - $takeReceive, // temp due before discount allocation
                ];

                $receiveLeft -= $takeReceive;
            }

            // 2) Second pass: allocate discount only against remaining dues
            foreach ($rowsAlloc as &$r) {
                if ($discountLeft <= 0) {
                    break;
                }

                $canDiscount = max(0.0, $r['amount'] - $r['receive'] - $r['discount']);
                if ($canDiscount <= 0) {
                    continue;
                }

                $takeDiscount = min($discountLeft, $canDiscount);
                $r['discount'] += $takeDiscount;
                $r['due'] = $r['amount'] - $r['receive'] - $r['discount'];

                $discountLeft -= $takeDiscount;
            }
            unset($r);

            // 3) Totals per type
            $typeAmount   = 0.0;
            $typeReceive  = 0.0;
            $typeDiscount = 0.0;
            $typeDue      = 0.0;

            foreach ($rowsAlloc as $r) {
                $typeAmount += $r['amount'];
                $typeReceive += $r['receive'];
                $typeDiscount += $r['discount'];
                $typeDue += $r['due'];
            }

            $allocation[$type] = $rowsAlloc;
            $typeTotals[$type] = $typeAmount;
            $typeResult[$type] = [
                'total_amount'   => $typeAmount,
                'total_receive'  => $typeReceive,
                'total_discount' => $typeDiscount,
                'total_due'      => $typeDue,
                'status'         => $typeDue == 0.0 ? 'paid' : (($typeReceive + $typeDiscount) > 0.0 ? 'partial' : 'unpaid'),
            ];
        }

        // dd("all location",$allocation, "type result",$typeResult, "type totals", $typeTotals, "total amount", $totalAmount, "total receive", $totalReceive, "total discount", $totalDiscount, "overall due", $overallDue);

        // ---------- 6) Persist everything atomically ----------
        DB::transaction(function () use (
            $student, $studentId, $classId, $sessionId, $monthY,
            $totalAmount, $totalReceive, $totalDiscount, $overallDue,
            $typeResult, $allocation, $request
        ) {
            // Next receipt no
            $last      = FundDetail::whereNotNull('receipt_no')->orderByDesc('receipt_no')->first();
            $receiptNo = $last ? $last->receipt_no + 1 : 1;

            // Remove placeholders (transfer bucket)
            $lastDate = FundDetail::where('student_id', $studentId)
                ->where('month', $monthY)
                ->whereIn('type', [6, 16])
                ->where('payment_type', 1)
                ->whereNull('deleted_at')
                ->whereNull('receipt_type')
                ->whereNull('fund_detail_type_id')
                ->whereNull('fee_id')
                ->max('date');

            // dd($request->all(), $lastDate);

            // MoneyReceipt header (store raw receive/discount; current_receive is net)
            $mr                  = new MoneyReceipt();
            $mr->student_id      = $studentId;
            $mr->receipt_no      = $receiptNo;
            $mr->register_no     = $student->register_no;
            $mr->date            = $lastDate ?: now()->format('Y-m-d');
            $mr->name            = $student->name;
            $mr->class_id        = $student->class_id;
            $mr->month           = $monthY;
            $mr->amount          = $totalReceive;  // raw receive bucket
            $mr->current_receive = $totalReceive;  // net
            $mr->discount        = $totalDiscount; // raw discount bucket
            $mr->due             = $overallDue;
            $mr->total           = $totalAmount;
            $mr->type            = 1;
            $mr->receipt_type    = 'new';
            $mr->save();

            $labels  = [1 => 'academic', 2 => 'hostel', 3 => 'other', 4 => 'onabashik'];
            $labelOf = fn(int $t) => $labels[$t] ?? 'special';

            foreach ($typeResult as $type => $res) {
                if ($res['status'] === 'unpaid') {
                    // nothing was applied to this type
                    continue;
                }

                // Per-type summary row
                $fdt = FundDetailType::create([
                    'student_id'         => $studentId,
                    'class_id'           => $classId,
                    'session_id'         => $sessionId,
                    'receipt_no'         => $receiptNo,
                    'purpose'            => 'Student Monthly Payment',
                    'student_payment_id' => $mr->id,
                    'fee_type'           => $labelOf((int) $type),
                    'month'              => $monthY,
                    'total'              => $res['total_amount'],
                    'prev_receive'       => 0,
                    'prev_discount'      => 0,
                    'receive'            => $res['total_receive'],
                    'discount'           => $res['total_discount'], // <-- include per-type discount
                    'due'                => $res['total_due'],
                    'date'               => $lastDate ?: now()->format('Y-m-d'),
                    'type'               => 6, // transfer type
                    'payment_type'       => 1,
                    'fund_type'          => 1,
                    'receipt_type'       => 'new',
                ]);

                // Per-fee child rows
                foreach (($allocation[$type] ?? []) as $row) {
                    // Save row if anything applied
                    if (($row['receive'] ?? 0) <= 0) {
                        continue;
                    }

                    FundDetail::create([
                        'fund_id'             => $request->fund_id ?? 3,
                        'student_id'          => $studentId,
                        'receipt_no'          => $receiptNo,
                        'student_payment_id'  => $mr->id,
                        'purpose'             => 'Student Monthly Payment',
                        'amount'              => $row['receive'] ?? 0.0, // allocated receive
                        'discount'            => $row['discount'] ?? 0.0,
                        'total'               => $row['amount'],
                        'due'                 => $row['due'] ?? 0.0, // allocated discount
                        'month'               => $monthY,
                        'type'                => 6,
                        'payment_type'        => 1,
                        'fund_type'           => 1,
                        'date'                => $lastDate ?: now()->format('Y-m-d'),
                        'fund_detail_type_id' => $fdt->id,
                        'fee_id'              => $row['fee_id'],
                        'receipt_type'        => 'new',
                    ]);
                }

                MoneyReceipt::where('student_id', $studentId)
                    ->where('month', $monthY)
                    ->whereNull('deleted_at')
                    ->whereNull('receipt_type')
                    ->whereNull('current_receive')
                    ->whereNull('total')
                    ->whereNotIn('type', [14])
                    ->delete();

                FundDetail::where('student_id', $studentId)
                    ->where('month', $monthY)
                    ->whereIn('type', [6, 16])
                    ->where('payment_type', 1)
                    ->whereNull('deleted_at')
                    ->whereNull('receipt_type')
                    ->whereNull('fund_detail_type_id')
                    ->whereNull('fee_id')

                    ->delete();
            }
        });

        return redirect()->back()->with('success', 'Payment Transfer successfully.');
    }

    public function SendSmsSingle(Request $request)
    {
        $student = Student::findOrFail($request->student_id);

        $site = DB::table('settings')->first();

        // dd($site, $status);
        $formattedMonth = Carbon::parse($request->month)->format('F Y');
        // dd($request->all(), $formattedMonth);

        if ($student->gurdian_mobile) {
            $text = $site->name . ', ' . $student->name . ', আপনার সন্তান ' . $formattedMonth .
            ' মাসিক বেতন Due ' . $request->due . ' Tk';
            $this->SendSms($student->gurdian_mobile, $text);
        }

        return response()->json(['message' => 'SMS Sent successfully']);
    }

    public function SendSmsAll(Request $request)
    {
        $studentData    = $request->studentData ?? [];
        $formattedMonth = Carbon::parse($request->month)->format('F Y');
        if (empty($studentData)) {
            return response()->json(['message' => 'Please select at least one student.']);
        } else {
            foreach ($studentData as $key => $student) {

                $data = Student::findOrFail($student['id']);
                $site = DB::table('settings')->first();
                if ($data->gurdian_mobile) {
                    $text = $site->name . ', ' . $data->name . ', আপনার সন্তান ' . $formattedMonth .
                        ' মাসিক বেতন Due ' . $student['due'] . ' Tk';
                    $this->SendSms($data->gurdian_mobile, $text);

                }

            }

        }
        return response()->json(['message' => 'All SMS sent successfully']);
    }

}
