<?php
namespace App\Http\Controllers;

use App\Models\AdmitCard;
use App\Models\ExamRouting;
use App\Models\ExamSeat;
use App\Models\ExamWiseGrades;
use App\Models\InvoiceDesign;
use App\Models\Result;
use App\Models\ResultDetail;
use App\Models\Setting;
use App\Models\SmsStatus;
use App\Models\Student;
use App\Models\StudentClass;
use App\Models\StudentExam;
use App\Models\StudentSession;
use App\Models\Subject;
use App\Models\SubjectAssign;
use App\Models\SubjectGrade;
use App\Models\SubjectWiseGrades;
use App\Traits\DateFormatter;
use App\Traits\DeleteTrait;
//grade mark update by ashiq 21.05.25
use App\Traits\SmsTrait;
//grade mark update by ashiq end 21.05.25
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Yajra\DataTables\Facades\DataTables;

class ResultController extends Controller
{
    use SmsTrait;
    use DeleteTrait;
    use DateFormatter;
    public function ExamSeat(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = ExamSeat::with('stusession', 'exam', 'stuclass')->groupBy('exam_id', 'session_id', 'class_id')->get();

                return Datatables::of($data)

                    ->addColumn('stuclass', function ($data) {
                        return $data->stuclass->name ? $data->stuclass->name : '';
                    })

                    ->addColumn('stusession', function ($data) {
                        return $data->stusession->name ? $data->stusession->name : '';
                    })

                    ->addColumn('exam', function ($data) {
                        return $data->exam->name ? $data->exam->name : '';
                    })

                    ->addColumn('action', function ($data) {
    $exam_id    = $data->exam_id;
    $session_id = $data->session_id;
    $class_id   = $data->class_id;

    $details = '<a href="' . route('exam.seat.show', [$exam_id, $session_id, $class_id]) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i></a> ';
    $edit    = '<a href="' . route('exam.seat.edit', [$exam_id, $session_id, $class_id]) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i></a> ';

    $delete = '';
    if ($this->DeleteData()) {
        $delete = '<a href="#" data-remote="' . route('exam.seat.destroy', [$exam_id, $session_id, $class_id]) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i></a>';
    }

    // Inline all actions
    return $details . $edit . $delete;
})


                    ->addIndexColumn()
                    ->rawColumns(['stusession', 'action', 'stuclass'])
                    ->toJson();
            }
            return view('dashboard.exam_seat.index');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function ExamSeatCreate(Request $request)
    {
        try {
            $students    = Student::select('id', 'name', 'roll_no', 'register_no')->get();
            $classes     = StudentClass::all();
            $exams       = StudentExam::all();
            $stusessions = StudentSession::all();
            return view('dashboard.exam_seat.create', compact('students', 'classes', 'exams', 'stusessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function ExamSeatStore(Request $request)
    {
        $messages = [
            'class_id.required'   => 'Select class',
            'session_id.required' => 'Select session',
            'exam_id.required'    => 'Select exam',
            'student_id.required' => 'Select at least one student',
        ];

        $request->validate([
            'class_id'   => 'required',
            'session_id' => 'required',
            'exam_id'    => 'required',
            'student_id' => 'required|array|min:1',
            'seat_no'    => 'required|array|min:1',
        ], $messages);

        try {
            $exists = ExamSeat::where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->where('exam_id', $request->exam_id)
                ->exists();

            if ($exists) {
                return redirect()->back()->with('error', 'Seat plan already exists for this class, session, and exam.');
            }

            // dd($request->all());

            foreach ($request->seat_no as $studentId => $seatNo) {
                if ($seatNo == null) {
                    continue;
                }
                $data             = new ExamSeat();
                $data->student_id = $studentId;
                $data->class_id   = $request->class_id;
                $data->session_id = $request->session_id;
                $data->exam_id    = $request->exam_id;
                $data->seat_no    = $seatNo;
                $data->status     = 1;
                $data->date       = now()->format('Y-m-d');
                $data->created_by = Auth::user()->id;
                $data->save();

                $admitCard               = new AdmitCard();
                $admitCard->date         = now()->format('Y-m-d');
                $admitCard->exam_seat_id = $data->id;
                $admitCard->student_id   = $studentId;
                $admitCard->class_id     = $request->class_id;
                $admitCard->session_id   = $request->session_id;
                $admitCard->exam_id      = $request->exam_id;
                $admitCard->seat_no      = $seatNo;
                $admitCard->status       = 1;
                $admitCard->created_by   = Auth::user()->id;
                $admitCard->save();
            }

            return redirect()->route('exam.seat')->with('success', 'Seat created successfully');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function ExamSeatShow($exam_id, $session_id, $class_id)
    {
        try {
            $data = ExamSeat::with('exam', 'stusession', 'stuclass')
                ->where('exam_id', $exam_id)
                ->where('session_id', $session_id)
                ->where('class_id', $class_id)
                ->firstOrFail();

            $students = ExamSeat::with('students:id,name,roll_no,register_no')
                ->where('exam_id', $exam_id)
                ->where('session_id', $session_id)
                ->where('class_id', $class_id)
                ->get();

            return view('dashboard.exam_seat.show', compact('data', 'students'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function ExamSeatEdit($exam_id, $session_id, $class_id)
    {
        try {
            $data = ExamSeat::with('exam', 'stusession', 'stuclass')
                ->where('exam_id', $exam_id)
                ->where('session_id', $session_id)
                ->where('class_id', $class_id)
                ->firstOrFail();

            $students = ExamSeat::with('students:id,name,roll_no,register_no,status')
                ->where('exam_id', $exam_id)
                ->where('session_id', $session_id)
                ->where('class_id', $class_id)
            // ->whereNotNull('seat_no')
                ->whereHas('students', function ($q) {
                    $q->where('status', 1)->orWhereNull('status');
                })
                ->get();

            $classes     = StudentClass::all();
            $exams       = StudentExam::all();
            $stusessions = StudentSession::all();
            return view('dashboard.exam_seat.edit', compact('data', 'students', 'classes', 'exams', 'stusessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function ExamSeatUpdate(Request $request, $exam_id, $session_id, $class_id)
    {
        $messages = [
            'class_id.required'   => 'Select class',
            'session_id.required' => 'Select session',
            'exam_id.required'    => 'Select exam',
            'student_id.required' => 'Select at least one student',
        ];

        $request->validate([
            'class_id'   => 'required',
            'session_id' => 'required',
            'exam_id'    => 'required',
            'student_id' => 'required|array|min:1',
            'seat_no'    => 'required|array|min:1',
        ], $messages);

        try {
            ExamSeat::where('session_id', $session_id)
                ->where('exam_id', $exam_id)
                ->where('class_id', $class_id)
                ->delete();

            AdmitCard::where('session_id', $session_id)
                ->where('exam_id', $exam_id)
                ->where('class_id', $class_id)
                ->where('exam_seat_id', '!=', null)
                ->delete();
            // ✅ Check if seat plan already exists BEFORE deleting
            $exists = ExamSeat::where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->where('exam_id', $request->exam_id)
                ->exists();

            if ($exists) {
                return redirect()->back()->with('error', 'Seat plan already exists for this class, session, and exam.');
            }

            // ✅ Then delete old data (if any)

            // ✅ Now insert ExamSeat and use inserted ID for AdmitCard
            foreach ($request->student_id as $key => $studentId) {
                $seatNo = $request->seat_no[$key] ?? null;

                if (empty($seatNo)) {
                    continue;
                }

                // Insert ExamSeat
                $examSeat = ExamSeat::create([
                    'student_id' => $studentId,
                    'class_id'   => $request->class_id,
                    'session_id' => $request->session_id,
                    'exam_id'    => $request->exam_id,
                    'seat_no'    => $seatNo,
                    'status'     => 1,
                    'date'       => now()->format('Y-m-d'),
                    'updated_by' => Auth::user()->id,
                ]);

                // Insert AdmitCard with linked exam_seat_id
                AdmitCard::create([
                    'exam_seat_id' => $examSeat->id,
                    'student_id'   => $studentId,
                    'class_id'     => $request->class_id,
                    'session_id'   => $request->session_id,
                    'exam_id'      => $request->exam_id,
                    'seat_no'      => $seatNo,
                    'status'       => 1,
                    'date'         => now()->format('Y-m-d'),
                    'updated_by'   => Auth::user()->id,
                ]);
            }

            return redirect()->route('exam.seat')->with('success', 'Seat updated successfully');
        } catch (\Exception $exception) {
            dd($exception);
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function ExamSeatDestroy($exam_id, $session_id, $class_id)
    {
        try {
            ExamSeat::where('exam_id', $exam_id)
                ->where('session_id', $session_id)
                ->where('class_id', $class_id)
                ->delete();

            AdmitCard::where('exam_id', $exam_id)
                ->where('session_id', $session_id)
                ->where('class_id', $class_id)
                ->delete();

            return response()->json([
                'success' => true,
                'message' => 'Exam seat deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Exam seat delete failed: ' . $exception->getMessage(),
            ]);
        }
    }

    public function AdmitCard(Request $request)
    {
        try {

            if ($request->ajax()) {

                $query = AdmitCard::with('students:id,name,roll_no,register_no,class_id,session_id,status', 'users:id,name', 'exams', 'stuclass');

                if ($request->class_id) {
                    $query->whereHas('students', function ($q) use ($request) {
                        $q->where('class_id', $request->class_id);
                    });
                }

                if ($request->session_id) {
                    $query->whereHas('students', function ($q) use ($request) {
                        $q->where('session_id', $request->session_id);
                    });
                }

                if ($request->exam_id) {
                    $query->where('exam_id', $request->exam_id);
                }

                $data = $query->whereNull('deleted_at')->get();
                // dd($data, $request->all());

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        return Carbon::parse($data->date)->format('d-M-Y') ?? '';
                    })

                    ->addColumn('student', function ($data) {
                        $name = isset($data->students) ? $data->students->name : '--';
                        $reg  = isset($data->students) ? $data->students->register_no : '--';
                        $roll = isset($data->students) ? $data->students->roll_no : '--';
                        return $name . '<br>' . $reg . '<br>' . $roll;
                    })

                    ->addColumn('exam', function ($data) {
                        $name = isset($data->exams) ? $data->exams->name : '--';
                        return $name;
                    })

                    ->addColumn('className', function ($data) {
                        $name = isset($data->stuclass) ? $data->stuclass->name : '--';
                        return $name;
                    })

                    ->addColumn('givenBy', function ($data) {
                        return $data->users->name ?? '';
                    })

                    ->addColumn('action', function ($data) {
                        if (isset($data->students) && ($data->students->status == 1 || $data->students->status == null)) {
                            $delete = '';
                            if ($this->DeleteData()) {

                                $delete = '<button id="messageShow" class="btn btn-sm btn-danger btn-delete" data-remote=" ' . route('admitcard.destroy', $data->id) . ' " title="Delete"><i class="fa fa-trash-alt"></i></button>';
                            }

                            $print = '<a title="a5 page" class="btn btn-sm btn-primary ml-1" href="' . route('admitcard.print', $data->id) . ' " ><i class="fas fa-file"></i></a>';
                            return $delete . ' ' . $print;
                        }
                    })

                    ->addIndexColumn()
                    ->rawColumns(['date', 'exam', 'givenBy', 'student', 'action', 'className'])
                    ->toJson();
            }
            $exams    = StudentExam::all();
            $classes  = StudentClass::all();
            $sessions = StudentSession::all();
            // dd($sessions);
            return view('dashboard.admitcard.index', compact('exams', 'classes', 'sessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function AdmitCardCreate()
    {
        try {
            $exam = ExamSeat::with('exam')->latest()->first();

            if (! $exam) {
                return redirect()->back()->with('error', 'Add Student Seat No For Exam');
            }

            $existingStudentIds = AdmitCard::where('exam_id', $exam->exam_id)
                ->where('class_id', $exam->class_id)
                ->where('session_id', $exam->session_id)
                ->where('status', 1)
                ->pluck('student_id')
                ->toArray();

            $students = ExamSeat::with('students:id,name,roll_no,register_no', 'stuclass')
                ->where('exam_id', $exam->exam_id)
                ->where('class_id', $exam->class_id)
                ->where('session_id', $exam->session_id)
                ->where('status', 2)
                ->whereNotNull('seat_no')
                ->whereNotIn('student_id', $existingStudentIds)
                ->get();

            if ($students->count() == 0) {
                return redirect()->route('admitcard')
                    ->with('success', 'All admit cards have already been created');
            }

            return view('dashboard.admitcard.create', compact('students', 'exam'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function AdmitCardStore(Request $request)
    {
        $messages = [
            'student_id.required' => 'Select student',
        ];

        $request->validate([
            'student_id' => 'required',
        ], $messages);

        $existing = AdmitCard::where('student_id', $request->student_id)
            ->where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->where('exam_id', $request->exam_id)
            ->exists();

        if ($existing) {
            return redirect()->back()->with('error', 'Admit Card already printed for this student!');
        }

        DB::beginTransaction();

        try {
            $exam = ExamSeat::where('student_id', $request->student_id)
                ->where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->where('exam_id', $request->exam_id)
                ->first();

            $exam->status     = 1;
            $exam->updated_by = Auth::user()->id;
            $exam->save();

            $data               = new AdmitCard();
            $data->date         = now()->format('Y-m-d');
            $data->exam_seat_id = $exam->id;
            $data->student_id   = $request->student_id;
            $data->class_id     = $request->class_id;
            $data->session_id   = $request->session_id;
            $data->exam_id      = $request->exam_id;
            $data->seat_no      = $exam->seat_no;
            $data->status       = 1;
            $data->created_by   = Auth::user()->id;
            $data->save();

            $student = Student::findOrFail($data->student_id);
            $site    = DB::table('settings')->first();
            $to      = $student->gurdian_mobile;
            $text    = $student->name . ' আপনার প্রবেশপত্র সম্পূর্ণ হয়েছে ';
            $status  = optional(SmsStatus::first())->student_admitcard;

            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            DB::commit();

            return redirect()->back()->with('success', 'Admit card created successfully.');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function AdmitCardPrint($id)
    {
        // AdmitCard data load
        $data = AdmitCard::with([
            'students:id,name,roll_no,register_no,class_id,session_id,father_name,district,police_station,image,student_type,register_no',
            'stuclass',
            'stusession',
            'exams',
        ])->findOrFail($id);

        $invoice = InvoiceDesign::first();
        $setting = Setting::first();

        // Check if matching exam_routing exists
        $examRouting = ExamRouting::where('class_id', $data->class_id)
            ->where('session_id', $data->session_id)
            ->where('exam_id', $data->exam_id)
            ->first();

        if ($examRouting) {
            // If match found, return exam_routing specific print view
            return view('dashboard.admitcard.exam_routing_print', compact('data', 'invoice', 'setting', 'examRouting'));
            // return view('dashboard.admitcard.exam_routing_print', compact('data', 'invoiceDesign', 'setting', 'examRouting'));

        } else {
            // Else return current print view
            return view('dashboard.admitcard.print', compact('data', 'invoice', 'setting'));
        }
    }
    public function AdmitCardPrintAll(Request $request)
    {
        // dd($request->all()); 

        $exam_id    = $request->input('exam_id');   
        $class_id   = $request->input('class_id');   
        $session_id = $request->input('session_id'); 

       
        $admitCardsQuery = AdmitCard::with([
            'students:id,name,roll_no,register_no,class_id,session_id,father_name,district,police_station,image,student_type',
            'stuclass',                                                                                                        //
            'stusession',                                                                                                      // 
            'exams',                                                                                                           //
        ]);

      
        if (! empty($exam_id)) {
            $admitCardsQuery->where('exam_id', $exam_id);
        }

       
        if (! empty($class_id)) {
            $admitCardsQuery->where('class_id', $class_id);
        }

        
        if (! empty($session_id)) {
            $admitCardsQuery->where('session_id', $session_id);
        }

      
        $data = $admitCardsQuery->orderBy('id', 'asc')->get();

        
        $invoice = InvoiceDesign::first();
        $setting = Setting::first();

       
        return view('dashboard.admitcard.print_all', compact('data', 'invoice', 'setting'));
    }

    public function AdmitCardDestroy($id)
    {
        try {
            $data    = AdmitCard::findOrFail($id);
            $student = ExamSeat::where('student_id', $data->student_id)
                ->where('class_id', $data->class_id)
                ->where('session_id', $data->session_id)
                ->where('exam_id', $data->exam_id)
                ->first();

            if ($student) {
                $student->status     = 2;
                $student->updated_by = Auth::user()->id;
                $student->save();
            }

            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Admitcard deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Admitcard delete failed',
            ]);
        }
    }

    public function ResultSheet(Request $request)
    {
        try {
            if ($request->ajax()) {
                $query    = Result::with('stuclass:id,name', 'students:id,name,roll_no,register_no', 'stusession', 'exam');
                   if ($request->class_id) {
                    $query->whereHas('students', function ($q) use ($request) {
                        $q->where('class_id', $request->class_id);
                    });
                }

                if ($request->session_id) {
                    $query->whereHas('students', function ($q) use ($request) {
                        $q->where('session_id', $request->session_id);
                    });
                }

                if ($request->exam_id) {
                    $query->where('exam_id', $request->exam_id);
                }

                $data = $query->orderBy('id', 'desc')->whereNull('deleted_at')->get();


                $results = ResultDetail::with('subjects:id,name')->whereIn('result_id', $data->pluck('id'))->get();
                $data->each(function ($result) use ($results) {
                    $resultDetails = $results->where('result_id', $result->id);
                    $totalMarks    = $resultDetails->sum(function ($detail) {
                        return (float) ($detail->mark ?? 0);
                    });

                    $result->mark = $totalMarks;
                });

                return Datatables::of($data)
                    ->addColumn('student', function ($data) {
                        $name = isset($data->students) ? $data->students->name : '--';
                        // $roll = isset($data->students) ? $data->students->roll_no : '--';
                        $reg = isset($data->students) ? $data->students->register_no : '--';
                        return $name . '<br>' . $reg;
                    })

                    ->addColumn('date', function ($data) {
                        $from = Carbon::parse($data->created_at)->format('d-M-Y') ?? '';
                            return '<input type="hidden" class="result_ids" name="result_id[]" value="' . $data->id . '">' . $from;
                        // return $from;
                    })
                    ->addColumn('exam', function ($data) {
                        $name = isset($data->exam) ? $data->exam->name : '--';
                        return $name;
                    })
                    ->addColumn('className', function ($data) {
                        $name = isset($data->stuclass) ? $data->stuclass->name : '--';
                        return $name;
                    })
                    ->addColumn('result', function ($data) {
                        return isset($data->mark) ? round($data->mark, 2) : '--';
                    })
                    ->addColumn('action', function ($data) {
                        $btn = '<a id="show" href="' . route('result.show', $data->id) . '" class="btn btn-sm btn-primary show" title="show"><i class="fa fa-eye"></i></a>';
                        return $btn;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['date','student', 'exam', 'result', 'action', 'className'])
                    ->toJson();
            }

        $exams = StudentExam::all();
        $classes = StudentClass::all();
        $sessions = StudentSession::all();
            return view('dashboard.results.result_sheet', compact('exams', 'classes', 'sessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function Results(Request $request)
    {
        try {

            if ($request->ajax()) {

                $query = Result::with('stuclass:id,name', 'students:id,name,roll_no,register_no,status', 'stusession', 'exam');
                 if ($request->class_id) {
                    $query->whereHas('students', function ($q) use ($request) {
                        $q->where('class_id', $request->class_id);
                    });
                }

                if ($request->session_id) {
                    $query->whereHas('students', function ($q) use ($request) {
                        $q->where('session_id', $request->session_id);
                    });
                }

                if ($request->exam_id) {
                    $query->where('exam_id', $request->exam_id);
                }

                $data = $query->orderBy('id', 'desc')->whereNull('deleted_at')->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        $from = Carbon::parse($data->created_at)->format('d-M-Y') ?? '';
                            return '<input type="hidden" class="result_ids" name="result_id[]" value="' . $data->id . '">' . $from;
                        // return $from;
                    })

                    ->addColumn('student', function ($data) {
                        $name = isset($data->students) ? $data->students->name : '--';
                        $roll = isset($data->students) ? $data->students->roll_no : '--';
                        $reg  = isset($data->students) ? $data->students->register_no : '--';
                        return $name . '<br>' . $roll . '<br>' . $reg;
                    })

                    ->addColumn('exam', function ($data) {
                        $name = isset($data->exam) ? $data->exam->name : '--';
                        return $name;
                    })

                    ->addColumn('session', function ($data) {
                        $name = isset($data->stusession) ? $data->stusession->name : '--';
                        return $name;
                    })

                    ->addColumn('class', function ($data) {
                        $name = isset($data->stuclass) ? $data->stuclass->name : '--';
                        return $name;
                    })
                    ->addColumn('subject_status', function ($data) {
                        $totalSubjects = \App\Models\SubjectAssign::where('class_id', $data->class_id)->count();
                        $givenSubjects = $data->resultDetails->count();

                        $subjectStatus = "Total Subjects: $totalSubjects, Given: $givenSubjects";
                        return $subjectStatus;
                    })

                   ->addColumn('action', function ($data) {
    if (isset($data->students) && ($data->students->status == 1 || $data->students->status == null)) {
        $details = '<a href="' . route('result.show', $data->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i></a> ';
        $edit    = '<a href="' . route('result.edit', $data->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i></a> ';
        $print   = '<a href="' . route('result.show', $data->id) . '" class="btn btn-sm btn-primary" title="A5 Print"><i class="fa fa-file"></i></a> ';
        $delete  = '<a href="#" data-remote="' . route('result.destroy', $data->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i></a>';

        return $details . $edit . $print . $delete;
    }
})


                    ->addIndexColumn()
                    ->rawColumns(['date', 'class', 'student', 'exam', 'session', 'subject_status', 'action'])
                    ->toJson();
            }

            $exams = StudentExam::all();
            $classes = StudentClass::all();
            $sessions = StudentSession::all();
            return view('dashboard.results.index', compact('exams', 'classes', 'sessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function ResultCreate()
    {
        $exams    = StudentExam::all();
        $class    = StudentClass::all();
        $sessions = StudentSession::all();
        return view('dashboard.results.create', compact('exams', 'class', 'sessions'));
    }

    public function ResultCreateBySubject()
    {
        $exams    = StudentExam::all();
        $class    = StudentClass::all();
        $sessions = StudentSession::all();
        return view('dashboard.results.create_by_subject', compact('exams', 'class', 'sessions'));
    }

    public function studentBySubject(Request $request)
    {
        $examId     = $request->exam_id;
        $session_id = $request->session_id;
        $class_id   = $request->class_id;
        $subject_id = $request->subject_id; // ✅ required input

        // Students from AdmitCard
        $students = AdmitCard::with(['students:id,name,register_no,roll_no'])
            ->where('class_id', $class_id)
            ->where('session_id', $session_id)
            ->where('exam_id', $examId)
            ->get();

        $data = [];

        foreach ($students as $admit) {
            $student = $admit->students;
            if (! $student) {
                continue;
            }

            // Step 1: find Result for this student
            $result = Result::where('student_id', $student->id)
                ->where('class_id', $class_id)
                ->where('session_id', $session_id)
                ->where('exam_id', $examId)
                ->first();

            // Step 2: if result exists, check if subject mark exists
            $mark = null;

             $subjectWiseGrade = SubjectWiseGrades::where('class_id', $class_id)
                ->where('exam_id', $examId)
                ->where('session_id', $session_id)
                ->where('subject_id', $subject_id)
                ->whereNull('deleted_at')
                ->first();
                
            $examWiseGrade = ExamWiseGrades::where('exam_id', $examId)
            ->where('session_id', $session_id)
            ->whereNull('deleted_at')->first();

            if ($result) {
                $mark = ResultDetail::where('result_id', $result->id)
                    ->where('subject_id', $subject_id)
                    ->value('mark');
            }

            // Step 3: push student with mark
            $data[] = [
                'id'          => $student->id,
                'name'        => $student->name,
                'register_no' => $student->register_no,
                'roll_no'     => $student->roll_no,
                'exam_mark' => $subjectWiseGrade->exam_mark ?? $examWiseGrade->exam_mark ?? 100, // may be null or existing value
                'mark'        => $mark, // may be null or existing value
            ];
        }

        return response()->json($data);
    }

    public function ResultStore(Request $request)
    {
        $existing = Result::where('student_id', $request->student_id)
            ->where('session_id', $request->session_id)
            ->where('exam_id', $request->exam_id)
            ->where('class_id', $request->class_id)
            ->exists();

        if ($existing) {
            return redirect()->back()->with('error', 'Result already exists');
        }

        DB::beginTransaction(); // ✅ শুধু একবার রাখো

        try {
            $data             = new Result();
            $data->date       = now()->format('Y-m-d');
            $data->student_id = $request->student_id;
            $data->class_id   = $request->class_id;
            $data->session_id = $request->session_id;
            $data->exam_id    = $request->exam_id;
            $data->created_by = Auth::user()->id;
            $data->save();

            $totalMarks   = 0;
            $subjectMarks = [];

            foreach ($request->subject_id as $key => $subjectId) {
                $details             = new ResultDetail();
                $details->result_id  = $data->id;
                $details->student_id = $data->student_id;
                $details->mark       = $request->mark[$key];
                $details->subject_id = $subjectId;
                $details->created_by = Auth::user()->id;
                $details->save();

                $subject        = Subject::find($subjectId);
                $subjectMarks[] = $subject->name . ': ' . $details->mark;
                $totalMarks += $details->mark;
            }

            // ⬇️ এই ২টি ফিল্ড অবশ্যই টেবিলে থাকতে হবে
            $data->total_mark = $totalMarks;
            $data->save();

            // Merit position calculate
            $results = Result::where('class_id', $request->class_id)
                ->where('exam_id', $request->exam_id)
                ->where('session_id', $request->session_id)
                ->orderByDesc('total_mark')
                ->get();

            // dd($results);

            $position      = 1;
            $previous_mark = null;
            // $counter = 1;

            foreach ($results as $result) {
                if ($previous_mark !== null && $result->total_mark == $previous_mark) {
                    $result->merit_position = $position;
                    $previous_mark          = $result->total_mark;
                } else if ($previous_mark !== null && $result->total_mark !== $previous_mark) {
                    $result->merit_position = $position + 1;
                    $position               = $position + 1;
                    $previous_mark          = $result->total_mark;
                } else {
                    $result->merit_position = $position;
                    $position               = $position;
                    $previous_mark          = $result->total_mark;
                }

                $result->save();

                // $previous_mark = $result->total_mark;
                // $counter++;
            }

            $student = Student::findOrFail($request->student_id);
            $site    = DB::table('settings')->first();
            $to      = $student->gurdian_mobile;

            $text = $student->name . ' আপনার ' . $data->exam->name . ' এর সর্বমোট নাম্বার ' . $totalMarks;

            $status = optional(SmsStatus::first())->student_result;
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            DB::commit();
            return redirect()->route('results')->with('success', 'Result created successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function ResultStoreBySubject(Request $request)
    {
        // dd($request->all());
        DB::beginTransaction();

        try {
            $class_id   = $request->class_id;
            $session_id = $request->session_id;
            $exam_id    = $request->exam_id;
            $subject_id = $request->subject_id;

            $alreadyAssignedList = [];

            foreach ($request->student_id as $index => $student_id) {
                $result = Result::firstOrCreate([
                    'student_id' => $student_id,
                    'class_id'   => $class_id,
                    'session_id' => $session_id,
                    'exam_id'    => $exam_id,
                ], [
                    'date'       => now()->format('Y-m-d'),
                    'created_by' => Auth::id(),
                ]);
                Log::info($result);

                // 🟢 Move this check inside the loop for each student
                $alreadyAssigned = ResultDetail::where('result_id', $result->id)
                    ->where('subject_id', $subject_id)
                    ->exists();

                if ($alreadyAssigned) {
                    $alreadyAssignedList[] = $student_id;
                    continue;
                }

                $resultDetails = ResultDetail::create([
                    'result_id'  => $result->id,
                    'student_id' => $student_id,
                    'subject_id' => $subject_id,
                    'mark'       => $request->mark[$index],
                    'created_by' => Auth::id(),
                ]);

                Log::warning("Insert failed for student ID: $resultDetails");
            }

            // Recalculate total and merit
            $results = Result::where('class_id', $class_id)
                ->where('session_id', $session_id)
                ->where('exam_id', $exam_id)
                ->get();

            foreach ($results as $result) {
                $total = ResultDetail::where('result_id', $result->id)->sum('mark');
                $result->update(['total_mark' => $total]);
            }

            $sorted   = $results->sortByDesc('total_mark')->values();
            $position = 1;
            $prevMark = null;

            foreach ($sorted as $result) {
                if ($prevMark !== null && $result->total_mark == $prevMark) {
                    $result->merit_position = $position;
                    $prevMark               = $result->total_mark;
                } else if ($prevMark !== null && $result->total_mark !== $prevMark) {
                    $result->merit_position = $position + 1;
                    $position               = $position + 1;
                    $prevMark               = $result->total_mark;
                } else {
                    $result->merit_position = $position;
                    $position               = $position;
                    $prevMark               = $result->total_mark;
                }
                $result->update(['merit_position' => $position]);
                // $prevMark = $result->total_mark;
                // $counter++;
            }

            DB::commit();
            return redirect()->back()->with('success', 'Subject-wise results stored successfully!');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Error: ' . $e->getMessage());
        }
    }

    public function ResultEdit($id)
    {
        try {
            $data      = Result::with('stuclass:id,name', 'exam:id,name', 'stusession')->findOrFail($id);
            $results   = ResultDetail::with('subjects')->where('result_id', $data->id)->get();
            $examMarks = SubjectWiseGrades::where('class_id', $data->class_id)
                ->where('exam_id', $data->exam_id)
                ->where('session_id', $data->session_id)
                ->get()
                ->pluck('exam_mark', 'subject_id'); // key: subject_id, value: exam_mark

            $exams    = StudentExam::all();
            $class    = StudentClass::all();
            $sessions = StudentSession::all();
            $students = AdmitCard::with('students:id,name,register_no,roll_no')->where('exam_id', $data->exam_id)->where('session_id', $data->session_id)->where('class_id', $data->class_id)->get();
            return view('dashboard.results.edit', compact('data', 'results', 'exams', 'students', 'class', 'sessions', 'examMarks'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    //grade mark update by ashiq 21.05.25
   public function ResultShow($id)
{
    try {
        $data = Result::with(
            'stuclass:id,name',
            'exam:id,name',
            'stusession',
            'students:id,name,image,class_id,session_id,register_no,roll_no'
        )->findOrFail($id);

        $results = ResultDetail::with('subjects:id,name')
            ->where('result_id', $data->id)
            ->get();

        // Global grade bands (used for % → Grade/GPA)
        $globalGrades  = SubjectGrade::all();

        $totalMark     = 0.0;
        $totalExamMark = 0;
        $totalSub      = $results->count();

        // Preload exam-wise scale once
        $examWiseGrade = ExamWiseGrades::where('exam_id', $data->exam_id)
            ->where('session_id', $data->session_id)
            ->whereNull('deleted_at')
            ->first();

        foreach ($results as $result) {
            // Prefer subject-wise JSON grading
            $subjectWiseGrade = SubjectWiseGrades::where('subject_id', $result->subject_id)
                ->where('exam_id', $data->exam_id)
                ->where('class_id', $data->class_id)
                ->where('session_id', $data->session_id)
                ->whereNull('deleted_at')
                ->first();

            $grade = null;

            if ($subjectWiseGrade) {
                $start_marks = json_decode($subjectWiseGrade->start_mark);
                $end_marks   = json_decode($subjectWiseGrade->end_mark);
                $grades      = json_decode($subjectWiseGrade->grade);
                $gpas        = json_decode($subjectWiseGrade->gpa);

                foreach ($start_marks as $index => $start) {
                    $start = (float) $start;
                    $end   = (float) ($end_marks[$index] ?? 0);
                    if ($result->mark >= $start && $result->mark <= $end) {
                        $grade = (object) [
                            'name'      => $grades[$index] ?? 'N/A',
                            'gpa'       => $gpas[$index] ?? 0,
                            'exam_mark' => $subjectWiseGrade->exam_mark
                                ?? ($examWiseGrade->exam_mark ?? 100),
                        ];
                        break;
                    }
                }
            } elseif ($examWiseGrade) {
                $start_marks = json_decode($examWiseGrade->start_mark);
                $end_marks   = json_decode($examWiseGrade->end_mark);
                $grades      = json_decode($examWiseGrade->grade);
                $gpas        = json_decode($examWiseGrade->gpa);

                foreach ($start_marks as $index => $start) {
                    $start = (float) $start;
                    $end   = (float) ($end_marks[$index] ?? 0);
                    if ($result->mark >= $start && $result->mark <= $end) {
                        $grade = (object) [
                            'name'      => $grades[$index] ?? 'N/A',
                            'gpa'       => $gpas[$index] ?? 0,
                            'exam_mark' => $examWiseGrade->exam_mark ?? 100,
                        ];
                        break;
                    }
                }
            }

            // Fallback to global band (by obtained mark range) only for per-subject display
            if (! $grade) {
                $grade = $globalGrades->filter(function ($g) use ($result) {
                    [$start, $end] = explode('-', str_replace(' ', '', $g->mark));
                    return $result->mark >= (int) $start && $result->mark <= (int) $end;
                })->sortByDesc(function ($g) {
                    [$start] = explode('-', str_replace(' ', '', $g->mark));
                    return (int) $start;
                })->first();
            }

            // Set per-subject fields for the view
            $result->grade     = $grade->name ?? $grade->grade ?? 'N/A';
            $result->exam_mark = (int) ($grade->exam_mark ?? 100);
            $result->gpa       = number_format((float) ($grade->gpa ?? 0), 2);
            // (Optional) flag to style red in blade when GPA == 0
            $result->gpa_zero  = ((float)$result->gpa) == 0.0;

            // Totals for percentage
            $totalExamMark += (int) $result->exam_mark;
            $totalMark     += (float) $result->mark;
        }

        // === NEW: percentage-based average and global final grade ===
        $avgPercent = $totalExamMark > 0 ? ($totalMark / $totalExamMark) * 100 : 0.0;

        $avgGrade = $globalGrades->first(function ($g) use ($avgPercent) {
            $band = str_replace(' ', '', $g->mark);
            if (strpos($band, '-') !== false) {
                [$start, $end] = explode('-', $band, 2);
                return $avgPercent >= (float) $start && $avgPercent <= (float) $end;
            }
            return $avgPercent >= (float) $band; // single threshold bands like "80"
        });

        // Final GPA taken from the global band mapped by percentage
        $avgGpa = number_format((float) ($avgGrade->gpa ?? 0), 2);

        $invoice = InvoiceDesign::first();

        return view('dashboard.results.show', [
            'data'          => $data,
            'results'       => $results,
            'avgGpa'        => $avgGpa,                  // final GPA from global band
            'avgGrade'      => $avgGrade,                // final grade object (global)
            'avgPercent'    => number_format($avgPercent, 2), // for showing গড় মার্ক (%)
            'totalMark'     => $totalMark,
            'totalExamMark' => $totalExamMark,
            'invoice'       => $invoice,
        ]);
    } catch (\Exception $exception) {
        return redirect()->back()->with('error', $exception->getMessage());
    }
}

    //grade mark update by ashiq end 21.05.25

    public function ResultPrint($id)
    {
        try {
            $data      = Result::with('stuclass:id,name', 'exam:id,name', 'stusession', 'students:id,name,class_id,session_id,register_no,roll_no,birth_date,father_name')->findOrFail($id);
            $results   = ResultDetail::with('subjects:id,name')->where('result_id', $data->id)->get();
            $grades    = SubjectGrade::all();
            $totalGpa  = 0;
            $totalMark = 0;
            $totalSub  = $results->count();

            foreach ($results as $result) {
                $grade = $grades->filter(function ($grade) use ($result) {
                    return (int) $result->mark >= (int) $grade->mark;
                })->sortByDesc('mark')->first();

                $result->grade = $grade ? $grade->name : '';
                $result->gpa   = $grade ? $grade->gpa : '';

                $totalGpa += (float) ($result->gpa ?? 0);
                $totalMark += $result->mark ?? 0;
            }

            $avgGpa = $totalSub > 0 ? $totalGpa / $totalSub : 0;

            $avgGrade = $grades->filter(function ($grade) use ($avgGpa) {
                return (float) $avgGpa >= (float) $grade->gpa;
            })->sortByDesc('gpa')->first();

            $setting = Setting::first();

            return view('dashboard.results.print', compact('data', 'results', 'avgGpa', 'avgGrade', 'totalGpa', 'totalMark', 'setting'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function ResultPrintAll(Request $request)
    {
    try {
        $resultIds = $request->input('result_ids');
        // dd($resultIds);
        if (empty($resultIds)) {
            return redirect()->back()->with('error', 'No result selected.');
        }

        $globalGrades = SubjectGrade::all();
        $setting = Setting::first();
        $invoice = InvoiceDesign::first();

        $allData = [];

        foreach ($resultIds as $id) {
            $data = Result::with(
                'stuclass:id,name',
                'exam:id,name',
                'stusession',
                'students:id,name,image,class_id,session_id,register_no,roll_no'
            )->findOrFail($id);

            $results = ResultDetail::with('subjects:id,name')->where('result_id', $data->id)->get();

            $totalMark = 0;
            $totalExamMark = 0;
            $totalGpa = 0;
            $totalSub = $results->count();

            foreach ($results as $result) {
                $grade = null;

                $subjectWiseGrade = SubjectWiseGrades::where('subject_id', $result->subject_id)
                    ->where('exam_id', $data->exam_id)
                    ->where('class_id', $data->class_id)
                    ->where('session_id', $data->session_id)
                    ->whereNull('deleted_at')
                    ->first();

                $examWiseGrade = ExamWiseGrades::where('exam_id', $data->exam_id)
                    ->where('session_id', $data->session_id)
                    ->whereNull('deleted_at')
                    ->first();

                if ($subjectWiseGrade) {
                    $start_marks = json_decode($subjectWiseGrade->start_mark);
                    $end_marks   = json_decode($subjectWiseGrade->end_mark);
                    $grades      = json_decode($subjectWiseGrade->grade);
                    $gpas        = json_decode($subjectWiseGrade->gpa);

                    foreach ($start_marks as $index => $start) {
                        $start = (float) $start;
                        $end   = (float) ($end_marks[$index] ?? 0);

                        if ($result->mark >= $start && $result->mark <= $end) {
                            $grade = (object) [
                                'name'      => $grades[$index] ?? 'N/A',
                                'gpa'       => $gpas[$index] ?? 0,
                                'exam_mark' => $subjectWiseGrade->exam_mark ?? $examWiseGrade->exam_mark ?? 0,
                            ];
                            break;
                        }
                    }
                } elseif ($examWiseGrade) {
                    $start_marks = json_decode($examWiseGrade->start_mark);
                    $end_marks   = json_decode($examWiseGrade->end_mark);
                    $grades      = json_decode($examWiseGrade->grade);
                    $gpas        = json_decode($examWiseGrade->gpa);

                    foreach ($start_marks as $index => $start) {
                        $start = (float) $start;
                        $end   = (float) ($end_marks[$index] ?? 0);

                        if ($result->mark >= $start && $result->mark <= $end) {
                            $grade = (object) [
                                'name'      => $grades[$index] ?? 'N/A',
                                'gpa'       => $gpas[$index] ?? 0,
                                'exam_mark' => $examWiseGrade->exam_mark ?? 100,
                            ];
                            break;
                        }
                    }
                }

                // fallback to global grade
                if (! $grade) {
                    $grade = $globalGrades->filter(function ($g) use ($result) {
                        [$start, $end] = explode('-', str_replace(' ', '', $g->mark));
                        return $result->mark >= (int) $start && $result->mark <= (int) $end;
                    })->sortByDesc(function ($g) {
                        [$start] = explode('-', str_replace(' ', '', $g->mark));
                        return (int) $start;
                    })->first();
                }

                $result->grade     = $grade->name ?? $grade->grade ?? 'N/A';
                $result->exam_mark = $grade->exam_mark ?? 100;
                $result->gpa       = number_format((float) ($grade->gpa ?? 0), 2);

                $totalExamMark += (int) $result->exam_mark;
                $totalMark += (float) $result->mark;
                $totalGpa += (float) $result->gpa;
            }

            $avgGpa = $totalSub > 0 ? number_format(round($totalGpa / $totalSub, 2), 2) : '0.00';

            $avgGrade = $globalGrades->filter(function ($grade) use ($avgGpa) {
                return $avgGpa >= $grade->gpa;
            })->sortByDesc('gpa')->first();

            $allData[] = [
                'data'           => $data,
                'results'        => $results,
                'totalExamMark'  => $totalExamMark,
                'totalMark'      => $totalMark,
                'totalGpa'       => $totalGpa,
                'avgGpa'         => $avgGpa,
                'avgGrade'       => $avgGrade ?? null,
            ];
        }

        return view('dashboard.results.print_all', compact('allData', 'setting', 'invoice'));
    } catch (\Exception $e) {
        return redirect()->back()->with('error', $e->getMessage());
    }
}

    public function ResultUpdate(Request $request, $id)
    {
        $existing = Result::where('student_id', $request->student_id)
            ->where('session_id', $request->session_id)
            ->where('exam_id', $request->exam_id)
            ->where('class_id', $request->class_id)
            ->where('id', '!=', $id)
            ->exists();

        if ($existing) {
            return redirect()->back()->with('error', 'Result already exists');
        }

        DB::beginTransaction();

        try {
            $data             = Result::findOrFail($id);
            $data->date       = now()->format('Y-m-d');
            $data->student_id = $request->student_id;
            $data->class_id   = $request->class_id;
            $data->session_id = $request->session_id;
            $data->exam_id    = $request->exam_id;
            $data->created_by = Auth::user()->id;
            $data->total_mark = array_sum($request->mark ?? []);
            $data->save();

            ResultDetail::where('result_id', $data->id)->where('student_id', $data->student_id)->delete();

            $totalMarks   = 0;
            $subjectMarks = [];

            foreach ($request->subject_id as $key => $subjectId) {
                $details             = new ResultDetail();
                $details->result_id  = $data->id;
                $details->student_id = $data->student_id;
                $details->mark       = $request->mark[$key];
                $details->subject_id = $subjectId;
                $details->updated_by = Auth::user()->id;
                $details->save();

                $subject        = Subject::find($subjectId);
                $subjectMarks[] = $subject->name . ': ' . $details->mark;
                $totalMarks += $details->mark;
            }

            $student = Student::findOrFail($request->student_id);
            $site    = DB::table('settings')->first();
            $to      = $student->gurdian_mobile;
            $text    = $student->name . ' আপনার ' . $data->exam->name . ' এর সর্বমোট নাম্বার ' . $totalMarks;
            $status  = optional(SmsStatus::first())->student_result;
            if ($status == 1) {
                $this->SendSms($to, $text);
            }
            $results = Result::where('class_id', $request->class_id)
                ->where('exam_id', $request->exam_id)
                ->where('session_id', $request->session_id)
                ->orderByDesc('total_mark')
                ->get();

            // dd($results, $request->all(), $data);

            $position      = 1;
            $previous_mark = null;
            // $previous_position = 1;

            foreach ($results as $index => $result) {
                if ($previous_mark !== null && $result->total_mark == $previous_mark) {
                    $result->merit_position = $position;
                    $previous_mark          = $result->total_mark;
                } else if ($previous_mark !== null && $result->total_mark !== $previous_mark) {
                    $result->merit_position = $position + 1;
                    $position               = $position + 1;
                    $previous_mark          = $result->total_mark;
                } else {
                    $result->merit_position = $position;
                    $position               = $position;
                    $previous_mark          = $result->total_mark;
                }

                $result->save();

                // $position++;
            }

            DB::commit();

            return redirect()->route('results')->with('success', 'Result updated successfully');
        } catch (\Exception $exception) {
            DB::rollBack();
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function ResultDestroy($id)
    {
        try {
            $data = Result::findOrFail($id);
            ResultDetail::where('result_id', $data->id)->where('student_id', $data->student_id)->delete();
            $data->delete();
            return response()->json([
                'success' => true,
                'message' => 'Result deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Result delete failed',
            ]);
        }
    }

    public function ClassWiseExam(Request $request)
    {
        try {
            $exams = Result::with('exam:id,name')
                ->where('class_id', $request->class_id)
                ->groupBy('exam_id')
                ->get();
            return response()->json($exams);
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
 public function ClassWiseExamResult(Request $request)
{
    try {
        // Exams of this class/exam/session
        $exams = Result::where('exam_id', $request->exam_id)
            ->where('class_id', $request->class_id)
            ->when($request->session_id, function ($query, $session_id) {
                return $query->where('session_id', $session_id);
            })
            ->get();

        $className   = StudentClass::find($request->class_id)->name;
        $examName    = StudentExam::find($request->exam_id)->name;
        $sessionName = StudentSession::find($request->session_id)->name;

        // All result details (active students only), grouped by student
        $results = ResultDetail::with(['students:id,name,roll_no,status'])
            ->whereIn('result_id', $exams->pluck('id'))
            ->get()
            ->filter(function ($detail) {
                return $detail->students && ($detail->students->status == 1 || is_null($detail->students->status));
            })
            ->groupBy('students.id');

        // Subjects of this class
        $subjects = SubjectAssign::with(['subjects:id,name'])
            ->where('class_id', $request->class_id)
            ->get();

        $invoice = InvoiceDesign::first();

        // Global grade bands (used for % → Grade/GPA)
        $grades = SubjectGrade::all();

        // ===== Pre-compute per-subject MAX/EXAM marks for header row =====
        $subjectWiseGrades = SubjectWiseGrades::where('exam_id', $request->exam_id)
            ->where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->whereNull('deleted_at')
            ->get()
            ->keyBy('subject_id');

        $examWiseGrade = ExamWiseGrades::where('exam_id', $request->exam_id)
            ->where('session_id', $request->session_id)
            ->whereNull('deleted_at')
            ->first();

        $examMarksBySubject = [];
        foreach ($subjects as $sub) {
            $sid = $sub->subject_id;
            $examMarksBySubject[$sid] = $subjectWiseGrades->has($sid)
                ? (int) ($subjectWiseGrades[$sid]->exam_mark ?? 100)
                : (int) ($examWiseGrade->exam_mark ?? 100);
        }
        // ===== END =====

        // Compute totals, percentage, and global grade per student
        $studentRanks = [];
        foreach ($results as $studentId => $studentResults) {
            $totalMark     = 0.0;
            $totalExamMark = 0;

            foreach ($studentResults as $result) {
                // Pick grade scale for per-subject meta (only to know exam_mark fallback)
                $subjectWiseGrade = SubjectWiseGrades::where('subject_id', $result->subject_id)
                    ->where('exam_id', $request->exam_id)
                    ->where('class_id', $request->class_id)
                    ->where('session_id', $request->session_id)
                    ->whereNull('deleted_at')
                    ->first();

                // Per-subject computed helpers (kept for cell styling/consistency)
                $grade = null;
                if ($subjectWiseGrade) {
                    $start_marks   = json_decode($subjectWiseGrade->start_mark);
                    $end_marks     = json_decode($subjectWiseGrade->end_mark);
                    $subjectGrades = json_decode($subjectWiseGrade->grade);
                    $gpas          = json_decode($subjectWiseGrade->gpa);

                    foreach ($start_marks as $index => $start) {
                        $start = (float) $start;
                        $end   = (float) ($end_marks[$index] ?? 0);
                        if ($result->mark >= $start && $result->mark <= $end) {
                            $grade = (object) [
                                'name'      => $subjectGrades[$index] ?? 'N/A',
                                'gpa'       => $gpas[$index] ?? 0,
                                'exam_mark' => $subjectWiseGrade->exam_mark ?? 100,
                            ];
                            break;
                        }
                    }
                } elseif ($examWiseGrade) {
                    $start_marks   = json_decode($examWiseGrade->start_mark);
                    $end_marks     = json_decode($examWiseGrade->end_mark);
                    $subjectGrades = json_decode($examWiseGrade->grade);
                    $gpas          = json_decode($examWiseGrade->gpa);

                    foreach ($start_marks as $index => $start) {
                        $start = (float) $start;
                        $end   = (float) ($end_marks[$index] ?? 0);
                        if ($result->mark >= $start && $result->mark <= $end) {
                            $grade = (object) [
                                'name'      => $subjectGrades[$index] ?? 'N/A',
                                'gpa'       => $gpas[$index] ?? 0,
                                'exam_mark' => $examWiseGrade->exam_mark ?? 100,
                            ];
                            break;
                        }
                    }
                }

                if (! $grade) {
                    // fallback to generic grading bands
                    $grade = $grades->filter(function ($g) use ($result) {
                        [$start, $end] = explode('-', str_replace(' ', '', $g->mark));
                        return $result->mark >= (int) $start && $result->mark <= (int) $end;
                    })->sortByDesc(function ($g) {
                        [$start, $end] = explode('-', str_replace(' ', '', $g->mark));
                        return (int) $start;
                    })->first();
                }

                // Set per-subject fields used in table
                $result->grade     = $grade->name ?? $grade->grade ?? 'N/A';

                // dd($result, $grade);
                $result->exam_mark = isset($examMarksBySubject[$result->subject_id])
                    ? $examMarksBySubject[$result->subject_id]
                    : (int) ($grade->exam_mark ?? 100);
                $result->gpa       = number_format((float) ($grade->gpa ?? 0), 2);

                // Aggregation
                $mark = is_numeric($result->mark) ? (float) $result->mark : 0.0;
                $totalExamMark += (int) $result->exam_mark;
                $totalMark     += $mark;
            }

            // Percentage-based average and global grade
            $percent = $totalExamMark > 0 ? ($totalMark / $totalExamMark) * 100 : 0.0;

            $globalGrade = $grades->first(function ($g) use ($percent) {
                $band = str_replace(' ', '', $g->mark);
                if (strpos($band, '-') !== false) {
                    [$start, $end] = explode('-', $band, 2);
                    return $percent >= (float) $start && $percent <= (float) $end;
                }
                return $percent >= (float) $band; // single-threshold style like "80"
            });

            $divisionName = $globalGrade->name ?? 'অনুত্তীর্ণ';
            $avgGpa       = number_format((float) ($globalGrade->gpa ?? 0), 2);
            $avgMark      = number_format($percent, 2); // shown as percentage

            $studentRanks[$studentId] = [
                'totalMark' => $totalMark,
                'avgMark'   => $avgMark,   // percentage "xx.xx"
                'avgGpa'    => $avgGpa,    // global GPA "x.xx"
                'division'  => $divisionName,
            ];
        }

        // Sort by totalMark DESC
        uasort($studentRanks, function ($a, $b) {
            return $b['totalMark'] <=> $a['totalMark'];
        });

        $invoiceHeader = $invoice->invoice_header ?? null;
        $invoiceFooter = $invoice->invoice_footer ?? null;

        // ================= Build HTML =================
        $html  = '<hr><div class="result-table-container">';
        $html .= '<style>
            .result-table thead tr.exam-max th{writing-mode:horizontal-tb!important;transform:none!important;}
            .gpa-zero { color:red; font-weight:bold; }
            .low-mark { color:black; }
        </style>';

        if ($invoiceHeader) {
            $html .= '<div class="header-image" style="text-align:center; margin-bottom: 10px;">';
            $html .= '<img src="' . asset("img/" . $invoiceHeader) . '" alt="Header" style="max-width:100%; width:100%; height:200px;">';
            $html .= '<div style="display:flex;justify-content:center;align-items:center;gap:16px;"><strong>Class: ' . e($className) . '</strong><strong>Exam: ' . e($examName) . '</strong><strong>Session: ' . e($sessionName) . '</strong></div>';
            $html .= '</div>';
        }

        $html .= '<table class="result-table table table-bordered table-hover print-table">';
        $html .= '<thead>';

        // Main header
        $html .= '<tr class="head-main">';
        $html .= '<th>ক্র.নং</th>';
        $html .= '<th>রোল নং</th>';
        $html .= '<th>পরীক্ষার্থীর নাম</th>';
        foreach ($subjects as $subject) {
            $html .= '<th>' . e($subject->subjects->name) . '</th>';
        }
        $html .= '<th>মোট</th>';
        $html .= '<th>গড় মার্ক</th>';
        $html .= '<th>বিভাগ</th>';
        $html .= '<th>GPA</th>';
        $html .= '<th>মেধাস্থান</th>';
        $html .= '</tr>';

        // Second header: subject-wise MAX exam marks
        $html .= '<tr class="exam-max">';
        $html .= '<th colspan="3">Exam Mark</th>';
        $totalExamMarks = 0;
        foreach ($subjects as $subject) {
            $sid = $subject->subject_id;
            $max = $examMarksBySubject[$sid] ?? 100;
            $totalExamMarks += $max;
            $html .= '<th>' . (int) $max . '</th>';
        }
        $html .= '<th colspan="">' . $totalExamMarks . '</th>';
        $html .= '<th colspan="4"></th>';
        $html .= '</tr>';

        $html .= '</thead>';
        $html .= '<tbody>';

        $rank          = 1;
        $previousTotal = null;
        $previousRank  = 0;
        $loop          = 1;

        foreach ($studentRanks as $studentId => $rankData) {
            $studentResults = $results[$studentId];
            $firstResult    = $studentResults->first();

            if ($previousTotal !== null && $rankData['totalMark'] == $previousTotal) {
                $rankData['rank'] = $previousRank;
                $previousTotal    = $rankData['totalMark'];
            } elseif ($previousTotal !== null && $rankData['totalMark'] !== $previousTotal) {
                $rankData['rank'] = $previousRank + 1;
                $previousTotal    = $rankData['totalMark'];
                $previousRank     = $rankData['rank'];
            } else {
                $rankData['rank'] = $rank;
                $previousTotal    = $rankData['totalMark'];
                $previousRank     = $rank;
            }

            $html .= '<tr>';
            $html .= '<td>' . $loop . '</td>';
            $html .= '<td>' . e($firstResult->students->roll_no) . '</td>';
            $html .= '<td>' . e($firstResult->students->name) . '</td>';

            // Subject-wise cells (with GPA=0 => red)
            foreach ($subjects as $subject) {
                $subjectMark = $studentResults->where('subject_id', $subject->subject_id)->first();
                $mark        = $subjectMark ? (is_numeric($subjectMark->mark) ? (float) $subjectMark->mark : '-') : '-';

                // default style
                $markClass = '';

                // If subject GPA == 0 => red & bold
                if ($subjectMark && (float)($subjectMark->gpa ?? 0) == 0.0) {
                    $markClass = 'class="gpa-zero"';
                }
                // else if very low mark (legacy rule)
                elseif (is_numeric($mark) && $mark <= 20) {
                    $markClass = 'class="low-mark"';
                }

                $html .= '<td ' . $markClass . '>' . (is_numeric($mark) ? $mark : '-') . '</td>';
            }

            $html .= '<td>' . $rankData['totalMark'] . '</td>';
            $html .= '<td>' . $rankData['avgMark'] . '</td>';           // percentage
            $html .= '<td>' . e($rankData['division']) . '</td>';       // global grade name
            $html .= '<td>' . e($rankData['avgGpa']) . '</td>';         // global GPA
            $html .= '<td>' . $rankData['rank'] . '</td>';
            $html .= '</tr>';

            $loop++;
        }
        unset($rankData);

        $html .= '</tbody>';
        $html .= '</table>';

        if ($invoiceFooter) {
            $html .= '<div class="footer-image" style="text-align:center; margin-top: 10px;">';
            $html .= '<img src="' . asset("img/" . $invoiceFooter) . '" alt="Footer" style="max-width:100%; height:auto;">';
            $html .= '</div>';
        }

        $html .= '<button onclick="window.print()" class="btn btn-danger float-right">Print</button>';
        $html .= '</div>';

        return response()->json(['html' => $html]);
    } catch (\Exception $exception) {
        return response()->json(['error' => $exception->getMessage()], 500);
    }
}


    public function AllResult()
    {
        try {
            $classes  = StudentClass::all();
            $sessions = StudentSession::all();
            $invoice  = InvoiceDesign::first();
            return view('dashboard.results.all_result', compact('classes', 'sessions', 'invoice'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
}
