<?php
namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\AdmissionFee;
use App\Models\AdmitCard;
use App\Models\ExamWiseGrades;
use App\Models\FeeAssign;
use App\Models\FeeAssignDetail;
use App\Models\Fund;
use App\Models\FundDetail;
use App\Models\InvoiceDesign;
use App\Models\Leave;
use App\Models\MoneyReceipt;
use App\Models\Result;
use App\Models\Setting;
use App\Models\SmsStatus;
use App\Models\Student;
use App\Models\StudentClass;
use App\Models\StudentComment;
use App\Models\StudentExam;
use App\Models\StudentFeeDetail;
use App\Models\StudentHistory;
use App\Models\StudentNeshab;
use App\Models\StudentSession;
use App\Models\SubjectAssign;
use App\Models\SubjectWiseGrades;
use App\Models\User;
use App\Traits\DateFormatter;
use App\Traits\DeleteTrait;
use App\Traits\SmsTrait;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Yajra\DataTables\Facades\DataTables;

class StudentController extends Controller
{
    use SmsTrait;
    use DateFormatter;
    use DeleteTrait;

    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request)
    {
        try {

            if ($request->ajax()) {

                $query = Student::with('stuclass')
                    ->where(function ($query) {
                        $query->where('status', 1)
                            ->orWhereNull('status');
                    })
                    ->when($request->class_id, function ($q) use ($request) {
                        $q->where('class_id', $request->class_id);
                    })
                    ->when($request->student_type, function ($q) use ($request) {
                        $q->where('student_type', $request->student_type);
                    })
                    ->when($request->etim_type, function ($q) use ($request) {
                        $q->where('etim', $request->etim_type);
                    })

                    ->when($request->gender, function ($q) use ($request) {
                        $q->where('gender', $request->gender);
                    })

                    ->when($request->admission_type, function ($q) use ($request) {
                        $q->where('admission_type', $request->admission_type);
                    })
                    ->when($request->session_id, function ($q) use ($request) {
                        $q->where('session_id', $request->session_id);
                    });

                $data = $query->whereNull('deleted_at')->get();

                return Datatables::of($data)

                    ->addColumn('class', function ($data) {
                        $class = $data->stuclass->name ?? '--';
                        return $class;
                    })

                    ->addColumn('student_type', function ($data) {
                        $button = '';
                        if ($data->student_type == 2) {
                            $button = '<span class="badge badge-primary" title="Residential">আবাসিক</span>';
                        } elseif ($data->student_type == 1) {
                            $button = '<span class="badge badge-primary" title="Non-Residential">অনাবাসিক</span>';
                        } elseif ($data->student_type == 3) {
                            $button = '<span class="badge badge-primary" title="de-care">ডে-কেয়ার</span>';
                        }
                        return $button;
                    })
                    ->addColumn('admission_type', function ($data) {
                        $button = '';
                        if ($data->admission_type == 1) {
                            $button = '<span">  নতুন ভর্তি</span>';
                        } elseif ($data->admission_type == 2) {
                            $button = '<span > পুরাতন ভর্তি</span>';
                        }
                        return $button;
                    })

                    ->addColumn('admission_fee', function ($data) {
                        return ($data->admission_fee ?? '');
                    })
                    ->addColumn('monthly_dis', function ($data) {
                        return ($data->monthly_dis ?? '');
                    })
                    ->editColumn('register_no', function ($data) {
                        return ($data->register_no ?? 'N/A');
                    })
                    ->addColumn('roll_no', function ($data) {
                        return $data->roll_no ?? 'N/A';
                    })
                    ->addColumn('father_name', function ($data) {
                        return ($data->father_name ?? 'N/A');
                    })
                    ->addColumn('address', function ($data) {
                        return ($data->gurdian_address ?? 'N/A');
                    })
                    ->addColumn('gurdian_mobile', function ($data) {
                        $mobile = $data->gurdian_mobile ?? '--';
                        return $mobile != '--'
                            ? '<a id="mobileNumber" href="tel:' . $mobile . '" style="color:black !important; text-decoration:none;">' . $mobile . '</a>'
                            : '--';

                    })
                    ->addColumn('image', function ($data) {
                        // Default logo
                        $logo = Setting::first()->logo ?? 'no-image.png';

                        // Student image path
                        $image = $data->image ?? null;

                        // Student image full path (server side check)
                        $imagePath = public_path('backend/img/student/' . $image);

                        // Check if student image exists
                        if ($image && file_exists($imagePath)) {
                            $imgSrc = asset('backend/img/student/' . $image);
                        } else {
                            // Use default logo if image missing
                            $imgSrc = asset('img/' . $logo);
                        }

                        // Return image tag
                        return '<img src="' . $imgSrc . '" width="50px" height="50px" class="rounded-circle" alt="Student Image">';
                    })

                    ->addColumn('action', function ($data) {

                        $result = '';

                        $idcard = ' <li><a title="a6 page" class="dropdown-item" href="' . route('student.idcard', $data->id) . ' " ><i class="fas fa-file"></i> ID-Card</a></li>';

                        $details = '<li><a class="dropdown-item" href="' . route('student.show', $data->id) . ' " ><i class="fa fa-eye" ></i> Details</a></li>';

                        $edit = '<li><a class="dropdown-item" href="' . route('student.edit', $data->id) . ' "><i class="fa fa-edit"></i> Edit</a></li>';

                        $form = '<li><a title="a4 page" class="dropdown-item" href="' . route('form-print', $data->id) . ' "><i class="fa fa-file"></i> Form Print</a></li>';

                        $fee = '<li><a title="a6 page" class="dropdown-item" href="' . route('fee.print', $data->id) . ' "><i class="fa fa-file"></i> Fee Print</a></li>';

                        $status = '<li> <a title="Deactivate" class="dropdown-item status" href="' . route('student.status', $data->id) . '"> <i class="fa fa-bars"></i> Deactivate  </a> </li>';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<li><a class="dropdown-item btn-delete" href="#" data-remote=" ' . route('student.destroy', $data->id) . ' "><i class="fa fa-trash"></i> Delete</a></li>';
                        }

                        $result = $idcard . $details . $edit . $form . $fee . $status . $delete;

                        return '<div class="btn-group open">
                        <a class="badge badge-primary dropdown-toggle" href="#" role="button"  data-toggle="dropdown">Actions<i class="ik ik-chevron-down mr-0 align-middle"></i></a>
                        <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">' . $result . '
                        </ul>
                    </div>';
                    })

                    ->addIndexColumn()
                    ->rawColumns(['class', 'student_type', 'action', 'admission_type', 'admission_fee', 'father_name', 'register_no', 'roll_no', 'monthly_dis', 'gurdian_mobile', 'address', 'image'])
                    ->toJson();
            }
            $classes  = StudentClass::all();
            $sessions = StudentSession::all();
            return view('dashboard.students.student.index', compact('classes', 'sessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function InactiveStudent(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = Student::with('stuclass')->where('status', 2)->get();

                return Datatables::of($data)

                    ->addColumn('name', function ($data) {
                        $name = $data->name ?? '--';
                        $reg  = $data->register_no ?? '--';
                        return $name . '<br>' . $reg;
                    })

                    ->addColumn('date', function ($data) {
                        $date = Carbon::parse($data->updated_at)->format('d-m-Y') ?? '--';
                        return $date;
                    })

                    ->addColumn('class', function ($data) {
                        $class = $data->stuclass->name ?? '--';
                        return $class;
                    })

                    ->addColumn('action', function ($data) {

                        $btn = '<a title="active" class="btn btn-primary status" href="' . route('student.status', $data->id) . '"> <i class="fa fa-bars"></i></a> ';

                        return $btn;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['class', 'name', 'action', 'date'])
                    ->toJson();
            }
            return view('dashboard.students.student.inactive');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function najeraStudents(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = DB::table('students')
                    ->leftJoin('student_classes', 'student_classes.id', '=', 'students.class_id')
                    ->select('students.name', 'students.id', 'students.class_id', 'students.student_type', 'students.register_no', 'student_classes.name')
                    ->where('students.class_id', 2)
                    ->whereNull('students.deleted_at')
                    ->orderBy('id', 'DESC')
                    ->get();

                return Datatables::of($data)
                    ->addColumn('class', function ($data) {
                        $class = $data->name ?? '--';
                        return $class;
                    })

                    ->addColumn('student_type', function ($data) {

                        if ($data->student_type == 1) {
                            $button = '<span class="badge badge-primary" title="Residential">আবাসিক</span>';
                        } elseif ($data->student_type == 2) {
                            $button = '<span class="badge badge-primary" title="Non-Residential">অনাবাসিক</span>';
                        }
                        return $button;
                    })

                    ->addColumn('action', function ($data) {

                        $result = '';

                        $fees = ' <li><a class="dropdown-item" href="' . route('student-fee', $data->id) . ' " ><i class="fas fa-dollar-sign"></i> Fees Collect</a></li>';

                        $details = '<li><a class="dropdown-item" href="' . route('student.show', $data->id) . ' " ><i class="fa fa-eye" ></i> Details</a></li>';

                        $edit = '<li><a class="dropdown-item" href="' . route('student.edit', $data->id) . ' "><i class="fa fa-edit"></i> Edit</a></li>';

                        $delete = '<li><a class="dropdown-item btn-delete" href="#" data-remote=" ' . route('student.destroy', $data->id) . ' "><i class="fa fa-trash"></i> Delete</a></li>';

                        $result = $fees . $details . $edit . $delete;

                        return '<div class="btn-group open">
                        <a class="badge badge-primary dropdown-toggle" href="#" role="button"  data-toggle="dropdown">Actions<i class="ik ik-chevron-down mr-0 align-middle"></i></a>
                        <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">' . $result . '
                        </ul>
                    </div>';

                    })

                    ->addIndexColumn()
                    ->rawColumns(['class', 'student_type', 'action'])
                    ->toJson();
            }
            return view('dashboard.students.student.najera');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function hefojStudents(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = DB::table('students')
                    ->leftJoin('student_classes', 'student_classes.id', '=', 'students.class_id')
                    ->select('students.name', 'students.id', 'students.class_id', 'students.student_type', 'students.register_no', 'student_classes.name')
                    ->where('students.class_id', 3)
                    ->whereNull('students.deleted_at')
                    ->orderBy('id', 'DESC')
                    ->get();

                return Datatables::of($data)
                    ->addColumn('class', function ($data) {
                        $class = $data->name ?? '--';
                        return $class;
                    })

                    ->addColumn('student_type', function ($data) {

                        if ($data->student_type == 1) {
                            $button = '<span class="badge badge-primary" title="Residential">আবাসিক</span>';
                        } elseif ($data->student_type == 2) {
                            $button = '<span class="badge badge-primary" title="Non-Residential">অনাবাসিক</span>';
                        }
                        return $button;
                    })

                    ->addColumn('admission_type', function ($data) {

                        if ($data->admission_type == 1) {
                            $button = '<span class="badge badge-primary" title="new student"> নতুন ভর্তি</span>';
                        } elseif ($data->admission_type == 2) {
                            $button = '<span class="badge badge-primary" title="old student"> পুরান ভর্তি</span>';
                        }
                        return $button;
                    })

                    ->addColumn('action', function ($data) {

                        $result = '';

                        $fees = ' <li><a class="dropdown-item" href="' . route('student-fee', $data->id) . ' " ><i class="fas fa-dollar-sign"></i> Fees Collect</a></li>';

                        $details = '<li><a class="dropdown-item" href="' . route('student.show', $data->id) . ' " ><i class="fa fa-eye" ></i> Details</a></li>';

                        $edit = '<li><a class="dropdown-item" href="' . route('student.edit', $data->id) . ' "><i class="fa fa-edit"></i> Edit</a></li>';

                        $delete = '<li><a class="dropdown-item btn-delete" href="#" data-remote=" ' . route('student.destroy', $data->id) . ' "><i class="fa fa-trash"></i> Delete</a></li>';

                        $result = $fees . $details . $edit . $delete;

                        return '<div class="btn-group open">
                        <a class="badge badge-primary dropdown-toggle" href="#" role="button"  data-toggle="dropdown">Actions<i class="ik ik-chevron-down mr-0 align-middle"></i></a>
                        <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">' . $result . '
                        </ul>
                    </div>';

                    })

                    ->addIndexColumn()
                    ->rawColumns(['class', 'student_type', 'action', 'admission_type'])
                    ->toJson();
            }
            return view('dashboard.students.student.hefoj');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {

        $class    = StudentClass::all();
        $sessions = StudentSession::all();
        $count    = Student::with('class')->count();
        $yearDate = date('Y');

        $bn_numbers   = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers   = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];
        $formNo       = str_replace($en_numbers, $bn_numbers, ($count + 1));
        $admissionFee = AdmissionFee::first();

        $fund = Fund::first();

        if (empty($fund)) {
            return redirect()->back()->with('error', 'Pelase create fund');
        }

        return view('dashboard.students.student.create', compact('class', 'formNo', 'sessions', 'admissionFee'));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */

    public function RollCheck(Request $request)
    {
        // dd("roll check");
        $class_id   = $request->class_id;
        $session_id = $request->session_id;
        $student_id = $request->student_id;

        $roll_no = '';

        if ($student_id) {
            $existing = Student::where('id', $student_id)
                ->where('class_id', $class_id)
                ->where('session_id', $session_id)
                ->first();

            if ($existing && $existing->roll_no !== null) {
                $roll_no = $existing->roll_no;
            } else {
                $max_roll = Student::where('class_id', $class_id)
                    ->where('session_id', $session_id)
                    ->whereNotNull('roll_no')
                    ->max('roll_no');

                $roll_no = $max_roll ? $max_roll + 1 : 1;
            }
        } else {
            $max_roll = Student::where('class_id', $class_id)
                ->where('session_id', $session_id)
                ->whereNotNull('roll_no')
                ->max('roll_no');

            $roll_no = $max_roll ? $max_roll + 1 : 1;
        }

        return response()->json($roll_no);
    }

    public function store(Request $request)
    {
        $messages = [
            // 'join_date.required' => 'তারিখ লিখুন',
            'name.required'           => 'নাম লিখুন',
            // 'father_name.required' => 'পিতার লিখুন',
            'class_id.required'       => 'ক্লাস নির্বাচন করুন',
            'student_type.required'   => 'শিক্ষার্থীর ধরন নির্বাচন করুন',
            // 'admission_fee.required' => 'ভর্তি ফি লিখুন',
            // 'monthly_fee.required' => 'মাসিক বেতন লিখুন',
            // 'village.required' => 'গ্রাম লিখুন',
            // 'post_office.required' => 'ডাকঘর লিখুন',
            // 'police_station.required' => 'থানা লিখুন',
            // 'district.required' => 'জেলা লিখুন',
            // 'gurdian_name.required' => 'অভিভাবকের নাম লিখুন',
            // 'relation.required' => 'সম্পর্ক লিখুন',
            'gurdian_mobile.required' => 'মোবাইল নং লিখুন',
            // 'gurdian_address.required' => 'অভিভাবকের ঠিকানা লিখুন',
        ];

        $request->validate([
            // 'join_date' => 'required',
            'name'           => 'required',
            // 'father_name' => 'required',
            'class_id'       => 'required',
            // 'monthly_fee' => 'required',
            // 'gurdian_name' => 'required',
            'gurdian_mobile' => 'required',
            // 'relation' => 'required',
            // 'gurdian_address' => 'required',
        ], $messages);

        DB::beginTransaction();

        // try {

        //  dd($request->all());

        $user              = new User();
        $user->name        = $request->name;
        $user->register_no = $request->register_no;
        $user->type        = 3;
        $user->password    = Hash::make('student');
        $user->show_pass   = 'student';
        $user->save();

        $student = new student();

        if ($request->hasfile('image')) {
            $file     = $request->file('image');
            $filename = time() . $file->getClientOriginalName();
            $file->move(public_path('/backend/img/student/'), $filename);
            $student->image = $filename;
        }

        $preReg = Student::where('gender', $request->gender)->latest()->first();

        if ($preReg) {
            $registerNoInt = (int) substr($preReg->register_no, 1);
            $registerNo    = $registerNoInt + 1;

            if ($request->gender == 1) {
                $currentReg = 'M' . $registerNo;
            } elseif ($request->gender == 2) {
                $currentReg = 'F' . $registerNo;
            }
        } else {
            if ($request->gender == 1) {
                $currentReg = 'M' . date('y') . date('md') . str_pad($student->getIncrementNumber('male'), 2, '0', STR_PAD_LEFT);
            } elseif ($request->gender == 2) {
                $currentReg = 'F' . date('y') . date('md') . str_pad($student->getIncrementNumber('female'), 2, '0', STR_PAD_LEFT);
            }
        }

        $student->register_no = $currentReg;

        $student->roll_no        = $request->roll_no;
        $student->user_id        = $user->id;
        $student->fingerprint_id = $request->fingerprint_id;
        $student->id_number      = $request->id_number;
        $student->session_id     = $request->session_id;
        $student->form_no        = $request->form_no;
        $student->join_date      = $this->formatAnyDateToYmd($request->join_date);
        $student->etim           = $request->etim;
        $student->etim_status    = 2;
        $student->name           = $request->name;
        $student->father_name    = $request->father_name;
        $student->mother_name    = $request->mother_name;
        $student->birth_date     = $request->birth_date ? $this->formatAnyDateToYmd($request->birth_date) : null;
        $student->class_id       = $request->class_id;
        $student->student_type   = $request->student_type;
        $student->admission_fee  = $request->admission_fee;
        // $student->admission_fee_due  = $request->admission_fee_due;
        $student->monthly_dis          = $request->monthly_dis;
        $student->discount             = $request->discount;
        $student->village              = $request->village;
        $student->post_office          = $request->post_office;
        $student->police_station       = $request->police_station;
        $student->district             = $request->district;
        $student->gurdian_name         = $request->gurdian_name;
        $student->gurdian_name_two     = $request->gurdian_name_two;
        $student->gurdian_name_three   = $request->gurdian_name_three;
        $student->relation             = $request->relation;
        $student->gurdian_mobile       = $request->gurdian_mobile;
        $student->gurdian_mobile_two   = $request->gurdian_mobile_two;
        $student->gurdian_mobile_three = $request->gurdian_mobile_three;
        $student->gurdian_address      = $request->gurdian_address;
        $student->gurdian_nid          = $request->gurdian_nid;
        $student->pre_madrasha         = $request->pre_madrasha;
        $student->pre_class            = $request->pre_class;
        $student->new_result           = $request->new_result;
        $student->admission_type       = $request->admission_type;
        $student->gender               = $request->gender;
        $student->birthcertificate     = $request->birthcertificate;

        $student->created_by = Auth::user()->id;

        $student->save();

        // foreach($request->fee_id as $key => $value) {
        //     $fee = new StudentFeeDetail();
        //     $fee->fee_assign_id = $request->fee_assign_id[$key];
        //     $fee->student_id = $student->id;
        //     $fee->class_id = $student->class_id;
        //     $fee->fee_id = $value;
        //     $fee->created_by = Auth::user()->id;
        //     $fee->save();
        // }

        $studentHistory                 = new StudentHistory();
        $studentHistory->student_id     = $student->id;
        $studentHistory->register_no    = $student->register_no;
        $studentHistory->session_id     = $student->session_id;
        $studentHistory->form_no        = $student->form_no;
        $studentHistory->name           = $request->name;
        $studentHistory->join_date      = $student->date;
        $studentHistory->student_type   = $request->student_type;
        $studentHistory->admission_fee  = $request->admission_fee_receive;
        $studentHistory->discount       = $request->discount;
        $studentHistory->due            = $request->due;
        $studentHistory->class_id       = $request->class_id;
        $studentHistory->gurdian_mobile = $request->gurdian_mobile;
        $studentHistory->created_by     = Auth::user()->id;
        $studentHistory->save();

        $lastFundDetail = FundDetail::whereNotNull('receipt_no')
            ->orderByDesc('receipt_no')
            ->first();
        $receipt_no = $lastFundDetail ? $lastFundDetail->receipt_no + 1 : 1;

        $fund = Fund::where('fund_type', 1)->first();

        $money                   = new MoneyReceipt();
        $money->student_id       = $student->id;
        $money->receipt_no       = $receipt_no;
        $money->register_no      = $student->register_no;
        $money->date             = $this->formatAnyDateToYmd($request->join_date);
        $money->name             = $student->name;
        $money->class_id         = $student->class_id;
        $money->type             = 14; //admission fee
        $money->month            = now()->format('Y-m');
        $money->received_purpose = 'Student Admission Fee';
        $money->amount           = $request->admission_fee_receive;
        $money->current_receive  = $request->admission_fee_receive;
        $money->discount         = $request->discount;
        $money->due              = $request->due;
        $money->total            = $request->admission_fee;

        $student->created_by = Auth::user()->id;
        $money->save();

        $fundDetail                     = new FundDetail();
        $fundDetail->fund_id            = $fund->id;
        $fundDetail->student_payment_id = $money->id;
        $fundDetail->student_id         = $student->id;
        $fundDetail->receipt_no         = $receipt_no;
        $fundDetail->purpose            = 'Student Admission Fee';
        $fundDetail->date               = $this->formatAnyDateToYmd($request->join_date);
        $fundDetail->type               = 14;
        $fundDetail->payment_type       = 1;
        $fundDetail->fund_type          = $fund->fund_type;
        $fundDetail->month              = now()->format('Y-m');
        $fundDetail->amount             = $request->admission_fee_receive;
        $fundDetail->discount           = $request->discount;
        $fundDetail->due                = $request->due ?? 0;
        $fundDetail->total              = $request->admission_fee ?? 0;
        $student->created_by            = Auth::user()->id;
        $fundDetail->save();

        $status = optional(optional(SmsStatus::first()))->student_admission;

        $site = DB::table('settings')->first();
        $to   = $student->gurdian_mobile;
        $text = $site->name . ' এর পক্ষ থেকে অভিনন্দন ' . $student->name . ', আমাদের প্রতিষ্ঠানে আপনার ভর্তি সম্পূর্ণ হয়েছে';
        if ($status == 1) {
            $this->SendSms($to, $text);
        }

        DB::commit();

        return redirect()->route('student.show', $student->id)
            ->with('success', 'Student created successfully');
        // } catch (\Exception $exception) {
        DB::rollBack();
        //     return redirect()->back()->with('error', $exception->getMessage());
        // }
    }

    // public function getIncrementNumber($gender)
    // {
    //     $latestStudent = Student::where('gender', $gender)
    //                             ->whereDate('created_at', now()->toDateString())
    //                             ->orderByDesc('created_at')
    //                             ->first();

    //     $lastIncrement = $latestStudent ? substr($latestStudent->register_no, -2) : '00';
    //     return (int)$lastIncrement + 1;
    // }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        $data           = Student::with('stuclass')->findOrFail($id);
        $addmissionFees = DB::table('fund_details')->where('student_id', $id)->where('payment_type', 1)->where('type', 14)->first();
        // dd($data);
        $count    = Student::with('stuclass')->count();
        $yearDate = date('Y');

        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

        $birthDay = $data->birth_date
            ? str_replace($en_numbers, $bn_numbers, Carbon::parse($data->birth_date)->format('d/m/Y'))
            : '';

        $year = str_replace($en_numbers, $bn_numbers, $yearDate);
        $date = $data->join_date
            ? str_replace($en_numbers, $bn_numbers, Carbon::parse($data->join_date)->format('d/m/Y'))
            : '';

        // ✅ join_date guard (YYYY-MM)
        $joinYm = $data->join_date ? Carbon::parse($data->join_date)->format('Y-m') : null;

        // ============ Fees (join_date হলে ওই মাস থেকে) ============
        $academicFees = FeeAssignDetail::with('fees')
            ->where('type', 1)
            ->where('class_id', $data->class_id)
            ->where('session_id', $data->session_id)
            ->when($joinYm, fn($q) => $q->where('month', '>=', $joinYm))
            ->whereNull('deleted_at')
            ->get()
            ->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->month)->format('Y-m');
            });

        $hostelFees = FeeAssignDetail::with('fees')
            ->where('type', 2)
            ->where('class_id', $data->class_id)
            ->where('session_id', $data->session_id)
            ->when($joinYm, fn($q) => $q->where('month', '>=', $joinYm))
            ->whereNull('deleted_at')
            ->get()
            ->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->month)->format('Y-m');
            });

        $otherFees = FeeAssignDetail::with('fees')
            ->where('type', 3)
            ->where('class_id', $data->class_id)
            ->where('session_id', $data->session_id)
            ->when($joinYm, fn($q) => $q->where('month', '>=', $joinYm))
            ->whereNull('deleted_at')
            ->get()
            ->groupBy(function ($item) {
                return \Carbon\Carbon::parse($item->month)->format('Y-m');
            });

        $admissionFee = AdmissionFee::first();

        return view('dashboard.students.student.show', compact(
            'data',
            'birthDay',
            'year',
            'date',
            'academicFees',
            'hostelFees',
            'otherFees', // ✅ আগে missing ছিল
            'admissionFee',
            'addmissionFees'
        ));
    }

    public function StudentIdcard($id)
    {
        $data    = Student::with('stuclass', 'stusession')->findOrFail($id);
        $setting = Setting::first();
        $invoice = InvoiceDesign::first();

        return view('dashboard.students.student.id_card', compact('data', 'setting', 'invoice'));

    }

    public function StudentIdcardPrintAll(Request $request)
    {

        $datas   = Student::whereIn('id', $request->student_ids)->with('stuclass', 'stusession')->get();
        $setting = Setting::first();
        $invoice = InvoiceDesign::first();
        // dd("dda");
        // dd($request->all(), $datas);
        if ($datas->isEmpty()) {
            return redirect()->back()->with('error', 'No students selected for ID card printing.');
        } else {
            return view('dashboard.students.student.id_card_print_all', compact('datas', 'setting', 'invoice'));
        }

    }

    public function StudentBookPrint($id)
    {
        $invoice  = InvoiceDesign::first();
        $data     = SubjectAssign::with('subjects', 'stuclass')->findOrFail($id);
        $subjects = SubjectAssign::with('subjects')->where('class_id', $data->class_id)->get();
        return view('dashboard.students.student.book_print', compact('data', 'invoice', 'subjects'));
    }
    public function FeePrint($id)
    {
        $invoice = InvoiceDesign::first();
        $data    = Student::with('stusession', 'stuclass')->findOrFail($id);
        $fund    = FundDetail::where('student_id', $id)->where('type', 14)->first();
        return view('dashboard.students.student.fee_print', compact('data', 'invoice', 'fund'));
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $data     = Student::findOrFail($id);
        $class    = StudentClass::all();
        $sessions = StudentSession::all();

        $academicFees = StudentFeeDetail::with('fees', 'feeAssign')
            ->where('student_id', $id)
            ->whereHas('feeAssign', function ($query) {
                $query->where('type', 1);
            })
            ->get();

        $hostelFees = StudentFeeDetail::with('fees', 'feeAssign')
            ->where('student_id', $id)
            ->whereHas('fees', function ($query) {
                $query->where('type', 2);
            })
            ->get();

        $admissionFee = AdmissionFee::first();
        $fees         = MoneyReceipt::where('student_id', $id)->where('type', 14)->first();
        // dd($admissionFee, $hostelFees, $academicFees);
        return view('dashboard.students.student.edit', compact('data', 'class', 'academicFees', 'hostelFees', 'sessions', 'admissionFee', 'fees'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $messages = [
            'name.required'           => 'নাম লিখুন',
            'class_id.required'       => 'ক্লাস নির্বাচন করুন',
            'gurdian_mobile.required' => 'মোবাইল নং লিখুন',
        ];

        $request->validate([
            'name'           => 'required',
            'class_id'       => 'required',
            'gurdian_mobile' => 'required',
            // 'image' => 'nullable|image|mimes:jpg,jpeg,png,webp|max:2048', // optional hardening
        ], $messages);

        DB::beginTransaction();

        try {
            $student = Student::findOrFail($id);

            // ---------- Image handling: only change when uploading or removing ----------
            $image       = $request->file('image');
            $removeImage = (bool) $request->input('remove_image'); // '1' => true

            if ($image) {
                // delete old if exists
                if ($student->image && file_exists(public_path('/backend/img/student/' . $student->image))) {
                    @unlink(public_path('/backend/img/student/' . $student->image));
                }
                $filename = time() . '_' . preg_replace('/\s+/', '_', $image->getClientOriginalName());
                $image->move(public_path('/backend/img/student/'), $filename);
                $student->image = $filename;

            } elseif ($removeImage) {
                if ($student->image && file_exists(public_path('/backend/img/student/' . $student->image))) {
                    @unlink(public_path('/backend/img/student/' . $student->image));
                }
                $student->image = null;

                // NOTE: no "else { $student->image = $request->current_image; }"
                // So existing image remains untouched when neither uploading nor removing.
            }

            // ---------- Register no: keep if same gender; regenerate if gender changes ----------
            $currentReg = $student->register_no;
            if ((int) $request->gender !== (int) $student->gender) {
                if ((int) $request->gender === 1) {
                    $currentReg = 'M' . date('y') . date('md') . str_pad($student->getIncrementNumber('male'), 2, '0', STR_PAD_LEFT);
                } elseif ((int) $request->gender === 2) {
                    $currentReg = 'F' . date('y') . date('md') . str_pad($student->getIncrementNumber('female'), 2, '0', STR_PAD_LEFT);
                }
            }

            // ---------- Assign fields ----------
            $student->register_no          = $currentReg;
            $student->roll_no              = $request->roll_no;
            $student->fingerprint_id       = $request->fingerprint_id;
            $student->session_id           = $request->session_id;
            $student->id_number            = $request->id_number;
            $student->join_date            = $request->join_date ? Carbon::parse($request->join_date)->format('Y-m-d') : $student->join_date;
            $student->name                 = $request->name;
            $student->father_name          = $request->father_name;
            $student->mother_name          = $request->mother_name;
            $student->birth_date           = $request->birth_date ? Carbon::parse($request->birth_date)->format('Y-m-d') : $student->birth_date;
            $student->class_id             = $request->class_id;
            $student->student_type         = $request->student_type;
            $student->admission_fee        = $request->admission_fee;
            $student->monthly_dis          = $request->monthly_dis;
            $student->discount             = $request->discount;
            $student->etim                 = $request->etim;
            $student->village              = $request->village;
            $student->post_office          = $request->post_office;
            $student->police_station       = $request->police_station;
            $student->district             = $request->district;
            $student->gurdian_name         = $request->gurdian_name;
            $student->gurdian_name_two     = $request->gurdian_name_two;
            $student->gurdian_name_three   = $request->gurdian_name_three;
            $student->relation             = $request->relation;
            $student->gurdian_mobile       = $request->gurdian_mobile;
            $student->gurdian_mobile_two   = $request->gurdian_mobile_two;
            $student->gurdian_mobile_three = $request->gurdian_mobile_three;
            $student->gurdian_address      = $request->gurdian_address;
            $student->gender               = $request->gender;
            $student->gurdian_nid          = $request->gurdian_nid;
            $student->admission_type       = $request->admission_type;
            $student->birthcertificate     = $request->birthcertificate;
            $student->pre_madrasha         = $request->pre_madrasha;
            $student->pre_class            = $request->pre_class;
            $student->new_result           = $request->new_result;
            $student->updated_by           = Auth::user()->id;
            $student->save();

            // ---------- Linked user update ----------
            $user = User::where('id', $student->user_id)->first();
            if ($user) {
                $user->name        = $request->name;
                $user->register_no = $student->register_no;
                $user->type        = 3;
                // Consider NOT resetting password on every update:
                // $user->password = Hash::make('student');
                $user->save();
            }

            // ---------- Student History ----------
            $studentHistory                = new StudentHistory();
            $studentHistory->student_id    = $student->id;
            $studentHistory->register_no   = $student->register_no;
            $studentHistory->form_no       = $student->form_no;
            $studentHistory->name          = $request->name;
            $studentHistory->join_date     = $this->formatAnyDateToYmd($request->join_date);
            $studentHistory->student_type  = $request->student_type;
            $studentHistory->admission_fee = $request->admission_fee;
            $studentHistory->discount      = $request->discount;

            $studentHistory->class_id       = $request->class_id;
            $studentHistory->gurdian_mobile = $request->gurdian_mobile;
            $studentHistory->updated_by     = Auth::user()->name;
            $studentHistory->save();

                                                                 // ---------- Money Receipt & Fund Detail (hardened) ----------
            $fund        = Fund::where('fund_type', 1)->first(); // may be null
            $joinDateYmd = $this->formatAnyDateToYmd($request->join_date);
            $joinMonth   = $request->join_date ? Carbon::parse($request->join_date)->format('Y-m') : null;

            // keep existing receipt no if record exists
            $existingMoney = MoneyReceipt::where('type', 14)
                ->where('student_id', $student->id)
                ->first();

            $money = MoneyReceipt::updateOrCreate(
                [
                    'type'       => 14,
                    'student_id' => $student->id,
                ],
                [
                    'receipt_no'       => optional($existingMoney)->receipt_no, // or generate if you need
                    'register_no'      => $student->register_no,
                    'date'             => $joinDateYmd,
                    'name'             => $student->name,
                    'class_id'         => $student->class_id,
                    'month'            => $joinMonth,
                    'received_purpose' => 'Student Admission Fee',
                    'updated_by'       => Auth::user()->id,
                    'amount'           => $request->admission_fee,
                    "amount"           => $request->admission_fee_receive,
                    'current_receive'  => $request->admission_fee_receive,
                    'discount'         => $request->discount,
                    'due'              => $request->due,
                    'total'            => $request->admission_fee,
                ]
            );

            // Only create/update FundDetail if Fund exists and MoneyReceipt exists
            if ($fund && $money) {
                FundDetail::updateOrCreate(
                    [
                        'type'               => 14,
                        'payment_type'       => 1,
                        'student_payment_id' => $money->id,
                    ],
                    [
                        'fund_id'    => $fund->id,
                        'student_id' => $student->id,
                        'purpose'    => 'Student Admission Fee',
                        'date'       => $joinDateYmd,
                        'fund_type'  => $fund->fund_type,
                        'month'      => $joinMonth,
                        'amount'     => $request->admission_fee_receive,
                        'discount'   => $request->discount,
                        'due'        => $request->due,
                        'total'      => $request->admission_fee,  
                        'updated_by' => Auth::user()->id,
                    ]
                );
            }

            // ---------- Conditional SMS ----------
            $status = optional(SmsStatus::first())->student_admission;
            $site   = DB::table('settings')->first();
            if ($status == 1 && $site && $student->gurdian_mobile) {
                $text = ($site->name ?? '') . ' এর পক্ষ থেকে অভিনন্দন ' . $student->name . ', আমাদের প্রতিষ্ঠানে আপনার ভর্তি সম্পূর্ণ হয়েছে';
                $this->SendSms($student->gurdian_mobile, $text);
            }

            DB::commit();

            return redirect()->route('student.show', $student->id)
                ->with('success', 'Student updated successfully');

        } catch (\Throwable $exception) {
            // dd($exception);
            DB::rollBack();
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {

            $data = Student::findOrFail($id);

            $tables = DB::select("
            SELECT TABLE_NAME
            FROM INFORMATION_SCHEMA.COLUMNS
            WHERE COLUMN_NAME = 'student_id'
            AND TABLE_SCHEMA = DATABASE()
            ");

            foreach ($tables as $table) {
                if ($table->TABLE_NAME !== 'students') {
                    DB::table($table->TABLE_NAME)->where('student_id', $data->id)->delete();
                }
            }
            $data->delete();

            return response()->json([
                'success' => true,
                'message' => 'Student deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Student delete failed',
            ]);
        }
    }

    public function StudentStatus(Request $request, $id)
    {
        try {
            $validated = $request->validate([
                'status' => 'required|in:1,2',
            ]);
            // dd($validated);

            $data = Student::findOrFail($id);

            $data->status = $validated['status'];
            $data->save();

            $sms_status = optional(SmsStatus::first())->student_deactivate;
            $site       = DB::table('settings')->first();
            $to         = $data->gurdian_mobile;
            $text       = $site->name . ' এর পক্ষ থেকে ' . $data->name . ' কে নিষ্ক্রিয় করা হয়েছে।';
            if ($sms_status == 1 && $data->status == 2) {
                // dd($sms_status);
                $this->SendSms($to, $text);
            }

            return response()->json([
                'success' => true,
                'message' => 'Status changed successfully',
            ]);

        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => $exception->getMessage(),
            ]);
        }
    }

    public function fee($id)
    {
        $data       = Student::with('stuclass')->findOrFail($id);
        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];
        $id         = MoneyReceipt::count();
        $receipt    = str_replace($en_numbers, $bn_numbers, ($id + 1));
        $class      = StudentClass::all();
        return view('dashboard.students.fee_collect.create', compact('data', 'receipt'));
    }

    public function print($id)
    {
        $data = Student::with('stuclass')->findOrFail($id);
        // dd($data);
        $count    = Student::with('stuclass')->count();
        $yearDate = date('Y');

        $bn_numbers = ["১", "২", "৩", "৪", "৫", "৬", "৭", "৮", "৯", "০"];
        $en_numbers = ["1", "2", "3", "4", "5", "6", "7", "8", "9", "0"];

        $birthDay = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->birth_date)->format('d/m/Y'));
        $formNo   = str_replace($en_numbers, $bn_numbers, $id);
        $year     = str_replace($en_numbers, $bn_numbers, $yearDate);
        $date     = str_replace($en_numbers, $bn_numbers, Carbon::parse($data->join_date)->format('d/m/Y'));
        $invoice  = InvoiceDesign::first();
        $setting  = Setting::first();

        return view('dashboard.students.student.print', compact('data', 'formNo', 'birthDay', 'year', 'date', 'invoice', 'setting'));
    }

    public function ClassFees(Request $request)
    {
        $academicFees = FeeAssign::with('class:id,name', 'studentsession:id,name')
            ->where('type', 1)
            ->where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->get();

        $academicFeesDetails = FeeAssignDetail::where('type', 1)
            ->whereHas('feeAssign', function ($query) use ($academicFees) {
                $query->whereIn('fee_assign_id', $academicFees->pluck('id'));
            })->get();

        $html          = '';
        $feesAvailable = true;

        $html .= $this->generateFeeTable($academicFeesDetails, "Academic Fee", $request->type == 1 ? 'col-md-12' : 'col-md-6');

        if ($request->type == 2) {

            $hostelFees = FeeAssign::with('class:id,name', 'studentsession:id,name')
                ->where('type', 2)
                ->where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->get();

            $hostelFeesDetails = FeeAssignDetail::where('type', 2)
                ->whereHas('feeAssign', function ($query) use ($hostelFees) {
                    $query->whereIn('fee_assign_id', $hostelFees->pluck('id'));
                })->get();

            $html .= $this->generateFeeTable($hostelFeesDetails, "Hostel Fee", 'col-md-6');
        }

        if ($academicFeesDetails->isEmpty() && ($request->type == 1 || $request->type == 2)) {
            $feesAvailable = false;
        }

        return response()->json([
            'html'          => $html,
            'feesAvailable' => $feesAvailable,
        ]);
    }

    private function generateFeeTable($fees, $title, $columnClass)
    {
        if ($fees->isEmpty()) {
            return '<div class="' . $columnClass . '" style="color: red;">No fees available for ' . $title . '</div>';
        }

        $html = '<div class="' . $columnClass . '">
                    <table class="table table-bordered">
                        <thead>
                            <tr>
                                <th>SN</th>
                                <th>' . $title . '</th>
                                <th class="text-center">Amount</th>
                            </tr>
                        </thead>
                        <tbody>';

        foreach ($fees as $key => $fee) {
            $html .= '<tr>
                        <td class="text-left" style="width: 24.40%;">' . ($key + 1) . '</td>
                        <td class="text-left" style="width: 30%;">' . $fee->fees->name . ' (' . Carbon::parse($fee->month)->format('F, Y') . ')</td>
                        <td class="text-center" style="width: 30%;">' . $fee->amount . '</td>
                      </tr>
                      <input type="hidden" name="fee_assign_id[]" value="' . $fee->id . '">
                      <input type="hidden" name="fee_id[]" value="' . $fee->fees->id . '">';
        }

        $html .= '</tbody></table></div>';

        return $html;
    }

    public function StudentFees(Request $request)
    {
        $monthString = $request->month;
        $carbonDate  = Carbon::createFromFormat('F-Y', $monthString);
        $month       = $carbonDate->format('Y-m');

        $academicFeesDetails = StudentFeeDetail::with('fees', 'feeAssign')
            ->whereHas('feeAssign', function ($query) use ($month) {
                $query->where('month', $month)
                    ->where('type', 1);
            })
            ->where('student_id', $request->student_id)
            ->get();

        $hostelFeesDetails = StudentFeeDetail::with('fees', 'feeAssign')
            ->whereHas('feeAssign', function ($query) use ($month) {
                $query->where('month', $month)
                    ->where('type', 2);
            })
            ->where('student_id', $request->student_id)
            ->get();

        $otherFees = FeeAssignDetail::with('fees')->where('type', 3)->where('month', $month)->get();

        $tablesCount = 0;
        if ($academicFeesDetails->isNotEmpty()) {
            $tablesCount++;
        }

        if ($hostelFeesDetails->isNotEmpty()) {
            $tablesCount++;
        }

        if ($otherFees->isNotEmpty()) {
            $tablesCount++;
        }

        $columnClass = 'col-md-12';
        if ($tablesCount === 2) {
            $columnClass = 'col-md-6';
        } elseif ($tablesCount === 3) {
            $columnClass = 'col-md-4';
        }

        $html = '';

        if ($academicFeesDetails->isNotEmpty()) {
            $html .= $this->StudentFeeTable($academicFeesDetails, "Academic Fee", $columnClass);
        }

        if ($hostelFeesDetails->isNotEmpty()) {
            $html .= $this->StudentFeeTable($hostelFeesDetails, "Hostel Fee", $columnClass);
        }

        if ($otherFees->isNotEmpty()) {
            $html .= $this->OtherFeesTable($otherFees, "Other Fee", $columnClass);
        }

        $total = $this->calculateTotal($academicFeesDetails) + $this->calculateTotal($hostelFeesDetails) + $this->calculateTotal($otherFees);

        $totalForm = $this->generateTotalFormHtml($total);

        return response()->json(['html' => $html, 'totalForm' => $totalForm, 'otherFees' => $otherFees]);
    }

    private function OtherFeesTable($fees, $title, $columnClass)
    {
        $html = '<div class="' . $columnClass . '">
                    <table class="table table-bordered">
                        <thead>
                            <tr>
                                <th>SN</th>
                                <th>' . $title . '</th>
                                <th class="text-center">Amount</th>
                            </tr>
                        </thead>
                        <tbody>';
        $total = 0;

        foreach ($fees as $key => $fee) {
            $total += $fee->amount;
            $html .= '<tr>
                        <td class="text-left" style="width: 24.40%;">' . ($key + 1) . '</td>
                        <td class="text-left" style="width: 30%;">' . $fee->fees->name . ' (' . Carbon::parse($fee->month)->format('F, Y') . ')</td>
                        <td class="text-center" style="width: 30%;">' . $fee->amount . '</td>
                      </tr>';
        }

        $html .= '<tr>
                    <td colspan="2" class="text-left">Total</td>
                    <td class="text-center totalAmount">' . $total . '</td>
                  </tr>';

        $html .= '</tbody></table></div>';

        return $html;
    }

    private function StudentFeeTable($fees, $title, $columnClass)
    {
        $html = '<div class="' . $columnClass . '">
                    <table class="table table-bordered">
                        <thead>
                            <tr>
                                <th>SN</th>
                                <th>' . $title . '</th>
                                <th class="text-center">Amount</th>
                            </tr>
                        </thead>
                        <tbody>';

        $total = 0;

        foreach ($fees as $key => $fee) {
            $total += $fee->feeAssign->amount;
            $html .= '<tr>
                        <td class="text-left" style="width: 24.40%;">' . ($key + 1) . '</td>
                        <td class="text-left" style="width: 30%;">' . $fee->fees->name . '</td>
                        <td class="text-center" style="width: 30%;">' . $fee->feeAssign->amount . '</td>
                    </tr>
                    <input type="hidden" name="fee_assign_id[]" value="' . $fee->id . '">
                    <input type="hidden" name="fee_id[]" value="' . $fee->fees->id . '">';
        }

        $html .= '<tr>
                    <td colspan="2" class="text-left">Total</td>
                    <td class="text-center totalAmount">' . $total . '</td>
                  </tr>';

        $html .= '</tbody></table></div>';

        return $html;
    }

    private function calculateTotal($fees)
    {
        return $fees->sum(function ($fee) {
            $feeAssignAmount = optional($fee->feeAssign)->amount ?? 0;
            $feeAmount       = optional($fee->fees)->amount ?? 0;
            return $feeAssignAmount + $feeAmount;
        });
    }

    private function generateTotalFormHtml($total)
    {
        return '<div class="col-md-3">
                <div>Sub Total: <span id="subtotal">' . $total . '</span></div>
                <div>Receive : <span id="receivedAmount"></span></div>
                <div style="border-top:1px solid black;margin-bottom:8px; width:130px">Due: <span id="dueAmount"></span></div>
            </div>';
    }

    public function StudentBook(Request $request)
    {
        try {
            if ($request->ajax()) {

                $data = SubjectAssign::with('subjects')->groupBy('class_id')->get();

                return Datatables::of($data)

                    ->addColumn('class', function ($data) {
                        $class = DB::table('student_classes')->where('id', $data->class_id)->first();
                        return $class->name ?? '';
                    })

                    ->addColumn('subject', function ($data) {
                        $subjects = SubjectAssign::where('class_id', $data->class_id)->get();
                        $result   = '';
                        foreach ($subjects as $subject) {
                            $name = $subject->subjects->name ?? '';
                            $code = $subject->subjects->code ?? '';
                            $result .= $name . '-' . $code . ', ';
                        }
                        return rtrim($result, ', ');
                    })

                    ->addColumn('action', function ($data) {

                        $print = '<a type="button" id="print" class="btn btn-sm btn-info print mr-1" href="' . route('student.book.print', $data->id) . ' " title="Print"><i class="fa fa-file"></i></a>';

                        return $print;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['class', 'action', 'subject'])
                    ->toJson();
            }
            return view('dashboard.students.student.book');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentLeave(Request $request)
    {
        try {

            if ($request->ajax()) {

                $data = Leave::with('stuclass:id,name', 'students:id,name,roll_no,register_no,status', 'users:id,name')->where('type', 2)->get();

                return Datatables::of($data)

                    ->addColumn('date', function ($data) {
                        $from = Carbon::parse($data->from_date)->format('d-M-Y') ?? '';
                        $to   = Carbon::parse($data->to_date)->format('d-M-Y') ?? '';
                        return $from . '<br> ' . $to;
                    })

                    ->addColumn('student', function ($data) {
                        $name  = isset($data->students) ? $data->students->name : '--';
                        $roll  = isset($data->students) ? $data->students->roll_no : '--';
                        $reg   = isset($data->students) ? $data->students->register_no : '--';
                        $class = isset($data->stuclass) ? $data->stuclass->name : '--';
                        return $name . '<br>' . $class . '<br>' . $roll . '<br>' . $reg;
                    })

                    ->addColumn('givedBy', function ($data) {
                        return $data->users->name ?? '';;
                    })

                    ->addColumn('action', function ($data) {
                        // Check if student is active (status = 1 or null)
                        if (isset($data->students) && ($data->students->status == 1 || $data->students->status == null)) {

                            $details = '<li><a class="dropdown-item" href="' . route('student.leave.show', $data->id) . '"><i class="fa fa-eye text-primary"></i> Details</a></li>';
                            $edit    = '<li><a class="dropdown-item" href="' . route('student.leave.edit', $data->id) . '"><i class="fa fa-edit text-success"></i> Edit</a></li>';
                            $delete  = '<li><a class="dropdown-item btn-delete" href="#" data-remote="' . route('student.leave.destroy', $data->id) . '"><i class="fa fa-trash text-red"></i> Delete</a></li>';

                            return '<div class="btn-group open">
                                <a class="badge badge-primary dropdown-toggle" href="#" role="button" data-toggle="dropdown">
                                    Actions <i class="ik ik-chevron-down mr-0 align-middle"></i>
                                </a>
                                <ul class="dropdown-menu" role="menu" style="width:auto; min-width:auto;">' . $details . $edit . $delete . '</ul>
                            </div>';
                        }

                        return '';
                    })

                    ->addIndexColumn()
                    ->rawColumns(['date', 'givedBy', 'student', 'action'])
                    ->toJson();
            }
            return view('dashboard.students.leaves.leave');
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function ClassWiseStudent(Request $request)
    {
        $students = Student::with('stusession')
            ->where(function ($query) {
                $query->where('status', 1)
                    ->orWhereNull('status');
            })
            ->where('class_id', $request->class_id)
            ->select('id', 'name', 'register_no', 'roll_no', 'session_id', 'class_id')
            ->get();
        return response()->json($students);
    }

    public function AdmitWiseStudent(Request $request)
    {
        $students = AdmitCard::with(['students:id,name,register_no,roll_no'])
            ->where('exam_id', $request->exam_id)
            ->where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->whereHas('students', function ($query) use ($request) {
                $query->whereDoesntHave('results', function ($q) use ($request) {
                    $q->where('exam_id', $request->exam_id)
                        ->where('class_id', $request->class_id)
                        ->where('session_id', $request->session_id);
                })
                    ->where(function ($query) {
                        $query->where('status', 1)
                            ->orWhereNull('status');
                    });
            })
            ->get();

        return response()->json($students);
    }

    public function SessionWiseStudent(Request $request)
    {
        $students = Student::where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->where(function ($query) {
                $query->where('status', 1)
                    ->orWhereNull('status');
            })
            ->select('id', 'name', 'register_no', 'roll_no', 'session_id', 'class_id')
            ->with([
                'examSeat'    => function ($q) use ($request) {
                    $q->where('exam_id', $request->exam_id);
                },
                'examComment' => function ($q) use ($request) {
                    $q->where('exam_id', $request->exam_id);
                },
            ])
            ->get();

        // dd($request->all(), $students);

        return response()->json($students);
    }

    public function SessionWiseClass(Request $request)
    {
        $data = Student::with('stuclass:id,name')->where('session_id', $request->session_id)
            ->select('id', 'name', 'register_no', 'roll_no', 'session_id', 'class_id', 'status')
            ->groupBy('class_id')
            ->where(function ($query) {
                $query->where('status', 1)
                    ->orWhereNull('status');
            })
            ->get();
        return response()->json($data);
    }

    public function ExamWiseStudent(Request $request)
    {
        $students = Result::with([
            'students:id,name,register_no,roll_no,status',
            'students.examComment' => function ($q) use ($request) {
                $q->where('exam_id', $request->exam_id)
                    ->where('class_id', $request->class_id)
                    ->where('session_id', $request->session_id);
            },
        ])
            ->where('exam_id', $request->exam_id)
            ->where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->get()
            ->pluck('students')
            ->filter(function ($student) {
                return $student->status === 1 || is_null($student->status);
            })
            ->unique('id')
            ->values();

        return response()->json($students);
    }

    public function checkSonadGiving(Request $request)
    {
        $student = Student::findOrFail($request->student_id);

        $exists = FundDetail::where('student_id', $student->id)
            ->where('exam_id', $request->exam_id)
            ->where('class_id', $student->class_id)
            ->where('session_id', $student->session_id)
            ->exists();

        return response()->json(['exists' => $exists]);
    }

    public function StudentWiseExam(Request $request)
    {
        $students = Result::with('exam')->where('student_id', $request->student_id)->get();
        return response()->json($students);
    }
    public function ResultWiseStudent(Request $request)
    {
        $resultExists = Result::with('students:id,name')
            ->where('student_id', $request->student_id)
            ->where('class_id', $request->class_id)
            ->where('session_id', $request->session_id)
            ->where('exam_id', $request->exam_id)
            ->exists();

        if ($resultExists) {
            return response()->json([
                'success' => false,
                'message' => 'This student result already exists.',
            ]);
        }

    }

    public function StudentLeaveCreate()
    {
        $classes = StudentClass::all();
        return view('dashboard.students.leaves.leave_create', compact('classes'));
    }
    public function StudentLeaveStore(Request $request)
    {
        $messages = [
            'student_id.required' => 'Select student',
            'class_id.required'   => 'Select class',
            'from_date.required'  => 'Enter from date',
            'to_date.required'    => 'Enter to date',
        ];

        $request->validate([
            'student_id' => 'required',
            'from_date'  => 'required',
            'class_id'   => 'required',
            'to_date'    => 'required',
        ], $messages);

        try {

            $from = Carbon::parse($request->from_date)->format('Y-m-d');
            $to   = Carbon::parse($request->to_date)->format('Y-m-d');

            $exists = Leave::where('student_id', $request->student_id)->where('class_id', $request->class_id)
                ->where(function ($query) use ($from, $to) {
                    $query->whereBetween('from_date', [$from, $to])
                        ->orWhereBetween('to_date', [$from, $to])
                        ->orWhere(function ($query) use ($from, $to) {
                            $query->where('from_date', '<=', $from)
                                ->where('to_date', '>=', $to);
                        });
                })
                ->exists();

            if ($exists) {
                return redirect()->back()->with('error', 'Leave for the selected dates has already been applied for.');
            }

            $data                = new Leave();
            $data->class_id      = $request->class_id;
            $data->student_id    = $request->student_id;
            $data->from_date     = Carbon::parse($request->from_date)->format('Y-m-d');
            $data->to_date       = Carbon::parse($request->to_date)->format('Y-m-d');
            $data->days          = $request->days;
            $data->type          = 2;
            $data->gurdian_name  = $request->gurdian_name;
            $data->gurdian_phone = $request->gurdian_phone;
            $data->leave_type    = $request->leave_type;
            $data->note          = $request->note;
            $data->created_by    = Auth::user()->id;
            $data->save();

            $student = Student::findOrFail($request->student_id);
            $to      = $student->gurdian_mobile;
            $site    = DB::table('settings')->first();
            $text    = $student->name . ', ' . $data->from_date . ' থেকে ' . $data->to_date . ' ছুটি হয়েছে ';
            $status  = optional(SmsStatus::first())->student_leave;
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            return redirect()->route('student.leave')->with('success', 'Leave created successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }
    public function StudentLeaveUpdate(Request $request, $id)
    {
        $messages = [
            'student_id.required' => 'Select student',
            'class_id.required'   => 'Select class',
            'from_date.required'  => 'Enter from date',
            'to_date.required'    => 'Enter to date',
        ];

        $request->validate([
            'student_id' => 'required',
            'from_date'  => 'required',
            'class_id'   => 'required',
            'to_date'    => 'required',
        ], $messages);

        try {

            $from   = Carbon::parse($request->from_date)->format('Y-m-d');
            $to     = Carbon::parse($request->to_date)->format('Y-m-d');
            $exists = Leave::where('student_id', $request->student_id)->where('class_id', $request->class_id)
                ->where('id', '!=', $id)
                ->where(function ($query) use ($from, $to) {
                    $query->whereBetween('from_date', [$from, $to])
                        ->orWhereBetween('to_date', [$from, $to])
                        ->orWhere(function ($query) use ($from, $to) {
                            $query->where('from_date', '<=', $from)
                                ->where('to_date', '>=', $to);
                        });
                })
                ->exists();

            if ($exists) {
                return redirect()->back()->with('error', 'Leave for the selected dates has already been applied for.');
            }

            $data                = Leave::findOrFail($id);
            $data->class_id      = $request->class_id;
            $data->student_id    = $request->student_id;
            $data->from_date     = Carbon::parse($request->from_date)->format('Y-m-d');
            $data->to_date       = Carbon::parse($request->to_date)->format('Y-m-d');
            $data->days          = $request->days;
            $data->type          = 2;
            $data->leave_type    = $request->leave_type;
            $data->gurdian_name  = $request->gurdian_name;
            $data->gurdian_phone = $request->gurdian_phone;
            $data->note          = $request->note;
            $data->created_by    = Auth::user()->id;
            $data->save();

            $student = Student::findOrFail($request->student_id);
            $site    = DB::table('settings')->first();
            $to      = $student->gurdian_mobile;
            $text    = $student->name . ', ' . $data->from_date . ' থেকে ' . $data->to_date . ' ছুটি হয়েছে ';
            $status  = optional(SmsStatus::first())->student_leave;
            if ($status == 1) {
                $this->SendSms($to, $text);
            }

            return redirect()->route('student.leave')->with('success', 'Leave updated successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function StudentLeaveDestroy($id)
    {
        try {
            $data = Leave::findOrFail($id);
            $data->delete();

            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }
    public function StudentLeaveShow($id)
    {
        try {
            $invoice = InvoiceDesign::first();
            $data    = Leave::with('stuclass:id,name', 'students:id,name,roll_no,register_no', 'users:id,name')->findOrFail($id);
            return view('dashboard.students.leaves.leave_show', compact('data', 'invoice'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentLeaveEdit($id)
    {
        try {
            $data    = Leave::with('stuclass:id,name', 'students:id,name,roll_no,register_no', 'users:id,name')->findOrFail($id);
            $classes = StudentClass::all();
            return view('dashboard.students.leaves.leave_edit', compact('data', 'classes'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function StudentNeshab(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = StudentNeshab::with('stuclass:id,name', 'exam:id,name', 'stusession')
                    ->get()
                    ->groupBy(function ($item) {
                        return $item->exam_id . '-' . $item->class_id . '-' . $item->session_id;
                    });

                return Datatables::of($data)
                    ->addColumn('exam', function ($data) {
                        $exam = $data->first()->exam;
                        return isset($exam) ? $exam->name : '--';
                    })
                    ->addColumn('stusession', function ($data) {
                        $stusession = $data->first()->stusession;
                        return isset($stusession) ? $stusession->name : '--';
                    })
                    ->addColumn('class', function ($data) {
                        $class = $data->first()->stuclass;
                        return isset($class) ? $class->name : '--';
                    })
                    ->addColumn('action', function ($data) {
                        $firstItem = $data->first();

                        $details = '<a href="' . route('student.neshab.show', $firstItem->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i></a> ';
                        $edit    = '<a href="' . route('student.neshab.edit', $firstItem->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i></a> ';

                        $delete = '';
                        if ($this->DeleteData()) {
                            $delete = '<a href="#" data-remote="' . route('student.neshab.destroy', $firstItem->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i></a>';
                        }

                        return $details . $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['exam', 'class', 'action', 'stusession'])
                    ->toJson();
            }

            return view('dashboard.students.neshab.index');
        } catch (\Exception $exception) {

            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentNeshabCreate()
    {
        $classes     = StudentClass::all();
        $exams       = StudentExam::all();
        $stusessions = StudentSession::all();
        return view('dashboard.students.neshab.create', compact('classes', 'exams', 'stusessions'));
    }

    public function SubjectFind(Request $request)
    {
        try {
            $subjects = SubjectAssign::with('subjects:id,name')->where('class_id', $request->class_id)->get();
            $html     = '';
            foreach ($subjects as $key => $subject) {
                $html .= '<tr>';
                $html .= '<td>' . ($key + 1) . '</td>';
                $html .= '<td>' . $subject->subjects->name . '</td>';
                $html .= '<td><input type="text" name="start_page[]" class="form-control" placeholder="start page" required></td>';
                $html .= '<td><input type="text" name="end_page[]" class="form-control" placeholder="end page" required></td>';
                $html .= '<td><textarea name="note[]" class="form-control" rows="1" placeholder="note"></textarea></td>';
                $html .= '<input type="hidden" name="subject_id[]" value="' . $subject->subject_id . '">';
                $html .= '</tr>';
            }

            return response()->json(['html' => $html]);
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function ClassWiseSubject(Request $request)
    {
        try {
            // dd($request->all());
            $subjects         = SubjectAssign::with('subjects:id,name')->where('class_id', $request->class_id)->get();
            $subjectWiseGrade = SubjectWiseGrades::where('class_id', $request->class_id)
                ->where('exam_id', $request->exam_id)
                ->where('session_id', $request->session_id)
                ->whereNull('deleted_at')
                ->get()
                ->keyBy('subject_id');
            $examWiseGrade = ExamWiseGrades::where('exam_id', $request->exam_id)->where('session_id', $request->session_id)->whereNull('deleted_at')->first();

            $html = '';
            foreach ($subjects as $key => $subject) {
                // dd($request->all(), $subjectWiseGrade, $subject, $key);
                $html .= '<tr>';
                $html .= '<td>' . ($key + 1) . '</td>';
                $html .= '<td>' . $subject->subjects->name . '</td>';
                $html .= '<td id="subjectMark" data-key="' . $key . '">' . ($subjectWiseGrade[$subject->subject_id]->exam_mark ?? $examWiseGrade->exam_mark ?? '100') . '</td>';
                $html .= '<td><input type="number" name="mark[]" class="form-control" placeholder="mark" required></td>';
                $html .= '<input type="hidden" name="subject_id[]" value="' . $subject->subject_id . '">';
                $html .= '</tr>';
            }

            // dd($subjects,$html, $subjectWiseGrade);

            return response()->json(['html' => $html]);
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentNeshabStore(Request $request)
    {
        $messages = [
            'exam_id.required'  => 'Select student',
            'class_id.required' => 'Select class',
        ];

        $request->validate([
            'exam_id'  => 'required',
            'class_id' => 'required',
        ], $messages);

        try {

            $existing = StudentNeshab::where('class_id', $request->class_id)->where('exam_id', $request->exam_id)
                ->where('session_id', $request->session_id)->exists();

            if ($existing) {
                return redirect()->back()->with('error', 'Neshab data already exists');
            }

            foreach ($request->subject_id as $key => $subjectId) {
                $data             = new StudentNeshab();
                $data->class_id   = $request->class_id;
                $data->session_id = $request->session_id;
                $data->exam_id    = $request->exam_id;
                $data->subject_id = $subjectId;
                $data->start_page = $request->start_page[$key];
                $data->end_page   = $request->end_page[$key];
                $data->note       = $request->note[$key];
                $data->created_by = Auth::user()->id;
                $data->save();
            }

            return redirect()->route('student.neshab')->with('success', 'Neshab created successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function StudentNeshabShow($id)
    {
        try {
            $data     = StudentNeshab::with('stuclass:id,name', 'exam:id,name', 'stusession')->findOrFail($id);
            $subjects = StudentNeshab::with('subject')->where('exam_id', $data->exam_id)->where('session_id', $data->session_id)->where('class_id', $data->class_id)->get();
            return view('dashboard.students.neshab.show', compact('data', 'subjects'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentNeshabCheck(Request $request)
    {
        try {
            $existing = StudentNeshab::where('class_id', $request->class_id)
                ->where('exam_id', $request->exam_id)
                ->where('session_id', $request->session_id)
                ->exists();

            return response()->json(['exists' => $existing]);

        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function getExistingData(Request $request)
    {
        try {
            $subjects = StudentNeshab::with('subject')
                ->where('exam_id', $request->exam_id)
                ->where('session_id', $request->session_id)
                ->where('class_id', $request->class_id)
                ->get();

            $html = '';
            foreach ($subjects as $key => $subject) {
                $html .= '<tr>';
                $html .= '<td>' . $key + 1 . '</td>';
                $html .= '<td>' . $subject->subject->name . '</td>';
                $html .= '<td><input name="start_page[]" type="text" class="form-control" value="' . $subject->start_page . '" required>';
                $html .= '<input type="hidden" name="subject_id[]" value="' . $subject->subject_id . '"></td>';
                $html .= '<td><input name="end_page[]" type="text" class="form-control" value="' . $subject->end_page . '" required></td>';
                $html .= '<td><textarea name="note[]" class="form-control" rows="1">' . $subject->note . '</textarea></td>';
                $html .= '</tr>';
            }

            return response()->json(['html' => $html]);

        } catch (\Exception $exception) {
            return response()->json(['error' => $exception->getMessage()], 500);
        }
    }

    public function StudentNeshabEdit($id)
    {
        try {
            $data     = StudentNeshab::with('stuclass:id,name', 'exam:id,name')->findOrFail($id);
            $subjects = StudentNeshab::with('subject')->where('exam_id', $data->exam_id)
                ->where('session_id', $data->session_id)
                ->where('class_id', $data->class_id)
                ->get();
            $classes     = StudentClass::all();
            $exams       = StudentExam::all();
            $stusessions = StudentSession::all();
            return view('dashboard.students.neshab.edit', compact('data', 'classes', 'subjects', 'exams', 'stusessions'));

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentNeshabUpdate(Request $request, $id)
    {
        $messages = [
            'exam_id.required'  => 'Select student',
            'class_id.required' => 'Select class',
        ];

        $request->validate([
            'exam_id'  => 'required',
            'class_id' => 'required',
        ], $messages);

        try {
            StudentNeshab::where('class_id', $request->class_id)
                ->where('exam_id', $request->exam_id)
                ->where('session_id', $request->session_id)
                ->delete();

            $existing = StudentNeshab::where('class_id', $request->class_id)->where('exam_id', $request->exam_id)
                ->where('session_id', $request->session_id)->exists();

            if ($existing) {
                return redirect()->back()->with('error', 'Neshab data already exists');
            }

            foreach ($request->subject_id as $key => $subjectId) {
                $data             = new StudentNeshab();
                $data->class_id   = $request->class_id;
                $data->exam_id    = $request->exam_id;
                $data->session_id = $request->session_id;
                $data->subject_id = $subjectId;
                $data->start_page = $request->start_page[$key];
                $data->end_page   = $request->end_page[$key];
                $data->note       = $request->note[$key];
                $data->created_by = Auth::user()->id;
                $data->save();
            }

            return redirect()->route('student.neshab')->with('success', 'Neshab updated successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function StudentNeshabDestroy($id)
    {
        try {
            $data = StudentNeshab::findOrFail($id);
            StudentNeshab::where('class_id', $data->class_id)->where('exam_id', $data->exam_id)->where('session_id', $data->session_id)->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }
    public function StudentComment(Request $request)
    {
        try {
            if ($request->ajax()) {
                $data = StudentComment::with('stuclass:id,name', 'exam:id,name', 'stusession')
                    ->get()
                    ->groupBy(function ($item) {
                        return $item->exam_id . '-' . $item->class_id . '-' . $item->session_id;
                    });

                return Datatables::of($data)
                    ->addColumn('exam', function ($data) {
                        $exam = $data->first()->exam;
                        return isset($exam) ? $exam->name : '--';
                    })
                    ->addColumn('class', function ($data) {
                        $class = $data->first()->stuclass;
                        return isset($class) ? $class->name : '--';
                    })
                    ->addColumn('stusession', function ($data) {
                        $stusession = $data->first()->stusession;
                        return isset($stusession) ? $stusession->name : '--';
                    })
                    ->addColumn('action', function ($data) {
                        $firstItem = $data->first();

                        $details = '<a href="' . route('student.comment.show', $firstItem->id) . '" class="btn btn-sm btn-info" title="Details"><i class="fa fa-eye"></i></a> ';
                        $edit    = '<a href="' . route('student.comment.edit', $firstItem->id) . '" class="btn btn-sm btn-success" title="Edit"><i class="fa fa-edit"></i></a> ';
                        $delete  = '<a href="#" data-remote="' . route('student.comment.destroy', $firstItem->id) . '" class="btn btn-sm btn-danger btn-delete" title="Delete"><i class="fa fa-trash"></i></a>';

                        return $details . $edit . $delete;
                    })

                    ->addIndexColumn()
                    ->rawColumns(['exam', 'class', 'action', 'stusession'])
                    ->toJson();
            }

            return view('dashboard.students.comment.index');
        } catch (\Exception $exception) {

            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentCommentCreate()
    {
        $classes     = StudentClass::all();
        $exams       = StudentExam::all();
        $stusessions = StudentSession::all();
        return view('dashboard.students.comment.create', compact('classes', 'exams', 'stusessions'));
    }

    public function StudentCommentStore(Request $request)
    {
        $messages = [
            'exam_id.required'    => 'Select student',
            'class_id.required'   => 'Select class',
            'session_id.required' => 'Select session',
        ];

        $request->validate([
            'exam_id'    => 'required',
            'class_id'   => 'required',
            'session_id' => 'required',
        ], $messages);

        try {
            $existing = StudentComment::where('exam_id', $request->exam_id)->exists();
            if ($existing) {
                return redirect()->back()->with('error', 'This exam comment already exists');
            }

            foreach ($request->student_id as $key => $studentId) {
                $data             = new StudentComment();
                $data->class_id   = $request->class_id;
                $data->session_id = $request->session_id;
                $data->exam_id    = $request->exam_id;
                $data->student_id = $studentId;
                $data->comment    = $request->comment[$key];
                $data->created_by = Auth::user()->id;
                $data->save();

                $student = Student::findOrFail($studentId);
                $exam    = StudentExam::findOrFail($data->exam_id);
                $to      = $student->gurdian_mobile;
                $text    = 'অভিনন্দন ' . $student->name . ' আপনার ' . $exam->name . ' সম্পর্কে শিক্ষকের মন্তব্য : ' . $data->comment;
                $status  = optional(SmsStatus::first())->student_comment;
                if ($status == 1) {
                    $this->SendSms($to, $text);
                }
            }

            return redirect()->route('student.comment')->with('success', 'Comment created successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function StudentCommentShow($id)
    {
        try {
            $data     = StudentComment::with('stuclass:id,name', 'exam:id,name', 'stusession')->findOrFail($id);
            $students = StudentComment::with('student:id,name,roll_no,register_no')->where('exam_id', $data->exam_id)->where('class_id', $data->class_id)->where('session_id', $data->session_id)->get();
            return view('dashboard.students.comment.show', compact('data', 'students'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function StudentCommentEdit($id)
    {
        try {
            $data     = StudentComment::with('stuclass:id,name', 'exam:id,name')->findOrFail($id);
            $students = StudentComment::with('student:id,name,status,roll_no,register_no,status')
                ->where('exam_id', $data->exam_id)->where('class_id', $data->class_id)
                ->where('session_id', $data->session_id)
                ->whereHas('student', function ($q) {
                    $q->where('status', 1)->orWhereNull('status');
                })
                ->get();
            $classes     = StudentClass::all();
            $exams       = StudentExam::all();
            $stusessions = StudentSession::all();
            return view('dashboard.students.comment.edit', compact('data', 'classes', 'students', 'exams', 'stusessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }
    public function StudentCommentUpdate(Request $request, $id)
    {
        $messages = [
            'exam_id.required'    => 'Select student',
            'class_id.required'   => 'Select class',
            'session_id.required' => 'Select session',
        ];

        $request->validate([
            'exam_id'    => 'required',
            'class_id'   => 'required',
            'session_id' => 'required',
        ], $messages);

        try {
            StudentComment::where('class_id', $request->class_id)
                ->where('session_id', $request->session_id)
                ->where('exam_id', $request->exam_id)
                ->delete();

            $existing = StudentComment::where('exam_id', $request->exam_id)->exists();
            if ($existing) {
                return redirect()->back()->with('error', 'This exam comment already exists');
            }

            foreach ($request->student_id as $key => $studentId) {
                $data             = new StudentComment();
                $data->class_id   = $request->class_id;
                $data->session_id = $request->session_id;
                $data->exam_id    = $request->exam_id;
                $data->student_id = $studentId;
                $data->comment    = $request->comment[$key];
                $data->created_by = Auth::user()->id;
                $data->save();

                $student = Student::findOrFail($studentId);
                $exam    = StudentExam::findOrFail($data->exam_id);
                $to      = $student->gurdian_mobile;
                $text    = 'অভিনন্দন ' . $student->name . ' আপনার ' . $exam->name . ' সম্পর্কে শিক্ষকের মন্তব্য : ' . $data->comment;
                $status  = optional(SmsStatus::first())->student_comment;
                if ($status == 1) {
                    $this->SendSms($to, $text);
                }
            }

            return redirect()->route('student.comment')->with('success', 'Comment updated successfully');

        } catch (\Exception $exception) {
            return redirect()->back()->with('error', 'Error: ' . $exception->getMessage());
        }
    }

    public function StudentCommentDestroy($id)
    {
        try {
            $data = StudentComment::findOrFail($id);
            StudentComment::where('class_id', $data->class_id)->where('exam_id', $data->exam_id)->where('session_id', $data->session_id)->delete();
            return response()->json([
                'success' => true,
                'message' => 'Data deleted successfully.',
            ]);
        } catch (\Exception $exception) {
            return response()->json([
                'success' => false,
                'message' => 'Data delete failed',
            ]);
        }
    }

    public function IDCard(Request $request)
    {
        try {
            if ($request->ajax()) {
                $query = Student::with(['stuclass', 'stusession'])
                    ->where(function ($q) {
                        $q->where('status', 1)
                            ->orWhereNull('status');
                    })
                    ->when($request->class_id, function ($q) use ($request) {
                        $q->where('class_id', $request->class_id);
                    })
                    ->when($request->session_id, function ($q) use ($request) {
                        $q->where('session_id', $request->session_id);
                    });

                $data = $query->whereNull('deleted_at')->get();

                $setting = Setting::first();

                return Datatables::of($data)

                    ->addColumn('student', function ($data) {
                        $name  = $data->name ?? '';
                        $class = $data->stuclass->name ?? '--';
                        $reg   = $data->register_no ?? '--';
                        return '<div class="text-center student-cell">' . $name . '<br>' . $class . '<br>' . $reg . '</div>';

                    })
                    ->addColumn('front_end', function ($data) use ($setting) {
                        $logo        = asset('img/' . ($setting->logo ?? ''));
                        $schoolName  = $setting->long_name ?? '';
                        $address     = $setting->address ?? '';
                        $photo       = asset('backend/img/student/' . ($data->image ?? ''));
                        $class       = $data->stuclass->name ?? '';
                        $session     = $data->stusession->name ?? '';
                        $roll        = $data->roll_no ?? '';
                        $gender      = $data->gender == 1 ? 'ছেলে' : ($data->gender == 2 ? 'মেয়ে' : '');
                        $mobile      = $data->gurdian_mobile ?? '';
                        $reg         = $data->register_no ?? '';
                        $father_name = $data->father_name ?? '';
                        // dd($data);

                        return '
                            <div class="student_card">
                                <div class="logo" style="background-image: url(' . $logo . ');"></div>
                                <input type="hidden" name="student_id[]" value="' . $data->id . '">
                                <div class="header">' . $schoolName . '</div>
                                <div class="sub-header">' . $address . '</div>
                                <div class="photo" style="background-image: url(' . $photo . ');"></div>
                                <div class="name">' . $data->name . '</div>
                                <div class="info">
                                পিতার নাম: ' . $father_name . '<br>
                                  ক্লাস: ' . $class . '<br>
                                  রোল নং: ' . $roll . '<br>
                                  লিঙ্গ: ' . $gender . '<br>
                                  মোবাইল: ' . $mobile . '<br>
                                  রেজিস্টার নাম্বার : ' . $reg . '
                                </div>
                            </div>
                        ';
                    })
                    ->addColumn('back_end', function ($data) use ($setting) {
                        $logo       = asset('img/' . ($setting->logo ?? ''));
                        $schoolName = $setting->long_name ?? '';
                        $address    = $setting->address ?? '';
                        $issue_date = \Carbon\Carbon::parse($data->date)->format('d/m/Y');
                        $valid_till = \Carbon\Carbon::parse($data->date)->addYear()->format('d/m/Y');

                        return '
                            <div class="student_card">
                                <div class="logo" style="background-image: url(' . $logo . ');"></div>
                                <div class="notice">
                                  ব্যবহারকারী ব্যতীত এই কার্ডটি অন্য কোথাও পাওয়া গেলে<br>
                                  মাদ্রাসার ঠিকানায় ফেরত দেওয়ার অনুরোধ রইল।
                                </div>
                                <div class="title">' . $schoolName . '</div>
                                <div class="event">
                                  ' . $address . '<br>
                                  ইস্যু তারিখ: ' . $issue_date . ' ইং<br>
                                  মেয়াদকাল: ' . $valid_till . ' ইং
                                </div>
                                <div class="condition">
                                  রাষ্ট্রদ্রোহী কাজে জড়িত না থাকার শর্তে<br>
                                  শুধুমাত্র অধ্যয়নকালীন সময়ের জন্য প্রযোজ্য
                                </div>
                            </div>
                        ';
                    })
                    ->addColumn('action', function ($data) {
                        $print = '<a title="a6 print" id="print" href="' . route('student.idcard', $data->id) . '" class="btn btn-sm btn-primary" title="print"><i class="fa fa-file"></i></a>';
                        return $print;
                    })
                    ->addIndexColumn()
                    ->rawColumns(['front_end', 'back_end', 'student', 'action'])
                    ->toJson();
            }
            $classes  = StudentClass::all();
            $sessions = StudentSession::all();
            $setting  = Setting::first();
            return view('dashboard.students.student.all_ids', compact('setting', 'classes', 'sessions'));
        } catch (\Exception $exception) {
            return redirect()->back()->with('error', $exception->getMessage());
        }
    }

    public function idDesign($id)
    {
        try {
            $data    = Student::findOrFail($id);
            $setting = Setting::first();
            return view('dashboard.students.student.id_card_design', compact('data', 'setting'));
        } catch (\Exception $e) {
            return redirect()->back()->with('error', $e->getMessage());
        }
    }

    //student admission due
    public function StudentAdmissionDue(Request $request)
    {
        if ($request->ajax()) {
            $admissionDue = DB::table('fund_details')
                ->join('students', 'fund_details.student_id', '=', 'students.id')
                ->where('fund_details.payment_type', 1) // Admission payment
                ->where('fund_details.type', 14)        // Admission type
                ->where('fund_details.due', '>', 0)
                ->orderBy('fund_details.id', 'desc')
                ->select([
                    'fund_details.id',
                    'fund_details.total',
                    'fund_details.amount',
                    'fund_details.discount',
                    'fund_details.due',
                    'students.name as student_name',
                    'students.register_no as student_reg',
                ]);

            return DataTables::of($admissionDue)
                ->addIndexColumn()
                ->addColumn('student', function ($row) {
                    return '<div class="fw-semibold">' . e($row->student_name) . '</div>
                        <div class="text-muted small">Reg: ' . e($row->student_reg) . '</div>';
                })
                ->addColumn('action', function ($row) {
                    return '<button class="btn btn-sm btn-primary collect-btn"
                        data-id="' . $row->id . '"
                        data-total="' . $row->total . '"
                        data-paid="' . $row->amount . '"
                        data-discount="' . $row->discount . '"
                        data-due="' . $row->due . '">Collection Due</button>';
                })
                ->rawColumns(['student', 'action']) // allow HTML
                ->make(true);
        }

        $setting = Setting::first();
        return view('dashboard.students.student.admission_due', compact('setting'));
    }
    public function collectAdmissionDue(Request $request)
    {
        $request->validate([
            'fund_detail_id' => 'required|integer',
            'receive_amount' => 'required|numeric|min:0.01',
            'note'           => 'nullable|string|max:500',
        ]);

        try {
            DB::transaction(function () use ($request) {
                // lock the row for update to prevent race conditions
                $fd = DB::table('fund_details')
                    ->where('id', $request->fund_detail_id)
                    ->lockForUpdate()
                    ->first();

                // dd($fd, $request->all());

                if (! $fd) {
                    abort(404, 'Record not found.');
                }

                $due      = (float) ($fd->due ?? 0);
                $paid     = (float) ($fd->amount ?? 0);
                $discount = (float) ($fd->discount ?? 0);
                $receive  = (float) $request->receive_amount;

                if ($due <= 0) {
                    abort(400, 'No due left to collect.');
                }

                // clamp receive to available due
                if ($receive > $due) {
                    $receive = $due;
                }

                DB::table('fund_details')->where('id', $fd->id)->update([
                    'amount'     => $paid + $receive,
                    'due'        => max(0, $due - $receive),
                    'updated_at' => now(),
                ]);

                DB::table('money_receipts')->where('id', $fd->student_payment_id)->update([
                    'amount'          => $paid + $receive,
                    'due'             => max(0, $due - $receive),
                    'current_receive' => $paid + $receive,
                    'updated_at'      => now(),
                ]);

                // OPTIONAL: record money-in transaction, audit log, cashier, etc.
                // DB::table('fund_collections')->insert([
                //     'fund_detail_id' => $fd->id,
                //     'amount'         => $receive,
                //     'note'           => $request->note,
                //     'created_at'     => now(),
                //     'updated_at'     => now(),
                //     'user_id'        => auth()->id(),
                // ]);
            });

            return back()->with('success', 'Due collected successfully.');
        } catch (\Throwable $e) {
            return back()->with('error', $e->getMessage());
        }
    }

}
